/* Hardware Tests for RS-EDP Platform Using MBED Module */
/* ************** */

/* Version 3.0 */


/* This module is to test the hardware on the MCBSTR91x Module. */


#include "mbed.h"                            /* Header file for mbed module */
#include "misra_types.h"                     /* MISRA Types */
#include "defines.h"                        /* User defines */
#include "RSEDP_Slave_Address_Defines.h"    /* Slave address of I2C Devices defined hhere */

#include "RSEDP_Mbed_Module.h"                /* all the low level peripheral drivers on te mbed module - IO ports, UART, SPI, I2C etc */
#include "RSEDP_Base_Board.h"                /* All the base board functions - EEPROM and erial latch */
#include "RSEDP_Comms_Module.h"                /* Functions to control the real time clock */
#include "RSEDP_Digital_IO_Module.h"        /* All the functions to control the serial input and output latches */
#include "RSEDP_Analogue_Module.h"            /* Functions to control digital pot and serial ADC */
#include "RSEDP_MC1.h"                        /* Brushed DC Motor Drive MC1 Module */
#include "RSEDP_Test_MC2.h"




/* Global Function prototype Here */
void RSEDP_test_all(void);


/* Local Menu functions */
static void Test_Processor_Module_Menu(void);
static void Test_Base_Board_Menu(void);
static void Test_Comms_Module_Menu(void);
static void Test_Digital_IO_Module_Menu(void);
static void Test_Analogue_Module_Menu(void);
static void Test_MC1_Module_Menu(void);

/*Processor Board Tests */
static void test_mbed_LEDs(void);
static void test_mbed_UART0(void);
static void test_mbed_UART1(void);
static void test_mbed_CNTRL_I2C_Master_Mode(void);
static void test_mbed_CNTRL_SPI_Master_Mode(void);

/* Base Board Tests */
static void test_BB_24LC32(uint8_t Slave_Address);
static void test_BB_PCA9675(uint8_t Slave_Address);
static void test_BB_RESET_Button(void);

/* Communication Board Tests */
static void test_PCF8583(uint8_t Slave_Address);

/* Digital I/O board Tests */
static void test_digital_output_PCA9555(uint8_t Slave_Address);
static void test_digital_input_PCA9555(uint8_t Slave_Address);

/* Analogue Module Tests */
static void test_MAX1x3x(uint8_t Slave_Address);
static void test_AD5263(uint8_t Slave_Address);

/* MC1 Tests */
static void test_MC1_VDCLINK(void);
static void test_MC1_VSENSE(void);
static void test_MC1_Read_Encoders(void);
static void test_MC1_Brakes(void);
static void test_MC1_Set_Direction(void);                
static void test_MC1_Speed_Ramp_UpDown(void);
static void test_MC1_Set_Speed_PWM_Duty(void);
static void test_MC1_Stop_Motor(void);


static void delay_ms(uint16_t time_delay);




/* Main Test Selection Menu */
void RSEDP_test_all(void)
    {
        uint16_t keypress = 0u;
    
        while(1)    
            {
                pc.printf("\n\n\rMain Top Level Test Menu\n\n\r");
                pc.printf("1. Processor Module Test\n\r");
                pc.printf("2. Base Board Tests\n\r");

                if (COMMS_MODULE_FITTED == YES)
                    {
                        pc.printf("3. Communications Module Tests\n\r");
                    }
                if (DIGITAL_IO_MODULE_FITTED == YES)
                    {
                        pc.printf("4. Digital I/O Module Tests\n\r");                
                    }
                if (ANALOGUE_MODULE_FITTED == YES)
                    {
                        pc.printf("5. Analogue Module Tests\n\r");
                    }
                if (MOTOR_MC1_MODULE_FITTED == YES)
                    {
                        pc.printf("6. Brushed DC Motor Drive MC1 Module Tests\n\r");
                    }
                if (MOTOR_MC2_MODULE_FITTED == YES)
                    {
                        pc.printf("7. Brushless DC Motor Drive MC2 Module Tests\n\r");
                    }
                pc.printf("\n\rPlease select one of the above tests\r\n");                    
        
                keypress = pc.getc();
                
                if (keypress == '1') Test_Processor_Module_Menu();
                if (keypress == '2') Test_Base_Board_Menu();
                if (COMMS_MODULE_FITTED == YES)
                    {
                        if (keypress == '3') Test_Comms_Module_Menu();
                    }
                if (DIGITAL_IO_MODULE_FITTED == YES)
                    {
                        if (keypress == '4') Test_Digital_IO_Module_Menu();
                    }
                if (ANALOGUE_MODULE_FITTED == YES)
                    {
                        if (keypress == '5') Test_Analogue_Module_Menu();
                    }
                if (MOTOR_MC1_MODULE_FITTED == YES)
                    {
                        if (keypress == '6') Test_MC1_Module_Menu();
                    }
                if (MOTOR_MC2_MODULE_FITTED == YES)
                    {
                        if (keypress == '7') I2C_MMSC_Test_Suite();
                    }
                    
            }
    }


/* Test The Microprocessor Module */
static void Test_Processor_Module_Menu(void)
    {
        uint16_t keypress=0;

        while(1)
            {                
                pc.printf("\n\n\rProcessor Module Test Menu\n\n\r");
                pc.printf("1. LED Test\n\r");
                pc.printf("2. UART0 Test\n\r");
                pc.printf("3. UART1 Test\n\r");
                pc.printf("4. CNTRL I2C Master Mode Test\n\r");
                pc.printf("5. CNTRL SPI Master Mode Test\n\r");                
            
                pc.printf("\n\rPlease select one of the above tests\r\n");                    

                keypress = pc.getc();
                
                if (keypress=='1') test_mbed_LEDs();
                if (keypress=='2') test_mbed_UART0();
                if (keypress=='3') test_mbed_UART1();
                if (keypress=='4') test_mbed_CNTRL_I2C_Master_Mode();
                if (keypress=='5') test_mbed_CNTRL_SPI_Master_Mode();                
                if (keypress=='\r') break;
    
            }        
    }


/* This is the menu for the Base Board Tests */
static void Test_Base_Board_Menu(void)
    {
        uint16_t keypress = 0u;

        while(1)
            {                
                pc.printf("\n\n\rBase Board Test Menu\n\n\r");
                pc.printf("1. DIP Switch via PCA9675 Serial Input Latch\n\r");
                pc.printf("2. 24LC32 Serial EEPROM Test\n\r");
                pc.printf("3. Reset Button Test\n\r");
                pc.printf("\n\rPlease select one of the above tests\r\n");                    

                keypress = pc.getc();
        
                if (keypress == '1') test_BB_PCA9675(PCA9675_BASE_BOARD);
                if (keypress == '2') test_BB_24LC32(M24C32_BASE_BOARD);
                if (keypress == '3') test_BB_RESET_Button();
                if (keypress == '\r') break;
            }        
    }

/* Test The Communications Module */
static void Test_Comms_Module_Menu(void)
    {
        uint16_t keypress=0;

        while(1)
            {                
                pc.printf("\n\n\rCommunications Module Test Menu\n\r");
                pc.printf("1 PCF8385 Real Time Clock Test\n\r");
//                pc.printf("2 CAN2.0B Test\n\r");
                pc.printf("\n\rPlease select one of the above tests\r\n");                    

                keypress = pc.getc();
                
                if (keypress=='1') test_PCF8583(PCF8583_COMMS_MODULE);
//                if (keypress=='2') test_CAN();
                if (keypress=='\r') break;
            }
    }

/* Test The Digital I/O Module */
static void Test_Digital_IO_Module_Menu(void)
    {
        uint16_t keypress=0;

        while(1)
            {                
                pc.printf("\n\n\rDigital I/O Module Test Menu\n\n\r");
                pc.printf("1. Serial Digital Output OUT_P0(x) & OUT_P1(x)\n\r");
                pc.printf("2. Serial Digital Input IN_P0(x) & IN_P1(x)\n\r");
                pc.printf("\n\rPlease select one of the above tests\r\n");                    

                keypress = pc.getc();
                if (keypress == '1') test_digital_output_PCA9555(PCA9555_DIGITAL_IO_OUT);
                if (keypress == '2') test_digital_input_PCA9555(PCA9555_DIGITAL_IO_IN);
                if (keypress == '\r') break;

            }    
    }


/* Test The Analogue Module */
static void Test_Analogue_Module_Menu(void)
    {
        uint16_t keypress=0;

        while(1)
            {                
                pc.printf("\n\n\rAnalogue Module Test Menu\n\n\r");
                pc.printf("1. MAX1x38 I2C Serial A to D Converter\n\r");                     /* Production 10 bit ADC option */
                pc.printf("2. AD5263 I2C Digital Potentiometer\n\r");
                pc.printf("\n\rPlease select one of the above tests\r\n");
                    
                keypress = pc.getc();
                
                if (keypress == '1') test_MAX1x3x(MAX1X3X_ANALOGUE_ADC);
                if (keypress == '2') test_AD5263(AD5263_ANALOGUE_DIGITAL_POT);
                if (keypress == '\r') break;

            }
    }


/* This is the menu for the Brushed DC Motor Control MC! Module Tests */
static void Test_MC1_Module_Menu(void)
    {
        uint16_t keypress = 0u;

        while(1)
            {                
                pc.printf("\n\n\rBrushed DC Motor Drive Module MC1 Test Menu\n\n\r");
                pc.printf("1. Read the VDCLINK Voltage\n\r");
                pc.printf("2. Read the VSENSE Voltage\n\r");
                pc.printf("3. Read the Encoder0 & Encoder 1 Input Signals\n\r");
                pc.printf("4 Test the Brake ON & OFF Signal\n\r");
                pc.printf("5 Set the direction of rotation\n\r");                
                pc.printf("6 Ramp the Motor Speed PWM Duty Up & Down\n\r");
                pc.printf("7 Set the Motor PWM Duty\n\r");                
                pc.printf("8 Stop the motor\n\r");                
                
                pc.printf("\n\rPlease select one of the above tests\r\n");                    

                keypress = pc.getc();
        
                if (keypress == '1') test_MC1_VDCLINK();
                if (keypress == '2') test_MC1_VSENSE();
                if (keypress == '3') test_MC1_Read_Encoders();
                if (keypress == '4') test_MC1_Brakes();
                if (keypress == '5') test_MC1_Set_Direction();                
                if (keypress == '6') test_MC1_Speed_Ramp_UpDown();
                if (keypress == '7') test_MC1_Set_Speed_PWM_Duty();
                if (keypress == '8') test_MC1_Stop_Motor();
                if (keypress == '\r') break;
            }        
    }







/* Processor Board Tests Starts Here */
/* LED Tests on the Processor Card */
static void test_mbed_LEDs(void)
    {
        uint8_t n = 0;

        
        pc.printf("\n\n\n\rTesting the LEDs on the mbed module...\n\n\r");
        for (n = 0; n < 5; n++)
            {
                User_Led1 = LED_ON;                        /* Flash the LED to signal the world the firmware is running */
                User_Led2 = LED_ON;                        /* Flash the LED to signal the world the firmware is running */
                User_Led3 = LED_ON;                        /* Flash the LED to signal the world the firmware is running */
                User_Led4 = LED_ON;                        /* Flash the LED to signal the world the firmware is running */
                wait(0.5);
                
                User_Led1 = LED_OFF;                    /* Flash the LED to signal the world the firmware is running */
                User_Led2 = LED_OFF;                    /* Flash the LED to signal the world the firmware is running */
                User_Led3 = LED_OFF;                    /* Flash the LED to signal the world the firmware is running */                                
                User_Led4 = LED_OFF;                    /* Flash the LED to signal the world the firmware is running */                
                wait(0.5);

            }
        pc.printf("Finished testing the LEDs...\n\n\r");
    }



/* UART0 Tests */
static void test_mbed_UART0(void)
    {
        sint32_t keypress = 0;
        
        pc.printf("\n\n\n\rTesting UART0\n\n\r");
        UART0.printf("\n\n\rIf you can read this information then UART0 TRANSMIT is working correctly...\n\r");        
        UART0.printf("Now to test the UART0 RECEIVE Input...\n\r");
        pc.printf("Now please keys on the keyboard and see if they are echoed on the monitor...\n\r");
        UART0.printf("Now please keys on the keyboard and see if they are echoed on the monitor...\n\r");
        UART0.printf("Press the Return key to quit...\n\r");
        pc.printf("Press the Return key to quit...\n\r");

        do    {
                keypress = UART0.getc();                                                    /* get a keypress */

                UART0.putc(keypress);
                UART0.printf("\r");

            } while(keypress != '\r');

        pc.printf("\n\rTest finished\n\r");    
        UART0.printf("\n\rTest finished\n\r");    
    }


/* UART1 Tests */
static void test_mbed_UART1(void)
    {
        sint32_t keypress = 0;
        
        pc.printf("\n\n\n\rTesting UART1\n\n\r");
        UART1.printf("\n\n\rIf you can read this information then UART1 TRANSMIT is working correctly...\n\r");        
        UART1.printf("Now to test the UART1 RECEIVE Input...\n\r");
        pc.printf("Now please keys on the keyboard and see if they are echoed on the monitor...\n\r");
        UART1.printf("Now please keys on the keyboard and see if they are echoed on the monitor...\n\r");
        UART1.printf("Press the Return key to quit...\n\r");
        pc.printf("Press the Return key to quit...\n\r");

        do    {
                keypress = UART1.getc();                                                    /* get a keypress */

                UART1.putc(keypress);
                UART1.printf("\r");

            } while(keypress != '\r');

        pc.printf("\n\rTest finished\n\r");    
        UART1.printf("\n\rTest finished\n\r");    
    }





/* Test The CNTRL I2C in Master Mode */
static void test_mbed_CNTRL_I2C_Master_Mode(void)
    {
        sint8_t cmd[1] = {0};
        sint8_t keypress = 0;
        sint32_t Ack_Status = 0;
        uint8_t n = 0;    
        
        pc.printf("\n\n\n\rTesting the CNTRL I2C Peripheral...\n\r");
        pc.printf("Press return key to finish test.\n\n\r");
        cmd[0] = 0x0;                                                                    /* dummy data to transmit */


        do    {
                pc.printf("Sending out the byte 0x00 to 8 bit address : %d \n\r", n);    

                Ack_Status = CNTRL_i2c.write(n, cmd, 1);                                 /* Send an I2C byte */
                if (Ack_Status == ACK)
                    {
                        pc.printf("Acknowledge received\n\r");
                    }
                else{
                        pc.printf("No Acknowledge received\n\r");
                    }
                
                n++;
                keypress = pc.getc();                                                    /* get a keypress */

            } while (keypress != '\r');
            
        pc.printf("Finished testing CNTRL I2C Peripheral.\n\n\r");
    }



/* Test The CNTRL SPI in Master Mode */
static void test_mbed_CNTRL_SPI_Master_Mode(void)
    {
        uint8_t n = 1;    
        sint8_t keypress = 0;

        pc.printf("\n\n\n\rTesting the CNTRL SPI Peripheral...\n\r");
        pc.printf("Press return key to finish test.\n\n\r");

        do    {
                pc.printf("Sending out the byte:%d...\n\r",n);    

                CNTRL_SPI_Write_Byte(n);
                n++;
            
                keypress = pc.getc();                                                    /* get a keypress */

            } while (keypress != '\r');
            
        pc.printf("Finished testing CNTRL SPI Peripheral.\n\n\r");        
    }


    





/* Base Board Tests Start Here */
/* *************************** */

/* Test The I2C PCA9675 Serial Latch Device on the base board */
static void test_BB_PCA9675(uint8_t Slave_Address)
    {

        uint8_t port_zero = 0;
        uint8_t port_one = 0;
        uint16_t keypress = 0;
        uint8_t n = 0;
        sint32_t Ack_Status = 0;
        uint8_t Switch_Status = 0;

        pc.printf("\n\n\n\rTest The DIP Switches Via The PCA9675 Serial Input Latch\n\n\r");

        pc.printf("Writing to the latch, to configure the chip for input...\n\r");
        Ack_Status = RSEDP_BB_PCA9675_write_data(Slave_Address, 0xff,0xff);                                                 /* Write 0xff to the output latches so we can read the dip switches */ 
        if (Ack_Status == ACK)
            {
                pc.printf("\n\rReading from the latch...\n\r"); 
                Ack_Status = RSEDP_BB_PCA9675_read_data(Slave_Address, &port_zero, &port_one);                                        /* Read two bytes from the device */
                if (Ack_Status == ACK)
                    {
                        pc.printf("Value read...Port0: 0x%.2X",port_zero);
                        pc.printf(" and Port1: 0x%.2X\r\n",port_one);

                        pc.printf("Turn each bit on and off independantly and press space bar to update screen...\n\r");
                        pc.printf("To terminate this test press the Return key\n\r"); 
                        
                        do {
                                for (n = 1; n < 9; n++)
                                    {
                                        Ack_Status = RS_EDP_BB_Read_DIP_Switch(n, &Switch_Status);
                                        if (Switch_Status == DIP_SWITCH_CLOSED) 
                                            {
                                                pc.printf("%.1d:ON  ",n);
                                            }
                                        else{
                                                pc.printf("%.1d:OFF ",n);
                                            }
                                    }
                                
                                pc.printf("\r");
                                
                                delay_ms(500);
                
                                keypress = pc.getc();                                                            /* get a keypress */
                
                            } while ((keypress != '\r') && (Ack_Status == ACK));
                    }
             }
        if (Ack_Status != ACK)
            {
                pc.printf("*** NO Acknowledge Received ***\n\r");
                pc.printf("Operation aborted\n\r");                
            }
                  
    }



/* Test The 32k bit 24LC32 Serial EEPROM on the Base Board */
static void test_BB_24LC32(uint8_t Slave_Address)
    {
        sint32_t Ack_Status = 0;
        sint8_t I2C_Tx_Array[32];
        sint8_t I2C_Rx_Array[256];
        sint8_t n = 0;
        uint16_t nn = 0;
        uint8_t error_flag = 0;

        
        pc.printf("\n\n\n\rTesting the 24LC32 Device....\n\n\r");
        
        for (n=0;n<32;n++)
            {
                I2C_Tx_Array[n] = 255 - (n*4);                                    /* Configure the array we want to write to the EEPROM */
            }
        
        n=0;
        while(1)                                                              /* Clear receieve array */
            {
                I2C_Rx_Array[n] = 0;
                n++;
                if (n == 0) break;
            }
        
        
        pc.printf("Writing one byte of data to the 24LC32 Device....\n\r");
        pc.printf("Writing the value 0x%.2X \n\r", EEPROM_TEST_BYTE);
        Ack_Status = RSEDP_BB_eeprom_store_byte(Slave_Address, EEPROM_TEST_BYTE, 0x00);    /* Store one byte in one location */             
        if (Ack_Status == ACK)
            {
            pc.printf("Reading one byte from the EEPROM.\n\r");
            Ack_Status = RSEDP_BB_eeprom_read_byte(Slave_Address, &n, 0x0);                                /* Read one byte from one location */
            if (Ack_Status == ACK)
                {
                    pc.printf("Value read...0x%.2X\n\r", n);
            
    
                    if (n != EEPROM_TEST_BYTE)
                        {
                            pc.printf("FAILED...Data read was different from the data written\n\r");
                            while(1);
                        }
                    else{
                            pc.printf("PASSED - Value read was the same as value written.\n\r");
                        }
            
        
                    /* Writing Pages To The EEPROM */
    
                    pc.printf("\n\rWriting pages of data to the 24LC32 Device using page write....\n\r");
                    Ack_Status = RSEDP_BB_eeprom_store_bytes(Slave_Address, I2C_Tx_Array, 32, 0x00);             /* Write a 32 byte byte page to the I2C serial EEPROM. Note On a page boundary ! */
    
                    if (Ack_Status == ACK)
                        {
                            pc.printf("Reading back 32 bytes from the EEPROM.\n\r");    
                            Ack_Status = RSEDP_BB_eeprom_read_bytes(Slave_Address, I2C_Rx_Array,32, 0x0);                 /* Read multiple bytes back from the EEPROM */
                            if (Ack_Status == ACK)
                                {
                                    error_flag = 0;
                                    for (n = 0; n < 32; n++)
                                        {
                                            if (I2C_Tx_Array[n] != I2C_Rx_Array[n]) 
                                                {
                                                    error_flag = 1;
                                                }
                                        }
                            
                                    if (error_flag == 1)
                                        {
                                            pc.printf("FAILED...Data read was different from the data written\n\r");
                                            pc.printf("The transmit array was...\n\r");                
                                            for (n = 0; n < 32; n++)
                                                {
                                                    pc.printf("0x%.2X ", I2C_Tx_Array[n]);
                                                }
                                            
                                            pc.printf("\n\r");
                                            pc.printf("The received array was...\n\r");                
                                            for (n=0;n<32;n++)
                                                {
                                                    pc.printf("0x%.2X ", I2C_Rx_Array[n]);
                                                }
                                
                                            while(1);
                                        }
                                        
                                    else{
                                            pc.printf("PASSED - Value read was the same as value written.\n\r");
                                        }
                                
        
                                    /* Writing To The Whole Memory Area */
                                    pc.printf("\n\nWriting to the whole EEPROM memory address space...\n\r");
                            
                                    nn = 0;
                                    do {
                                            Ack_Status = RSEDP_BB_eeprom_store_byte(Slave_Address, (nn/8),nn);
                                            pc.printf(".");
                                            nn++;
                                            wait(0.1);
                                         } while ((Ack_Status == ACK) && (nn < 0x800));
                                    
                                    pc.printf("\n\r\nChecking the EEPROM...\n\r");
    
                                    if (Ack_Status == ACK)
                                        {
                                            error_flag = 0;
                                            for (nn = 0; nn < 0x800; nn++)
                                                {
                                                    Ack_Status = RSEDP_BB_eeprom_read_byte(Slave_Address, &n, nn);
                                                    
                                                    if (n != nn / 8) 
                                                        {
                                                            error_flag = 1;
                                                        }
                                                }
                                            if (error_flag == 1)
                                                {
                                                    pc.printf("Test FAILED - Data Read did not match the data written\n\n\r");
                                                    while(1);
                                                }
                                            else{
                                                    pc.printf("Test PASSED - EEPROM is working correctly\n\n\r");
                                                }
                                        
                                            
                                         }
                                 }
                         }
                   }
            }
        if (Ack_Status != ACK)
            {
                 pc.printf("*** NO Acknowledge Receieve ***\n\r");
                 pc.printf("Operation aborted\n\r");
             }
        
        pc.printf("Finished testing the 24LC32 Device....\n\n\r");         
    }



/* This funtction test the RESET Button on the base board */
static void test_BB_RESET_Button(void)
    {
        pc.printf("\n\n\n\rTesting the RESET Button\n\r");
        pc.printf("Press The RESET Button and this program should reset back to the Main Test Menu\n\r");
        while(1);                /* Wait for reset to be pressed */
     }






/* Communications Board Tests */

/* Test the serial RTC Device */
static void test_PCF8583(uint8_t Slave_Address)
    {
        sint32_t Ack_Status = 0;
        uint8_t status_register = 0;
        uint8_t RTC_Array[8];
        uint16_t keypress = 0;
        uint16_t nn = 0;
        uint8_t value = 0;

        pc.printf("\n\n\n\rTesting the PCA8583 Real Time Clock Chip...\n\r");
        pc.printf("Testing in CLOCK mode...\n\r"); 
        pc.printf("Configuring the device...\n\r");         

        Ack_Status = RSEDP_COM_setup_PCF8583(PCF8583_COMMS_MODULE,'C');                        

                if (Ack_Status == ACK)
                    {
                        pc.printf("\n\rReading the status register...\n\r");
                        Ack_Status = RSEDP_COM_PCF8583_Read_Status_Register(Slave_Address, &status_register);        
                    }
                if (Ack_Status == ACK)
                    {
                        pc.printf("Status Register = 0x%2X\r\n",status_register);
                        pc.printf("\n\rReading the Real Time Clock Contents\n\r");
                        pc.printf("Press the retrun key to finish the test\n\r");
                
                        do {
                                Ack_Status = RSEDP_COM_PCF8583_Read_RTC(Slave_Address,RTC_Array);
                                pc.printf("Time is ");
                                RSEDP_COM_Print_Time_Date_Year(RTC_Array);
                                pc.printf("\r");
                                
                                keypress = pc.getc();
                            
                                delay_ms(100);
                
                            } while((keypress != '\r') && (Ack_Status == ACK));
                    }

                if (Ack_Status == ACK)
                    {
                        pc.printf("\n\n\rWriting a new time and date to the RTC device\n\r");
                        Ack_Status = RSEDP_COM_PCF8583_Set_Clock(Slave_Address,11,59,40,1,1);                                // 11:59AM and 45 seconds in AM/PM mode
                        RSEDP_COM_PCF8583_Set_Date(Slave_Address,6, 31,12,2008);            
                        if (Ack_Status == ACK)
                            {
                                pc.printf("\n\rReading back the Real Time Clock Contents\n\r");
                            }
                        pc.printf("Press the retrun key to finish the test\n\r");
                    }
                
                if (Ack_Status == ACK)
                    {
                        do {
                                Ack_Status = RSEDP_COM_PCF8583_Read_RTC(Slave_Address, RTC_Array);
                                pc.printf("Time is ");
                                RSEDP_COM_Print_Time_Date_Year(RTC_Array);
                                pc.printf("\r");
                        
                                keypress = pc.getc();
                            
                                delay_ms(100);
                
                            } while((keypress != '\r') && (Ack_Status == ACK)) ;
                    }
                if (Ack_Status == ACK)
                    {
                        pc.printf("\n\n\rChanging to 24 hour notation\n\r");
                        Ack_Status = RSEDP_COM_PCF8583_Set_Clock(Slave_Address,23,59,40,0,1);
                        if (Ack_Status == ACK)
                            {
                                Ack_Status = RSEDP_COM_PCF8583_Set_Date(Slave_Address,6, 31,12,2008);
                                pc.printf("\n\rReading back the Real Time Clock Contents\n\r");
                                pc.printf("Press the return key to finish the test\n\r");
                             }
                     }
                if (Ack_Status == ACK)
                    {
                       do    {
                                Ack_Status = RSEDP_COM_PCF8583_Read_RTC(Slave_Address, RTC_Array);
                                if (Ack_Status == ACK)
                                    {
                                        pc.printf("Time is ");
                                        RSEDP_COM_Print_Time_Date_Year(RTC_Array);
                                        pc.printf("\r");
    
                                        keypress = pc.getc();
                                    
                                        delay_ms(100);
                                     }
                        
                             } while((keypress != '\r') && (Ack_Status == ACK));
                    } 
                if (Ack_Status == ACK)
                    {
                         pc.printf("\n\n\rStopping the CLOCK\n\r\n");
                         pc.printf("Press the return key to finish the test\n\r");
                
                         Ack_Status = RSEDP_COM_PCF8583_Stop_Clock(Slave_Address);
                        do    {
                                  Ack_Status = RSEDP_COM_PCF8583_Read_RTC(Slave_Address, RTC_Array);
                                  if (Ack_Status == ACK)
                                      {
                                            pc.printf("Time is ");
                                            RSEDP_COM_Print_Time_Date_Year(RTC_Array);
                                            pc.printf("\r");
                                            keypress = pc.getc();
                                        
                                            delay_ms(100);
                                      }
                
                            } while((keypress != '\r') && (Ack_Status == ACK));
                    }
    
                if (Ack_Status == ACK)
                    {
                        pc.printf("\n\n\rStarting the CLOCK again\n\r\n");
                        pc.printf("Press the return key to finish the test\n\r");
        
                        Ack_Status = RSEDP_COM_PCF8583_Start_Clock(Slave_Address);
                        do    {
                                  Ack_Status = RSEDP_COM_PCF8583_Read_RTC(Slave_Address, RTC_Array);
                                  if (Ack_Status == ACK)
                                      {
                                            pc.printf("Time is ");
                                            RSEDP_COM_Print_Time_Date_Year(RTC_Array);
                                            pc.printf("\r");
                                            keypress = pc.getc();
                            
                                            delay_ms(100);
                                       }
                
                               } while((keypress != '\r') && (Ack_Status == ACK));
                    }
                
                if (Ack_Status == ACK)
                    {
                        pc.printf("\n\n\rTesting the SRAM area of the RTC...\n\r\n");
                        pc.printf("Writing Data to the memory...\n\r");
    
                        nn = 0x10;
                        do {
                                Ack_Status = RSEDP_COM_PCF8583_Write_SRAM(Slave_Address, nn/2, nn);
                                nn++;
                            } while ((nn<0x100) && ( Ack_Status == ACK));
                
                        if (Ack_Status == ACK)
                            {
                                pc.printf("Finished writing the SRAM area of the RTC.\n\r");
                                pc.printf("Checking the SRAM area of the RTC...\n\r");
                            }                                

                        if (Ack_Status == ACK)
                            {
                                nn = 0x10;
                                do  {
                                        Ack_Status = RSEDP_COM_PCF8583_Read_SRAM(Slave_Address, &value, nn);
                            
                                        if (value != nn/2) 
                                            {
                                                pc.printf("FAILED TEST - The data read is different from the data written\n\r");
                                                pc.printf("Failed at memory location 0x%.3X\n\r",nn);
                                                pc.printf("The data written was 0x%.2X\n\r",nn/2);
                                                pc.printf("The data read    was 0x%.2X\n\r",value);
                                                while(1);                                             /* Error reading from SRAM */
                                            }
                                        nn++;
                                    } while((nn < 0x100) && (Ack_Status == ACK)); 
                            }
                        if (Ack_Status == ACK)
                            {
                                 pc.printf("PASSED TEST\n\r");
                            }
        
                    }
        
        
                if (Ack_Status != ACK)
                     {
                         pc.printf("*** NO Acknowledge Receieve ***\n\r");
                         pc.printf("Operation aborted\n\r");
                     }                    

        pc.printf("\n\rFinished testing the PCF8583 Real Time Clock device\n\r\n");
    }




    
    
    
/* Digital I/O Module Tests */    

/* Test the ULN2003 output with the PCA9555A Device */
static void test_digital_output_PCA9555(uint8_t Slave_Address)
    {
        /* assume the I2C and I/O direction have been set up */
        
        sint32_t Ack_Status = 0;
        uint8_t port_pin = 1;
        uint8_t temp8 = 1;
                
        pc.printf("\n\n\n\rTesting the Digital Outputs under I2C Control...\n\r");
        pc.printf("Press the return key to sequence through the tests.\n\r");

        Ack_Status = RSEDP_DIO_setup_PCA9555(Slave_Address, 0x00, 0x00);                                                            /* Configure Both Port 0 and Port 1 as outputs */
        
        if (Ack_Status == ACK)
            {
                pc.printf("\n\rTesting the Port0 of the PCA9555...\n\r");    

                port_pin = 0;

                do   {
                        pc.printf("\n\rTurning ON the output OUT_P0(%.1d)", port_pin);
                        
                        Ack_Status = RSEDP_DIO_PCA9555_Write_Word(Slave_Address, temp8,0x00);
        
                        if (Ack_Status == ACK)
                            {
                                pc.getc();
                        
                                pc.printf("\n\rTurning OFF the output OUT_P0(%.1d)", port_pin);
                                
                                Ack_Status = RSEDP_DIO_PCA9555_Write_Word(Slave_Address, 0x00, 0x00);
                            }
                        if (Ack_Status == ACK)
                            {
                                pc.getc();
                             }
                
                        temp8 = (temp8 << 1);
                        pc.printf("\n\r");
                        port_pin++;
                           
                      } while ((port_pin < 8) && (Ack_Status == ACK));

            }
        if (Ack_Status == ACK)
            {
                pc.printf("\n\rTesting the Port1 of the PCA9555...\n\r");    
                temp8 = 0x01;
       
                port_pin = 0;
                for (port_pin = 0; port_pin < 8; port_pin++)
                do {
                        pc.printf("\n\rTurning ON the output OUT_P1(%.1d)", port_pin);
                
                        Ack_Status = RSEDP_DIO_PCA9555_Write_Word(Slave_Address, 0x00, temp8);
                        if (Ack_Status == ACK)
                            {
                                pc.getc();
                                
                                pc.printf("\n\rTurning OFF the output OUT_P1(%.1d)",port_pin);
                
                                Ack_Status = RSEDP_DIO_PCA9555_Write_Word(Slave_Address, 0x00,0x00);
                            }
                        pc.getc();

                        temp8 = (temp8 << 1);
                        pc.printf("\n\r");
                        port_pin++;
                    } while ((port_pin < 8) && (Ack_Status == ACK));
             }       
    
        if (Ack_Status != ACK)
             {
                  pc.printf("*** NO Acknowledge Receieve ***\n\r");
                  pc.printf("Operation aborted\n\r");
              }       
        pc.printf("\n\r\nFinished testing the Digital Output under I2C Control\r\r\n");    
    }






/* Test The Inputs to the PCA9555 serial I/O device */
/* The device is configured as an input device. */
static void test_digital_input_PCA9555(uint8_t Slave_Address)
    {
        
        sint32_t Ack_Status = 0;
        uint8_t port_one = 0;
        uint8_t port_zero = 0;
        uint16_t keypress = 0;
        uint16_t input_data = 0;
        uint8_t n= 0;

                
        pc.printf("\n\n\n\rTesting the Digital Inputs via I2C Control...\n\r");
        pc.printf("Press the return key to sequence through the tests.\n\r");
        
        Ack_Status = RSEDP_DIO_setup_PCA9555(Slave_Address, 0xff, 0xff);                                /* Configure Both Port 0 and Port 1 as inputs  */

        if (Ack_Status == ACK)
            {
                pc.printf("\n\rTesting the Ports of the I2C device in input mode...\n\r");    

                pc.printf("\n\rReading from the device...\n\r"); 
                pc.printf("To terminate this test press the Return key\n\r\n"); 
        
                pc.printf("Value read of the Port 0 and Port 1 pins are:\n\r");
                pc.printf("         Port 0                        Port 1       \n\r");
                pc.printf("0  1  2  3  4  5  6  7   {}   0  1  2  3  4  5  6  7\n\r");


                do    {
                            Ack_Status = RSEDP_DIO_PCA9555_Read_Data(Slave_Address, &port_zero, &port_one);          /* Read the DIP Switches onthe base board */

                            if (Ack_Status == ACK)
                                {
                                    input_data = ((port_one << 8) + port_zero);
                                    for (n = 0u; n < 16u; n++)
                                        {
                                            if (n == 8)
                                                {
                                                    pc.printf("      ");
                                                }
                                            if ((input_data & 0x0001) == 0u)
                                                {
                                                    pc.printf("O  ");
                                                }
                                            else{
                                                    pc.printf("1  ");
                                                }    
                            
                                            input_data = input_data >> 1;
                                        }
                                                    
                                    pc.printf("\r");
                                    keypress = pc.getc();
                               }
                    
                        } while ((keypress != '\r') && (Ack_Status == ACK));
            }
        if (Ack_Status != ACK)
             {
                  pc.printf("*** NO Acknowledge Receieve ***\n\r");
                  pc.printf("Operation aborted\n\r");
             }               
        
        pc.printf("\n\r\nFinished testing the Serial Digital Input\r\r\n");    
    
    }




/* Analogue Module Tests */

/* Test MAX1038 ADC On AN16 Module */
static void test_MAX1x3x(uint8_t Slave_Address)
    {
        sint32_t Ack_Status = 0;
        uint8_t uiChannel = 0u;
        uint8_t u8Result = 0u;
        uint16_t u16Result = 0u;
        uint8_t keypress = 0u;

        pc.printf("\n\rTesting the serial ADC on the Analogue Module...\n\r");

        pc.printf("\n\rConfiguring the MAX1x3x %d bit ADC converter...\n\r",ADC_BITSIZE);
        Ack_Status = RSEDP_AM_Init_MAX1x3x(Slave_Address) ;
        if (Ack_Status == ACK)
            {
                pc.printf("\n\rMAX1x3x Now Configured");
            
    
                /* Test all channels */
                pc.printf("\n\rPress 'RETURN' key to terminate the test\n\rChannel Readings Are:\n\r");
                pc.printf("CH00  CH01  CH02  CH03  CH04  CH05  CH06  CH07  CH08  CH09  CH10  CH11\n\r");
                do {        
                       for(uiChannel = 0u ; uiChannel < MAX1x3x_No_Of_Channels; uiChannel++)
                            {
                                if (ADC_BITSIZE == ADC_8BIT)                                                                /* 8 bit ADC fitted to analogue module */
                                    {
                                        Ack_Status = RSEDP_AM_MAX103x_ReadChannel(Slave_Address, &u8Result, uiChannel);            /* this line for 8 bit ADC */
                                        u16Result = u8Result;
                                    }
                                else{                                                                                /* 10 bit ADC fitted to analogue module */
                                        Ack_Status = RSEDP_AM_MAX113x_ReadChannel(Slave_Address, &u16Result, uiChannel);            /* this line for 10bit ADC */
                                    }
                            
                                 pc.printf("0x%.3X ", (u16Result & 0x01ff));
                             }

                        pc.printf("\r");
                        delay_ms(500u);
    
                        keypress = pc.getc();

                    } while ((keypress != '\r') && (Ack_Status == ACK));    
            }

        if (Ack_Status != ACK)
             {
                  pc.printf("*** NO Acknowledge Receieve ***\n\r");
                  pc.printf("Operation aborted\n\r");
              }                    

         pc.printf("\n\rfinished testing the ADC on the Analogue Module\n\r");
    }




/* Test AD5263 Digital Pot */
static void test_AD5263(uint8_t Slave_Address)
    {
        sint32_t Ack_Status = 0;
        uint8_t channel = 0;
        uint32_t resistance =0;

        pc.printf("\n\rTesting the serial digital potentiometer on the Analogue Module...\n\r");
        
        pc.printf("\n\rInitialising the AD5263 device...\n\r");                                                    /* Initialise the on board ADC */
        Ack_Status = RSEDP_AM_Init_AD5263(Slave_Address) ;

        if (Ack_Status == ACK)
            {
                pc.printf("\n\rAD5263 Now Configured");
         
               
                for (channel = 1; channel < 5; channel++)
                    {
                        pc.printf("\n\rTesting the variable resistor on channel: %1d\n\r", channel);    
                        resistance = 0u;
                        do  {
                                pc.printf("\rSetting Resistance to: %5d ohms", resistance);            
                                Ack_Status = RSEDP_AM_Set_AD5263_resistance(Slave_Address, channel, resistance);
                                resistance += (AD5263_Max_Resistance/256);       
                                delay_ms(50);
                            } while ((Ack_Status == ACK) && (resistance < AD5263_Max_Resistance));
                        pc.printf("\n\r");
                    }
             }   

        if (Ack_Status == ACK)
            {
                Ack_Status = RSEDP_AM_Init_AD5263(Slave_Address) ;                                                                      /* Reset the device back to power up state */
            }

        if (Ack_Status != ACK)
             {
                  pc.printf("*** NO Acknowledge Receieve ***\n\r");
                  pc.printf("Operation aborted\n\r");
              }               
                
        pc.printf("\n\rFinished testing the AD5263 Device\n\r");
       }





/* Motor Control MC1 Tests */

/* Test the VDCLINK Voltage */
static void test_MC1_VDCLINK(void)
    {
        sint8_t keypress = 0;
        uint16_t vdc_measured = 0;        
        float vbus_voltage = 0.0f;
        
        pc.printf("\n\rTesting the VDCLINK Voltage on the MC1 Brushed DC Motor Module...\n\r");
        do    {

                vdc_measured = RSEDP_MC1_Read_VDCLINK(AN2);
                vbus_voltage = ((float)vdc_measured / VSENSE_DIVISOR);
                
                pc.printf("The ADC Voltage measured is %4dmV, which is a bus voltage of %.0fV\r", vdc_measured, vbus_voltage);
                
                keypress = pc.getc();
            } while (keypress != '\r');

        pc.printf("\n\rFinished testing the VDCLINK bus voltage\n\r");    
    }
    


/* Test the motor current sense voltage */
static void test_MC1_VSENSE(void)
    {
        sint8_t keypress = 0;
        uint16_t vdc_measured = 0;        
        float motor_current = 0.0f;
        
        pc.printf("\n\rTesting the VSENSE Motor Current Voltage Feedback on the MC1 Brushed DC Motor Module...\n\r");
        do    {

                vdc_measured = RSEDP_MC1_Read_VSENSE(AN0);
                motor_current =( (float)vdc_measured /(CURRENT_TRANSFER_RATIO * MOTOR_CURRENT_SENSE_RESISTOR));        /* Work out current in mA */

                pc.printf("The ADC Voltage measured is %4dmV, which is aprrox motor_current of %.0fmA\r", vdc_measured, motor_current);
                
                keypress = pc.getc();
            } while (keypress != '\r');

        pc.printf("\n\rFinished testing the VSENSE motor current sense input\n\r");
    }





    
/* Test the encoder inputs */    
static void test_MC1_Read_Encoders(void)
    {
        sint8_t keypress = 0;
        uint8_t switch_status0 = 0;
        uint8_t switch_status1 = 0;
        
        pc.printf("\n\rTesting the Encoder0 & 1 from P301 connector Pin8 & Pin10 on the MC1 Module...\n\r");
        pc.printf("The Encoder inputs are:\n\r");

        do    {
                switch_status0 = RSEDP_MC1_Read_Encoder0_Input(EVG6_GPIO52);
                switch_status1 = RSEDP_MC1_Read_Encoder1_Input(EVG7_GPIO54);

                if (switch_status0 == 0)
                    {
                        pc.printf("Encoder0: OPEN  ,");
                    }
                else{
                        pc.printf("Encoder0: CLOSED,");
                    }    
                
                if (switch_status1 == 0)
                    {
                        pc.printf("Encoder1: OPEN  \r");
                    }
                else{
                        pc.printf("Encoder1: CLOSED\r");
                    }    

                keypress = pc.getc();
            } while (keypress != '\r');

        pc.printf("\n\rFinished testing the Encoder inputs from P301\n\r");
    }
    

/* test the Brake control signal */
static void test_MC1_Brakes(void)
    {
        sint8_t keypress = 0;
        
        pc.printf("\n\rTesting the Brake control signal to the motor control module...\n\r");

        do    {
                pc.printf("Turning ON the brakes...\n\r");
                RSEDP_MC1_Brake(BRAKE_ON, GPIO0);                                /* Brake ON */
                pc.printf("Press any key to turn the brake OFF\n\r");                
                pc.getc();
                RSEDP_MC1_Brake(BRAKE_OFF, GPIO0);                                /* Brake OFF */
                pc.printf("Press any key to turn the brake ON again or 'Return' to terminate test\n\r");                
                keypress = pc.getc();
            } while (keypress != '\r');
                
        pc.printf("\n\rFinished testing the Brake control signal \n\r");                
    }
    
/* test the direction Signal */
static void test_MC1_Set_Direction(void)
    {
        uint8_t motor_direction = FORWARD;
        uint8_t keypress = 0;
        
        pc.printf("\n\rTesting the motor rotation direction signal...\n\r");

        do    {
                pc.printf("Setting the motor direction as:");
                if (motor_direction == FORWARD)
                    {
                        pc.printf("FORWARD\r");
                    }
                if (motor_direction == REVERSE)
                    {
                        pc.printf("REVERSE\r");
                    }

                RSEDP_MC1_Set_Motor_Direction(motor_direction, EVG1_GPIO42);            /* Set the motor direction */

                keypress = pc.getc();
                
                if (motor_direction == FORWARD)
                    {
                        motor_direction = REVERSE;
                    }
                else{  
                        motor_direction = FORWARD;
                    }
            } while (keypress != '\r');                    
                
        pc.printf("\n\rFinished motor direction test\n\r");
    }


/* Testing the motor speed control */
static void test_MC1_Speed_Ramp_UpDown(void)
    {

        uint8_t n= 0;
        float motor_current = 0;
        uint16_t vdc_measured = 0;
            
        pc.printf("\n\rTesting the speed control of the motor control module...\n\r");
        
        pc.printf("Brake turned OFF\n\r");
        RSEDP_MC1_Brake(BRAKE_OFF, GPIO0);                                /* Brake OFF */
        
        pc.printf("Ramping Up\n\r");
        
        for (float ff = 0; ff < 1.0f; ff += 0.01f)        
            {
                RSEDP_MC1_Set_Motor_Speed_Duty(ff, EVG0_GPIO40);
                pc.printf("Duty %3.0f%%\r", (ff*100));                        
                wait(0.1);
            }
            
        pc.printf("\n\rat full speed\n\r");    

        for (n = 0; n < 100; n++)
            {
                vdc_measured = RSEDP_MC1_Read_VSENSE(AN0);
                motor_current =( (float)vdc_measured /(CURRENT_TRANSFER_RATIO * MOTOR_CURRENT_SENSE_RESISTOR));        /* Work out current in mA */

                pc.printf("The ADC Voltage measured is %4dmV, which is aprrox motor_current of %.0fmA\r", vdc_measured, motor_current);        
                wait(0.1);
            }


        pc.printf("Ramping Down\n\r");

        for (float ff=1.0; ff > 0.0f; ff-= 0.01f)        
            {
                RSEDP_MC1_Set_Motor_Speed_Duty(ff, EVG0_GPIO40);
                pc.printf("Duty %3.0f%%\r",(ff*100));                                    
                wait(0.1);
            }
            
        pc.printf("\n\rMotor Stopped\n\r");            
        RSEDP_MC1_Set_Motor_Speed_Duty(0.0f, EVG0_GPIO40);        

        RSEDP_MC1_Brake(BRAKE_ON, GPIO0);                                /* Brake ON */    
        pc.printf("Brake turned ON\n\r");
        pc.printf("\n\rFinished testing the speed control of the motor control module\n\r");                
    }
    
    
static void test_MC1_Set_Speed_PWM_Duty(void)
    {
        uint8_t keypress = 0;
        float ff = 0.0f;
        
        pc.printf("\n\rSet the Motor Speed PWM Duty Ratio...\n\r");
        pc.printf("Use the '+' and '-' Keys to increase the duty and 'return' key to end.\n\r");
        
        pc.printf("Brake turned OFF\n\r");
        RSEDP_MC1_Brake(BRAKE_OFF, GPIO0);                                /* Brake OFF */
        
        
        do    {
                keypress = pc.getc();
                if (keypress == '+')
                    {
                        ff = (ff + 0.1f);
                        if (ff > 1.0) ff = 1.0;
                    }
                if (keypress == '-')
                    {
                        ff = (ff - 0.1f);
                        if (ff < 0.0) ff = 0.0;
                    }

                RSEDP_MC1_Set_Motor_Speed_Duty(ff, EVG0_GPIO40);                                                    
                pc.printf("Duty %3.0f%%\r", (ff*100));                                                                                        
            } while (keypress != '\r');
            
        pc.printf("\n\rLeaving Menu option with PWM duty still set\n\r");    

    }



/* Stopping the motor */
static void test_MC1_Stop_Motor(void)
    {
        pc.printf("\n\rRamping Down the motor\n\r");

        for (float ff=0.5; ff > 0.0f; ff-= 0.01f)        
            {
                RSEDP_MC1_Set_Motor_Speed_Duty(ff, EVG0_GPIO40);            
                wait(0.1);
            }
            
        pc.printf("Motor Stopped\n\r");            
        RSEDP_MC1_Set_Motor_Speed_Duty(0.0f, EVG0_GPIO40);            

        RSEDP_MC1_Brake(BRAKE_ON, GPIO0);                                /* Brake ON */    
        pc.printf("Brake turned ON\n\r");

        pc.printf("\n\rFinished stopping the motor\n\r");                

    }













/* General Purpose Delay Routine */
static void delay_ms(uint16_t time_delay)
    {
        uint32_t nnnn=0;

        while(time_delay--!=0x0)
            {
                for (nnnn=0;nnnn<0x08ff;nnnn++);              /* 1ms Loop */
            }    
    }

