/* Serial EEPROM I2C Type 24LC32 C Source Code Driver File */
/* ******************************************************* */

/* Version 1.00 */
/* Start Date: 25/2/2008 */



/* Include Files Here */
#include "mbed.h"                                /* mbed header file */
#include "misra_types.h"                         /* MISRA Types header file */
#include "defines.h"
#include "mbed_Port_Structure.h"                /* Port structure for MBED Module */

#include "RSEDP_CNTRL_I2C.h"                    /* Control I2C Driver */


                                        
/* Function Prototypes Here */
sint32_t RSEDP_BB_setup_24LC32(uint8_t Slave_Address);                                                                /* Configure the I2C EEPROM */
sint32_t RSEDP_BB_eeprom_store_byte(uint8_t Slave_Address, sint8_t payload, uint16_t eeaddress);                    /* Write one byte to one memory location */
sint32_t RSEDP_BB_eeprom_store_bytes(uint8_t Slave_Address, sint8_t *payload, uint8_t qty, uint16_t eeaddress);     /* Called by main to write multiple bytes to the serial EEPROM */
sint32_t RSEDP_BB_eeprom_read_byte(uint8_t Slave_Address, sint8_t *read_value, uint16_t eeaddress);                    /* Read one byte from one location */
sint32_t RSEDP_BB_eeprom_read_bytes(uint8_t Slave_Address, sint8_t *payload, uint16_t qty, uint16_t eeaddress);        /* Read several byte from one location */

static void delay_small(void);                                                                                    /* Local static 1us delay routine  */




/* Setup the serial interface to the EEPROM */
sint32_t RSEDP_BB_setup_24LC32(uint8_t Slave_Address)                                                                /* Configure the 24LC32 */
    {
        sint32_t Ack_Status = 0;
        /* Assume the I2C bus is already configured. This is normally done on mbed_Port_Structure.c */
        
        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, 0, 1);                                                    /* Ping the device */
        return Ack_Status;
    }







/* Store one byte at one address location */
sint32_t RSEDP_BB_eeprom_store_byte(uint8_t Slave_Address, sint8_t payload, uint16_t eeaddress)                    /* Write one byte to one memory location */
    {
        sint32_t Ack_Status = 0;
        sint8_t tx_array[3];                                                     /* two byte array used to hold transmission packet */
        
        eeaddress = (eeaddress & 0x0fff);                                        /* Ensure address is not more than 12 bit */
        
        tx_array[0] = (uint8_t) (eeaddress >> 8);                               /* Extract high part */                                                
        tx_array[1] = (uint8_t) eeaddress;                                         /* Extract Low address */                                                
        tx_array[2] = payload;                                                    /* second member of array is the value to be stored */
        
        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, tx_array, 3);                /* Send a data packet to a slave, record acknowledge bit */
        delay_small();
        return Ack_Status;                                                           

    }



/* Write upto 32 bytes into the EEPROM all at once */
sint32_t RSEDP_BB_eeprom_store_bytes(uint8_t Slave_Address, sint8_t *payload, uint8_t qty, uint16_t eeaddress)     /* Called by main to write multiple bytes to the serial EEPROM */
    {
        sint32_t Ack_Status = 0;
        uint8_t n=0;                                                               /* Scratch pad variable */
        sint8_t tx_array[34];                                                      /* transmission array */
        
        eeaddress = (eeaddress & 0x0fff);                                        /* Cap range to 12 bit address */

        tx_array[0]=(uint8_t) (eeaddress >> 8);                                 /* Extract high address byte */                                        
        tx_array[1]=(uint8_t) eeaddress;                                         /* Extract Low address */                                            

        for (n = 2; n < (qty + 2); n++)                                          /* Copy the transmission data into a local array */
            {
                tx_array[n] = *payload;
                payload++;
            }
        
        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, tx_array, (qty+2));        /* Send data out on I2C bus */
        delay_small();                                                             /* EEPROM Busy period - self timed */
        return Ack_Status;
    }




/* Read One Byte From An Address */
sint32_t RSEDP_BB_eeprom_read_byte(uint8_t Slave_Address, sint8_t *payload, uint16_t eeaddress)    /* Read one byte from one location */
    {
        sint32_t Ack_Status = 0;
        sint8_t tx_array[3];
        sint8_t rx_array = 0;                                                    /* Variable used to receive data from slave */
    
        eeaddress = (eeaddress & 0x0fff);                                        /* Cap address to 12 bit */
        
        tx_array[0] = (uint8_t) (eeaddress >> 8);                                /* Extract high order byte */                                            
        tx_array[1] = (uint8_t) eeaddress;                                         /* Extract low address */                                                
        
        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, tx_array, 2);                /* Set the EEPROM address */
        
//        delay_small();                                                             /* EEPROM is busy writing */
        
        if (Ack_Status == ACK)
            {
                Ack_Status = CNTRL_I2C_Master_Mode_Receive(Slave_Address, &rx_array,1);                /* Read single byte address */
            }
        
        *payload = rx_array;                                                     /* Load value from EEPROM into receive pointer */
        return Ack_Status;

    }


/* Read several byte from the memory location */
sint32_t RSEDP_BB_eeprom_read_bytes(uint8_t Slave_Address, sint8_t *payload, uint16_t qty, uint16_t eeaddress)
    {
        sint32_t Ack_Status = 0;
        sint8_t tx_array[3];
        uint8_t address_low = 0;                                                /* EEPROM high address byte */
        uint8_t address_high = 0;                                                /* EEPROM low address byte */

        eeaddress = (eeaddress & 0x0fff);                                        /* Cap to 12 bit address */
        address_high = (uint8_t) (eeaddress >> 8);                                /* Extract high order address byte */
        address_low = (uint8_t) eeaddress;                                         /* Extract Low address byte */

        tx_array[0] = address_high;                                                /* first member of array is the EEPROM high address */
        tx_array[1] = address_low;                                                /* Low byte next */
        
        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, tx_array, 2);                /* Set the EEPROM address */

        delay_small();                                                            /* Write time delay busy period */
        
        if (Ack_Status == ACK)
            {
                Ack_Status = CNTRL_I2C_Master_Mode_Receive(Slave_Address, payload, qty);                /* Read address */
            }
        return Ack_Status;
    }



/* Small 5ms delay approx */
/* Note: Change this delay to suite the write time/busy period of the EEPROM */
static void delay_small(void)
{
    uint32_t nnnn = 0;

    for(nnnn = 0; nnnn < 0x6000; nnnn++)
        {
            ;
        }
}

