/* Motor Drive Master Functions */
/* **************************** */


/* This module allows the MCU to act as a Master Controller in the system */
/* It transmitts data down to the other Slave Modules and can also read parameters from the other dsPIC Slave Modules */





/* Includes Here */
#include "mbed.h"                                /* mbed header file */
#include "misra_types.h"                         /* MISRA Types header file */
#include "defines.h"
#include "RSEDP_Slave_Address_Defines.h"        /* Slave address of I2C Peripherals */

#include "RSEDP_CNTRL_I2C.h"                    /* Control I2C Driver */
#include "RS-EDP_AM_MC2_Globals.h"              /* Global variables referenced here for MC2 Motor Drive Module */



/* Function Prototypes */
/* Write Functions */
uint8_t I2C0_dsPIC_Reset(uint8_t slave_address);
uint8_t I2C0_dsPIC_Emergency_Stop(uint8_t slave_address);
uint8_t I2C0_dsPIC_Normal_Stop(uint8_t slave_address);
uint8_t I2C0_dsPIC_Ping(uint8_t slave_address);
uint8_t I2C0_dsPIC_Set_Motor_Speed_Demand_Forward(uint8_t slave_address, uint16_t motor_speed_Demand);
uint8_t I2C0_dsPIC_Set_Motor_Speed_Demand_Reverse(uint8_t slave_address, uint16_t motor_speed_Demand);
uint8_t I2C0_dsPIC_Set_Ramp_Up_Speed(uint8_t slave_address, uint16_t ramp_up_speed);
uint8_t I2C0_dsPIC_Set_Ramp_Down_Speed(uint8_t slave_address, uint16_t ramp_down_speed);
uint8_t I2C0_dsPIC_Set_Motor_Direction(uint8_t slave_address, uint8_t direction);
uint8_t I2C0_dsPIC_Start_Motor_Rotation(uint8_t slave_address);

uint8_t I2C0_dsPIC_Set_Rotation_Counts(uint8_t slave_address, uint32_t motor_rotation_counts);
static uint8_t I2C0_dsPIC_Set_Rotation_Counts_High_Word(uint8_t slave_address, uint16_t motor_rotation_counts_high);
static uint8_t I2C0_dsPIC_Set_Rotation_Counts_Low_Word(uint8_t slave_address, uint16_t motor_rotation_counts_low);
uint8_t I2C0_dsPIC_Goto_Home(uint8_t slave_address, uint8_t home_direction, uint8_t home_speed);


/* Read Functions */
uint8_t I2C0_dsPIC_Read_Tacho_Speed_Instantaneous(uint8_t slave_address, uint8_t *rx_array);
uint8_t I2C0_dsPIC_Read_Tacho_Speed_Average(uint8_t slave_address, uint8_t *rx_array);
uint8_t I2C0_dsPIC_Read_Motor_Current_Instantaneous(uint8_t slave_address, uint8_t *rx_array);
uint8_t I2C0_dsPIC_Read_Motor_Current_Average(uint8_t slave_address, uint8_t *rx_array);
uint8_t I2C0_dsPIC_Read_Vbus_Instantaneous(uint8_t slave_address, uint8_t *rx_array);
uint8_t I2C0_dsPIC_Read_Vbus_Average(uint8_t slave_address, uint8_t *rx_array);
uint8_t I2C0_dsPIC_Read_Demand_Pot_Instantaneous(uint8_t slave_address, uint8_t *rx_array);
uint8_t I2C0_dsPIC_Read_Demand_Pot_Average(uint8_t slave_address, uint8_t *rx_array);
uint8_t I2C0_dsPIC_Read_Hall_Sensor_Positions(uint8_t slave_address, uint8_t *rx_array);
uint8_t I2C0_dsPIC_Read_Motor_Status(uint8_t slave_address, uint8_t *rx_array);
uint8_t I2C0_dsPIC_Read_Maximum_RPM(uint8_t slave_address, uint8_t *rx_array);

uint8_t I2C0_dsPIC_Read_Quad_Encoder_Counter(uint8_t slave_address, uint8_t *rx_array);
static uint8_t I2C0_dsPIC_Read_Quad_Encoder_Counter_High(uint8_t slave_address, uint8_t *rx_array);
static uint8_t I2C0_dsPIC_Read_Quad_Encoder_Counter_Low(uint8_t slave_address, uint8_t *rx_array);



/* Reset A Slave Unit */
uint8_t I2C0_dsPIC_Reset(uint8_t slave_address)
    {
        /* 00 - Reset The Processor */
        /*      data bytes to follow will be */
        /*      databyte0 - 0x55 */
        /*      databyte1 - 0xaa */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {0x00u, 0x55u, 0xaau};

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }



/* Generate an emergency Stop */
uint8_t I2C0_dsPIC_Emergency_Stop(uint8_t slave_address)
    {
        /* 01 - Emergency Stop */
        /*         data bytes to follow will be */
        /*         databyte0 - 0x00 */
        /*         databyte1 - 0x00 */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {0x01u, 0x00u, 0x00u};

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }



/* Generate a normal stop condition */
uint8_t I2C0_dsPIC_Normal_Stop(uint8_t slave_address)
    {
        /* 02 - Normal Stop */
        /*         data bytes to follow will be */
        /*         databyte0 - 0x00 */
        /*         databyte1 - 0x00 */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {0x02u, 0x00u, 0x00u};

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }


/* Ping the Processor to see if it is attached to the bus */
uint8_t I2C0_dsPIC_Ping(uint8_t slave_address)
    {
        /* 03 - Ping Stop */
        /*         data bytes to follow will be */
        /*         databyte0 - don't care */
        /*         databyte1 - don't care */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {0x03u, 0x00u, 0x00u};

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }



/* Set the motor speed demand Forward */
uint8_t I2C0_dsPIC_Set_Motor_Speed_Demand_Forward(uint8_t slave_address, uint16_t motor_speed_demand)
    {
        /* 10 - Set motor Speed Demand Forward */
        /*         data bytes to follow will be a 16bit value in the range 0 - 1023 */
        /*         databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {10u, 0x00u, 0x00u};
        Tx_Array[1] = (uint8_t)(motor_speed_demand / 256u);
        Tx_Array[2] = (uint8_t)(motor_speed_demand % 256u);

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }



/* Set the motor speed demand Reverse */
uint8_t I2C0_dsPIC_Set_Motor_Speed_Demand_Reverse(uint8_t slave_address, uint16_t motor_speed_demand)
    {
        /* 11 - Set motor Speed Demand Forward */
        /*         data bytes to follow will be a 16bit value in the range 0 - 1023 */
        /*         databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {11u, 0x00u, 0x00u};
        Tx_Array[1] = (uint8_t)(motor_speed_demand / 256u);
        Tx_Array[2] = (uint8_t)(motor_speed_demand % 256u);

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }



/* Set the ramp up speed */
uint8_t I2C0_dsPIC_Set_Ramp_Up_Speed(uint8_t slave_address, uint16_t ramp_up_speed)
    {
        /* 12 - Set Ramp Up Speed */
        /*         data bytes to follow will be a 16bit value in the range 0 - 1023 */
        /*      databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {12u, 0x00u, 0x00u};
        Tx_Array[1] = (uint8_t)(ramp_up_speed / 256u);
        Tx_Array[2] = (uint8_t)(ramp_up_speed % 256u);

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }



/* Set the ramp down speed */
uint8_t I2C0_dsPIC_Set_Ramp_Down_Speed(uint8_t slave_address, uint16_t ramp_down_speed)
    {
        /* 13 - Set Ramp Up Speed */
        /*         data bytes to follow will be a 16bit value in the range 0 - 1023 */
        /*      databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {13u, 0x00u, 0x00u};
        Tx_Array[1] = (uint8_t)(ramp_down_speed / 256u);
        Tx_Array[2] = (uint8_t)(ramp_down_speed % 256u);

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }



/* Set the motor Direction */
uint8_t I2C0_dsPIC_Set_Motor_Direction(uint8_t slave_address, uint8_t direction)
    {
        /* 14 - Set Motor Direction */
        /*         data bytes to follow will be */
        /*         databyte0 - don't care */
        /*         databyte1 - direction */
        /*         where direction = 0 - forward */
        /*         where direction = 1 - reverse */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {14u, 0x00u, 0x00u};

        Tx_Array[2] = direction;

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }



/* Motor Start */
uint8_t I2C0_dsPIC_Start_Motor_Rotation(uint8_t slave_address)
    {
        /* 20 - Start Rotation */
        /*         data bytes to follow will be */
        /*         databyte0 - 0 */
        /*         databyte1 - 0 */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {20u, 0x00u, 0x00u};

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }




/* Send a 32byte Counts number to the slave peripheral */
uint8_t I2C0_dsPIC_Set_Rotation_Counts(uint8_t slave_address, uint32_t motor_rotation_counts)
    {
        /* Set Rotation Counts */
        /* data bytes to follow will be a 32bit value - sent as two packets */

        uint8_t Ack_Error = 0u;

        Ack_Error = I2C0_dsPIC_Set_Rotation_Counts_High_Word(slave_address, (uint16_t)(motor_rotation_counts >> 16u));

        if (Ack_Error == 0u)
            {
                Ack_Error = I2C0_dsPIC_Set_Rotation_Counts_Low_Word(slave_address, (uint16_t)(motor_rotation_counts));
            }

        return Ack_Error;
    }

/* Send the first 16 bits count data across the I2C bus to the slave device */
static uint8_t I2C0_dsPIC_Set_Rotation_Counts_High_Word(uint8_t slave_address, uint16_t motor_rotation_counts_high)
    {
        /* 21 - Set Rotation Counts High word */
        /*         data bytes to follow will be a 16bit value- High word of the 32bit variables */
        /*      databyte0 - highbyte (msb byte of the 32 bit word) */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {21u, 0x00u, 0x00u};
        Tx_Array[1] = (uint8_t)(motor_rotation_counts_high / 256u);
        Tx_Array[2] = (uint8_t)(motor_rotation_counts_high % 256u);

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }

/* Send the lower order 16 bit count data across the I2C network to the slave.*/
static uint8_t I2C0_dsPIC_Set_Rotation_Counts_Low_Word(uint8_t slave_address, uint16_t motor_rotation_counts_low)
    {
        /* 22 - Set Rotation Counts Low word */
        /*         data bytes to follow will be a 16bit value- Low word of the 32bit variables */
        /*      databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {22u, 0x00u, 0x00u};
        Tx_Array[1] = (uint8_t)(motor_rotation_counts_low / 256u);
        Tx_Array[2] = (uint8_t)(motor_rotation_counts_low % 256u);

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;
    }



/* Goto home position */
uint8_t I2C0_dsPIC_Goto_Home(uint8_t slave_address, uint8_t home_direction, uint8_t home_speed)
    {
        /* 24 - Goto Home Position */
        /*         data bytes to follow will be */
        /*         databyte0 - direction */
        /*         databyte1 - home_speed (8 bit limited 1-255) */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[3] = {24u, 0u, 0u};
        Tx_Array[1] = home_direction;
        Tx_Array[2] = home_speed;
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 3u);
        return Ack_Error;

    }






/* Read Functions Below Here */

/* Read The Instantaneous RPM figure */
uint8_t I2C0_dsPIC_Read_Tacho_Speed_Instantaneous(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 50 - Read tacho speed instantaneous RPM */
        /*      data bytes to follow will be the instantaneous RPM data */
        /*         databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {50u};
        sint8_t Rx_Temp[2] = {0u, 0u};

        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);              /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];
                
            }
        return Ack_Error;
    }



/* Read The Average RPM figure */
uint8_t I2C0_dsPIC_Read_Tacho_Speed_Average(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 51 - Read tacho speed average RPM */
        /*      data bytes to follow will be the average RPM data */
        /*         databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {51u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];
            }
        return Ack_Error;
    }




/* Read The Instantaneous Motor Current */
uint8_t I2C0_dsPIC_Read_Motor_Current_Instantaneous(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 52 - Read Motor Current Instantaneous in mA */
        /*         data bytes to follow will be the instantaneous motor current in mA */
        /*         databyte0 - highbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {52u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
            }
        return Ack_Error;
    }



/* Read The Average Motor Current */
uint8_t I2C0_dsPIC_Read_Motor_Current_Average(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 53 - Read Motor Current Average */
        /*         data bytes to follow will be the computed average motor current in mA */
        /*         databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {53u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];
            }
        return Ack_Error;


    }


/* Measure the Vbus Bridge voltage */
uint8_t I2C0_dsPIC_Read_Vbus_Instantaneous(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 54 - Read Vbus Voltage Instantaneous */
        /*         data bytes to follow will be the instantaneous Vbus Power Supply Voltage to the bridge in mV */
        /*         databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {54u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];                
            }
        return Ack_Error;
    }



/* Measure the Vbus Bridge Voltage Average */
uint8_t I2C0_dsPIC_Read_Vbus_Average(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 55 - Read Vbus Voltage Instantaneous */
        /*         data bytes to follow will be the computed average Vbus Power Supply Voltage to the bridge in mV */
        /*         databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {55u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];                
            }
        return Ack_Error;
    }


/* Measure The Demand Pot Instantaneous */
uint8_t I2C0_dsPIC_Read_Demand_Pot_Instantaneous(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 56 - Read Demand Pot instantaneous */
        /*         data bytes to follow will be the instanteneous Pot Demand setting in the range 0 - 1023 */
        /*         databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {56u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];                
            }
        return Ack_Error;
    }


uint8_t I2C0_dsPIC_Read_Demand_Pot_Average(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 57 - Read Demand Pot average */
        /*         data bytes to follow will be the average Pot Demand setting in the range 0 - 1023 */
        /*         databyte0 - highbyte */
        /*         databyte1 - lowbyte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {57u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];                
            }
        return Ack_Error;
    }



/* Read The Hall Effect Sensors */
uint8_t I2C0_dsPIC_Read_Hall_Sensor_Positions(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 58 - Read Hall Sensors Positions */
        /*         data bytes to follow will be the hall sensor reading */
        /*         databyte0 - 0 */
        /*         databyte1 - hall effect sensors */
        /*         where halleffect reading in bit form is 0b 0000 0ABC, where A is the logic level on the Hall Input A ('1' or '0') etc */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {58u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];                
            }
        return Ack_Error;
    }



/* Read The Fault Status Register */
uint8_t I2C0_dsPIC_Read_Motor_Status(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 59 - Read Motor Status - run/stop/direction/Faults etc */
        /*         data bytes to follow will be the Fault Status Register */
        /*         databyte0 - Flags Status Register - high */
        /*         databyte1 - Flags Status Register - low */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {59u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];                
            }
        return Ack_Error;
    }



/* Read the Maximum RPM Information */
uint8_t I2C0_dsPIC_Read_Maximum_RPM(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 60 - Read The Maxium RPM information */
        /*         data bytes to follow will be the Maximum RPM of the motor */
        /*         databyte0 - Max RPM - high byte */
        /*         databyte1 - Max RPM - low byte */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1] = {60u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];                
            }
        return Ack_Error;
    }


/* Read The 32bit Quadrature Encoder Counter */
uint8_t I2C0_dsPIC_Read_Quad_Encoder_Counter(uint8_t slave_address, uint8_t *rx_array)
    {
         uint8_t Ack_Error = 0u;

         Ack_Error = I2C0_dsPIC_Read_Quad_Encoder_Counter_High(slave_address, rx_array);
        if (Ack_Error == 0u)
            {
                Ack_Error = I2C0_dsPIC_Read_Quad_Encoder_Counter_Low(slave_address, (rx_array + 2u));
            }
        return Ack_Error;
    }


/* Read The High Word of the 32ibt Quadrature Encoder */
static uint8_t I2C0_dsPIC_Read_Quad_Encoder_Counter_High(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 61 - Read Quadrature Counter (high byte) */
         /* data bytes to follow will be the Quadrature Encoder high bytes */
        /* databyte0 - Quadrature Encoder Counter - High Byte (byte3 MSB) */
         /* databyte1 - Quadrature Encoder Counterr - Low byte (byte2) */
        /* Reading this byte will freeze the low byte, allwoing glitchless reading of the 32 bit counter */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1u] = {61u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];                
            }
        return Ack_Error;
    }


/* Read The Low Word of the 32bit Quadrature Encoder */
static uint8_t I2C0_dsPIC_Read_Quad_Encoder_Counter_Low(uint8_t slave_address, uint8_t *rx_array)
    {
        /* 62 - Read Quadrature Counter (low bytes) */
         /* data bytes to follow will be the Quadrature Encoder low bytes */
        /* databyte0 - Quadrature Encoder Counter - High Byte (byte3 MSB) */
         /* databyte1 - Quadrature Encoder Counterr - Low byte (byte2) */
        /* Reading this byte will unlock the 32 bit counter. Read the high byte first. */
        /* Note: It is important to read the high byte first */
        uint8_t Ack_Error = 0u;
        sint8_t Tx_Array[1u] = {62u};
        sint8_t Rx_Temp[2] = {0u, 0u};
        
        Ack_Error = CNTRL_I2C_Master_Mode_Transmit(slave_address, Tx_Array, 1u);                /* Send the command Byte */

        if (Ack_Error == 0u)
            {
                Ack_Error = CNTRL_I2C_Master_Mode_Receive(slave_address, Rx_Temp, 2u);        /* Read the return data */
                rx_array[0] = (uint8_t) Rx_Temp[0];
                rx_array[1] = (uint8_t) Rx_Temp[1];                
            }
        return Ack_Error;
    }



