#pragma once

#include <mbed.h>

#define POLOLU_ENCODER_EVENT_INC  0x40
#define POLOLU_ENCODER_EVENT_DEC  0x80
#define POLOLU_ENCODER_EVENT_ERR  0xC0

#define POLOLU_ENCODER_BUFFER_SIZE 256

typedef uint8_t PololuEncoderEvent;

class PololuEncoderBuffer
{
    private:
    volatile uint32_t consumerIndex;  // The index in the ring buffer that will be consumed next.
    volatile uint32_t producerIndex;  // The index in the ring buffer that will be populated next, which is currently empty.
    volatile uint8_t buffer[POLOLU_ENCODER_BUFFER_SIZE];

    public:
    bool hasEvents() const
    {
        return consumerIndex != producerIndex;
    }
    
    bool hasSpace() const
    {
        return ((producerIndex + 1) % POLOLU_ENCODER_BUFFER_SIZE) != consumerIndex;
    }
    
    uint8_t readEvent()
    {
        uint8_t event = buffer[consumerIndex];
        consumerIndex = (consumerIndex + 1) % POLOLU_ENCODER_BUFFER_SIZE;
        return event;
    }
    
    void writeEvent(uint8_t event)
    {
        buffer[producerIndex] = event;
        producerIndex = (producerIndex + 1) % POLOLU_ENCODER_BUFFER_SIZE;    
    }
};

class PololuEncoder
{
    public:
    PololuEncoder(PinName pinNameA, PinName pinNameB, PololuEncoderBuffer * buffer, uint8_t id)
      : pinA(pinNameA), pinB(pinNameB), buffer(buffer), id(id)
    {
    }
    
    void init()
    {
        pinA.rise(this, &PololuEncoder::isr);
        pinA.fall(this, &PololuEncoder::isr);
        pinB.rise(this, &PololuEncoder::isr);
        pinB.fall(this, &PololuEncoder::isr);
        previousState = readState();
        count = 0;
    }
    
    void isr()
    {
        uint8_t event = 0;
        uint8_t newState = readState();
        switch((previousState << 2) | newState)
        {
        case 0x1: // 0b0001
        case 0x7: // 0b0111
        case 0xE: // 0b1110
        case 0x8: // 0b1000
            event = id | POLOLU_ENCODER_EVENT_INC;
            count += 1;
            break;
        case 0x2: // 0b0010
        case 0xB: // 0b1011
        case 0xD: // 0b1101
        case 0x4: // 0b0100
            event = id | POLOLU_ENCODER_EVENT_DEC;
            count -= 1;
            break;
        case 0x3: // 0b0011
        case 0x6: // 0b0110
        case 0x9: // 0b1001
        case 0xC: // 0b1100
            event = id | POLOLU_ENCODER_EVENT_ERR;
            break;
        }        
        previousState = newState;
        
        if (event != 0 && buffer != NULL && buffer->hasSpace())
        {
            buffer->writeEvent(event);
        }
    }
    
    int32_t getCount()
    {
        return count;
    }
    
    private:
    uint8_t readState()
    {
        return pinA | (pinB << 1);
    }
    
    InterruptIn pinA;
    InterruptIn pinB;
    PololuEncoderBuffer * buffer;
    uint8_t previousState;
    volatile uint8_t id;
    volatile int32_t count;
};