#ifndef _throttle_h_
#define _throttle_h_

#include "mbed.h"

/** Enumeration of different throttle operation modes. 
 * Determines how the throttle output is calculated from
 * all possible inputs.
 */
typedef enum ThrottleMode {
    off = 0, /// Throttle output is 0
    raw = 1, /// Throttle output is throttle input (possibly with calibrated curve)
    torque = 2, /// Throttle output runs a PID loop on thorque output
    cruise_raw, /// Throttle output is held to grip throttle at time cruise was set
    cruise_speed, /// Throttle output runs cruise control PID on speed.
    cruise_torque, /// Throttle output runs cruise control PID on torque.
    droid, /// Throttle output is controlled by attached Android.
    calibrate_speed  = 0x55, /// Throttle to speed calibration mode
    calibrate_torque = 0x56, /// Throttle to torque calibration mode
    calibrate_input  = 0x57, /// Grip throttle calibration mode
} ThrottleMode;

/** State machine for running cruise control functions
 */
typedef enum CruiseState {
    waiting, /// Waiting for a valid precondition to be met to start running cruise control
    transition, /// A transition or ramp between cruise off and cruise on is in progress
    running, /// Full cruise control operating    
    inhibited, /// Something (breaks) cause the cruise control to be inhibited
} CruiseState;

/** A singleton class representing all aspects of throttle control (input and output)
 * as well as regenerative breaking function and break inhibits.
 */
class Throttle {
public:
    /** Get the singleton instance
     * All variables will be used asynchronously so they must be available throughout the program
     * @param I Motor current in Amps
     * @param v_f Front wheel speed in RPM
     * @param v_r Rear  wheel speed in RPM
     * @param cadence   cadence in RPM
     * @param break_l Left break input
     * @param break_r Right break input
     * @return A pointer to the singleton Throttle object
     */
    static Throttle *getThrottle(float *I, float *v_f, float *v_r, float *cadence, AnalogIn* break_l, AnalogIn* break_r);
    /// Set the throttle control mode
    void setMode(ThrottleMode m);
    /// Set a limit on motor current for closed loop regulation
    void setILimit(float I);
    /** Set a speed limit
     * @param v Speed in RPM
     * @param enforce if true, regenerative breaking will be used to slow the
     *        bike if it's going over the speed limit with the motor off.
     */
    void setSpeedLimit(float v, bool enforce=false);
    /// Set the external input (from Droid)
    void input(float target);

private:
    /// Private constructor
    Throttle(float *I, float *v_f, float *v_r, float *cadence, AnalogIn* break_l, AnalogIn* break_r);
    /// and destructor
    ~Throttle();
    /// Control loop tick
    void onTick();

    // The singleton instance
    static Throttle* instance;

    Ticker tick;
    ThrottleMode mode;
    CruiseState state;
    float *I, *v_f, *v_r, *cadence;
    AnalogIn *brkl, *brkr;
    float target, iLimit, speedLimit;
    bool enforceSpeedLimit;
    
    static const float break_inhibit_threshold = 0.1; /// The thrshold at which the breaks cut off the throttle
};


#endif
