/*    
 * A library for MAX31855 
 *   
 * Copyright (c) 2015 Seeed Technology Limited.  
 * Author      : Yihui Xiong
 *
 * The MIT License (MIT)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


#include "MAX31855.h"

#define LOG(args...)    printf(args)

MAX31855::MAX31855(SPI& _spi, PinName _ncs) : spi(_spi), ncs(_ncs)
{
    thermocoupleT = 0.0 / 0.0;
    internalT     = 0.0 / 0.0;
    lastReadTime = 0;
}

float MAX31855::read(temperature_type_t type) {
    uint8_t buf[4];
    
    uint32_t time = us_ticker_read();
    uint32_t duration = time - lastReadTime;
    if (duration > 250000) {   // more than 250ms
        //Set CS low to start transmission (interrupts conversion)
        ncs = 0;
    
        //Read in Probe tempeature
        for (int i = 0; i < 4; i++) {
            buf[i] = spi.write(0);
        }
        
        //Set CS high to stop transmission (restarts conversion)
        ncs = 1;
        
        if (buf[1] & 1) {
            uint8_t fault = buf[3] & 0x7;
            if (fault & 0x1) {
                LOG("The thermocouple is open (no connections).\r\n"); 
            }
            if (fault & 0x2) {
                LOG("The thermocouple is short-circuited to GND.\r\n"); 
            }
            if (fault & 0x4) {
                LOG("The thermocouple is short-circuited to VCC.\r\n"); 
            }
            return 0.0 / 0.0;
        }
        
        int t1 = ((buf[0] & 0x7F) << 6) + (buf[1] >> 2);
        if (buf[0] & 0x80) {            // sign bit
            t1 = -t1;
        }
        thermocoupleT = t1 * 0.25;
        
        int t2 = ((buf[2] & 0x7F) << 4) + (buf[3] >> 4);
        if (buf[2] & 0x80) {
            t2 = -t2;
        }
        internalT = t2 * 0.0625;
    }
    
    if (THERMOCOUPLE_T == type) {
        return thermocoupleT;
    } else {
        return internalT;
    }
}


