/*********************************************************
*TimedMovement.cpp                                       *
*Author: Dan Argust                                      *
*                                                        *  
*A library of functions that can be used to control the  * 
*RenBuggy.                                               *
*********************************************************/

#ifndef TIMEDMOVEMENT_C
#define TIMEDMOVEMENT_C

/* necessary includes */
#include "mbed.h"
#include "TimedMovement.h"
#include "SevenSegmentDisplay.h"

/* PwmOut is a class included in the mbed.h library that allows
a pin to be configured as a PWM output. This is used to control
the speed of the motors. Lmotor and Rmotor are chosen names for
the pins, LeftMotorPin and RightMotorPin (see TimedMovement.h)
specify the physical pins to be used. */
PwmOut Lmotor(LeftMotorPin);
PwmOut Rmotor(RightMotorPin);

DigitalIn CurrentButtonState(p7);
bool LastButtonState = 0;

SevenSegmentDisplay seg(0);

/* Function definitions contain the code that will execute when 
the function is called. These must have the same return type
and parameters as the function declarations. */

/****************************************************************
* Function: forward()                                           *
*                                                               *
* Moves the RenBuggy directly forwards                          *
*                                                               *
* Inputs: A floating point value representing the length of     *
* time the buggy will move for                                  *
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void hold(float time)
{
    seg.DisplayDigits(0,0);
    for (float i = time;i>0;i-=0.01){
        seg.DisplayDigits((int)i/10,(int)i%10);
        //if (CurrentButtonState == 1 and LastButtonState == 0){
        //    rollDice();
        //}
        //LastButtonState = CurrentButtonState;
        wait(0.01);
    }
}

extern void forward(float time)
{
    /* Lmotor and Rmotor are set to 1.0 (i.e. the motors will 
    operate at full speed). As both motors will move at the 
    same speed, the RenBuggy will go directly forward. */
    //Lmotor = Rmotor = 1.0;
    Lmotor = Rmotor = 1.0;
    /* the program will wait here for the length of time passed 
    to the function before continuing. hold() is used to wait a
    number of seconds and display them on the seven seg display */
    hold(time);
    stop();
}

/****************************************************************
* Function: left()                                              *
*                                                               *
* Turns the RenBuggy to the left                                *
*                                                               *
* Inputs: A floating point value representing the length of     *
* time the buggy will turn for                                  *
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void left(float time)
{
    Rmotor = 1.0;
    Lmotor = 0.0;
    hold(time);
    //readButton(time);
    stop();
}

/****************************************************************
* Function: right()                                             *
*                                                               *
* Turns the RenBuggy to the right                               *
*                                                               *
* Inputs: A floating point value representing the length of     *
* time the buggy will turn for                                  *
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void right(float time)
{
    Lmotor = 1.0;
    Rmotor = 0.0;
    hold(time);
    stop();
}

/****************************************************************
* Function: stop()                                              *
*                                                               *
* Brings the RenBuggy to a complete stop                        *
*                                                               *
* Inputs: none                                                  *
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void stop()
{
    Lmotor = Rmotor = 0;
}

extern void readButton(float time)
{
    seg.DisplayDigits(0,0);
    for (float i = time;i>0;i-=0.01)
    {
        if (CurrentButtonState == 1 and LastButtonState == 0){
            rollDice();
            //stop();
            //right(2.0);
        }
        LastButtonState = CurrentButtonState;
        wait(0.01);
    }
}

extern int rollDice()
{
    int tens;
    int units;
    for (int i = 20;i>0;i--){
        tens = rand()%9;
        units = rand()%9;
        seg.DisplayDigits(tens,units);
        wait(0.04);
    }
    return tens*10+units;
}
#endif // TIMEDMOVEMENT_C