
// ----------------------------------------------------------------------
// LaserMon-TestOutput.cpp
//
// Fredric L. Rice, June 2019
//
// For testing purposes only, this test output wave form can be
// jumpered in to the laser scan analog input signal so that the
// software can be developed without the need for an ISEM.
//
// This thread will set an analog output signal to contain a porch
// and then a ramp which increments from an initial step value up 
// to 3.3 volts, with the step being at 0.0 volts.
//
// ----------------------------------------------------------------------

#include "mbed.h"                   // The mbed operating system
#include "LCD_DISCO_F429ZI.h"       // For controlling the LCD
#include "TS_DISCO_F429ZI.h"        // For controlling the touch screen
#include "LaserMon-Main.h"          // For data exported to us

// ----------------------------------------------------------------------
// We describe the shape of the waveform
//
// ----------------------------------------------------------------------

#define PORCH_WIDTH     10                          // In milliseconds
#define RAMP_WIDTH      250                         // In milliseconds
#define SCAN_LENGTH     (PORCH_WIDTH + RAMP_WIDTH)  // In milliseconds
#define STEP_HEIGHT     1.8f                        // In volts

// ----------------------------------------------------------------------
// Local data storage
//
// ----------------------------------------------------------------------

    // We create an analog output
    static AnalogOut st_testSignalOut(TEST_SIGNAL_OUT);
    
    // For diagnostic purposes to show that the test output is working
    static DigitalOut st_testSignalLED(LED1);
    
    // To drive the porch
    static bool     b_onPorch;
    static uint16_t u16_countRemaining;
    
    // Across the ramp we increment the voltage starting from the
    // initial step up to 3.3 volts. Here we compute what the
    // incremental voltage should be so that when the ramp has
    // been completed, we end up at 3.3 volts, starting from the
    // initial step.
    static const float f_stepIncrement = ((3.3f - STEP_HEIGHT) / ((float)RAMP_WIDTH - 10.0f));
    
// ----------------------------------------------------------------------
// Data that we will export globally
//
// ----------------------------------------------------------------------

    // This is the value we maintain for ramp, adding to it the
    // incremental value every millisecond. We make it global so
    // that we can use the value for software development
    float f_rampVoltage = 0.0f;
    
// ----------------------------------------------------------------------
// TestOutputThread()
//
// This is called once a millisecond however it is not a thread, the
// thread class on this board ended up with timing that could not be
// controlled so the main() loop calls us once a millisecond.
//
// ----------------------------------------------------------------------
void TestOutputThread(void)
{
    static uint8_t u8_ledTimeoutTimer = 0;

    // Do we need to turn the scan-completed LED off?
    if (u8_ledTimeoutTimer > 0)
    {
        // Count down the timer and see if it expired
        if (0 == --u8_ledTimeoutTimer)
        {
            // Turn the LED off now that 10 milliseconds has expired
            st_testSignalLED = 0;
        }
    }

    // Are we stepping out the porch?
    if (true == b_onPorch)
    {
        // Drive the voltage down to zero for porch
        st_testSignalOut.write(f_rampVoltage = 0.0f);

        // Is the porch time remaining still have some time?
        if (u16_countRemaining > 0)
        {
            // Yes, so count it down 1 millisecond and see if it expired
            if (0 == --u16_countRemaining)
            {
                // The porch time has expired so now we're on ramp
                b_onPorch = false;
                
                // Set the ramp duration
                u16_countRemaining = RAMP_WIDTH;
                
                // Set the next output voltage to the step height
                f_rampVoltage = (STEP_HEIGHT - f_stepIncrement);
            }
        }
    }
    else
    {
        // We are stepping out the ramp. We go from the step height
        // all the way up to 3.3 volts, incrementally across the
        // entire length of the ramp.
        //
        // Add the incremental value to the ramp voltage going out
        f_rampVoltage += f_stepIncrement;
        
        // Write that voltage out
        st_testSignalOut.write(f_rampVoltage / 3.3f);

        // Is there any more ramp to go out?
        if (u16_countRemaining > 0)
        {
            // There is so count it down
            u16_countRemaining--;
        }
        
        // Is there no more ramp left to step out?
        if (0 == u16_countRemaining)
        {
            // No more ramp so we're back to porch
            b_onPorch = true;
            
            // 1 millisecond from now we start counting down porch again
            u16_countRemaining = PORCH_WIDTH;
            
            // At the completion of a scan we drive the LED
            st_testSignalLED = 1;
            
            // We flash the LED for 10 milliseconds
            u8_ledTimeoutTimer = 10;
        }
    }
}

// ----------------------------------------------------------------------
// TestOutputInit()
//
// This function initializes this module's locally-held data
//
// ----------------------------------------------------------------------
void TestOutputInit(void)
{
    // Initialize this module
    b_onPorch          = true;
    u16_countRemaining = PORCH_WIDTH;
}

// End of file

