
// ----------------------------------------------------------------------
// LaserMon-TEC.cpp
//
// Fredric L. Rice, June 2019
//
// ----------------------------------------------------------------------

#include "mbed.h"                   // The mbed operating system
#include "LCD_DISCO_F429ZI.h"       // For controlling the LCD
#include "TS_DISCO_F429ZI.h"        // For controlling the touch screen
#include "LaserMon-Main.h"          // For data exported to us
#include "LaserMon-TEC.h"           // Always include ourself

// ----------------------------------------------------------------------
// Local data storage
//
// ----------------------------------------------------------------------

    // We bring in the TEC voltage to scan what it is
    static AnalogIn st_TECInput(TEC_VOLTAGE_IN);
    
    // We store the last ten TEC readings and keep a running average
    // of the last ten so that the main module may find out what the
    // average is and display it
    static uint16_t u16_TECHistory[TEC_HISTORY_COUNT_MAX];
    static uint8_t  u8_TECHistoryCount;
    static uint16_t u16_TECHistoryRunningAverage;
    
// ----------------------------------------------------------------------
// TECGetLastTenAverage()
//
// ----------------------------------------------------------------------
uint16_t TECGetLastTenAverage(void)
{
    return u16_TECHistoryRunningAverage;
}

// ----------------------------------------------------------------------
//
//
// ----------------------------------------------------------------------
void TECThread(void)
{
    uint16_t u16_TECVoltage  = 0;
    float    f_rawTECVoltage = 0.0f;
    
    // Get the current voltage
    f_rawTECVoltage = st_TECInput.read() * 3.3f;
    
    // The TEC voltage is converted from a floating point in to
    // 16 bit value with the value after the decimal moved to
    // the right by two decimal place
    u16_TECVoltage = (uint16_t)(f_rawTECVoltage * 100.0f);
    
    // Inform the main module what the TEC voltage is
    LaserMonMainInformTECVoltage(u16_TECVoltage);
    
    // Do we have lessd than our TEC history values stored so far?
    if (u8_TECHistoryCount < TEC_HISTORY_COUNT_MAX)
    {
        // Since we have fewer than our maximum history, store it
        u16_TECHistory[u8_TECHistoryCount++] = u16_TECVoltage;
    }
    else
    {
#if 1
        // Discard the oldest history value that we have
        for (uint8_t u8_thisValue = 1; u8_thisValue < TEC_HISTORY_COUNT_MAX; u8_thisValue++)
        {
            // Move the history value over to the left once
            u16_TECHistory[u8_thisValue] = u16_TECHistory[u8_thisValue - 1];
        }
        
        // Now store the latest value in to the history array
        u16_TECHistory[TEC_HISTORY_COUNT_MAX - 1] = u16_TECVoltage;
#endif
    }
    
    // Compute the average TEC value
    u16_TECHistoryRunningAverage = 0;
    
    for (uint8_t u8_thisValue = 0; u8_thisValue < u8_TECHistoryCount; u8_thisValue++)
    {
        u16_TECHistoryRunningAverage += u16_TECHistory[u8_thisValue];
    }
    
    // Compute the average
    u16_TECHistoryRunningAverage /= u8_TECHistoryCount;
}

// ----------------------------------------------------------------------
//
//
// ----------------------------------------------------------------------
void TECInit(void)
{
    // Initialize locally-held data
    u8_TECHistoryCount            = 0;
    u16_TECHistoryRunningAverage  = 0;
}

// End of file

