
// ----------------------------------------------------------------------
// NextGen-Exerciser-Main.cpp
//
// Fredric L. Rice, May 2019
//
// o The LCD is cleared and configured
// o A canned AC and DC 2f water-in-ambient-air spectrum is plotted 
//   on the LCD after the spectrum is "normalized" by making each of
//   the sample values positive and then compressed so that it fits 
//   in the display
// o A thread is created to perform Modbus against a CPUM device
// o A thread is created to perform Modbus Tunneling with foreign
//   frames against an ISEM device
// o The main() thread wakes up once a second and performs whatever
//   other functionality might be desired
//
// ----------------------------------------------------------------------

#include "mbed.h"                       // The mbed operating system
#include "NextGen-Exerciser-Modbus.h"   // For the CPUM Modbus thread
#include "NextGen-Exerciser-Tunnel.h"   // For the Modbus Tunnel thread
#include "NextGen-Exerciser-Defines.h"  // For defined constants and MACROs
#include "LCD_DISCO_F429ZI.h"           // For controlling the LCD

// ----------------------------------------------------------------------
// Defined constants that we will use here
//
// ----------------------------------------------------------------------

#define LCD_WIDTH       240
#define LCD_HEIGHT      320

// ----------------------------------------------------------------------
// Allocate local data storage
//
// ----------------------------------------------------------------------

    // Describe a canned 2f AC spectrum  consisting of 512 floating point samples
    // Note that the values after the decimal point are not really needed, they
    // get discarded when the spectrum values are plotted as integer values.
    // These values were extracted after Spectra Processor decimated a spectrum
    // of 65536 samples. We "normalize" the values to make them all positive and
    // later we take the 512 samples and further "decimate" by averaging every 
    // two samples, resuling in 256 final values. The 256 values get plotted on 
    // to the LCD along the "width" of the display which is 320 pixels, leaving
    // 64 pixels to the right of the plot free to be used for other things.
    static float f_decimatedACSpectrum[] =
    {
        -1.502448670,9.237028121,18.872425000,27.111267000,33.651569300,38.191074300,40.437301633,40.118068700,
        36.992050100,30.858972544,21.569023133,9.031014442,-6.781046860,-25.823335644,-47.979820200,-73.062263400,
        -100.812744110,-130.908676110,-162.970184330,-196.569778440,-231.243743000,-266.505157440,-301.857879000,-336.811126000,
        -370.893982000,-403.669342000,-434.746948220,-463.794647220,-490.547790000,-514.816223110,-536.488708440,-555.534729000,
        -572.003357000,-586.019348110,-597.776916000,-607.531311000,-615.588317000,-622.292602000,-628.014892000,-633.139160110,
        -638.049499000,-643.118225000,-648.694641110,-655.095459000,-662.597473110,-671.431945000,-681.781494110,-693.779174000,
        -707.509704000,-723.013183000,-740.289856000,-759.307129000,-780.007019000,-802.314636220,-826.146362330,-851.418640110,
        -878.054870000,-905.992431000,-935.187500000,-965.618835440,-997.289367000,-1030.226684400,-1064.481080000,-1100.122430000,
        -1137.235840000,-1175.915650000,-1216.259270000,-1258.360100000,-1302.301140000,-1348.147820000,-1395.942993300,-1445.701293300,
        -1497.406372200,-1551.007934400,-1606.421874400,-1663.530761100,-1722.186401100,-1782.213134400,-1843.413452200,-1905.572753300,
        -1968.465942200,-2031.863770000,-2095.539062200,-2159.272210000,-2222.855460000,-2286.097900000,-2348.825683300,-2410.885253300,
        -2472.140130000,-2532.470214400,-2591.767090000,-2649.928460000,-2706.852294400,-2762.428950000,-2816.533440000,-2869.017333300,
        -2919.700190000,-2968.362304400,-3014.738520000,-3058.511710000,-3099.308830000,-3136.698974400,-3170.191162200,-3199.236572200,
        -3223.229492200,-3241.513671100,-3253.386710000,-3258.107421100,-3254.905273300,-3242.988030000,-3221.553710000,-3189.798340000,
        -3146.928950000,-3092.171142200,-3024.780761100,-2944.051020000,-2849.322510000,-2739.990234400,-2615.509270000,-2475.402343300,
        -2319.263183300,-2146.761962200,-1957.648560000,-1751.756340000,-1529.004510000,-1289.401733300,-1033.048700000,-760.140625000,
        -470.970947220,-165.934036220,154.471374000,489.638946000,838.851257330,1201.276733300,1575.967041100,1961.854370000,
        2357.750480000,2762.345940000,3174.210693300,3591.797120000,4013.442382200,4437.375480000,4861.724610000,5284.524414400,
        5703.730950000,6117.232910000,6522.867180000,6918.436030000,7301.727050000,7670.530761100,8022.663080000,8355.986320000,
        8668.431640000,8958.021484400,9222.888671100,9461.302734400,9671.681640000,9852.618164400,10002.893500000,10121.489200000,
        10207.603511000,10260.660100000,10280.314400000,10266.458900000,10219.221600000,10138.971600000,10026.307611000,9882.054680000,
        9707.252930000,9503.147460000,9271.170890000,9012.928710000,8730.179680000,8424.816400000,8098.844230000,7754.357421100,
        7393.519531100,7018.538080000,6631.642570000,6235.063470000,5831.010742200,5421.653800000,5009.104003300,4595.396972200,
        4182.478510000,3772.192871100,3366.271484400,2966.325440000,2573.839350000,2190.168450000,1816.535644400,1454.033320000,
        1103.623901100,766.144043000,442.308807330,132.717117330,-162.142257000,-441.884216330,-706.220947220,-954.955566440,
        -1187.975830000,-1405.248901100,-1606.816040000,-1792.787720000,-1963.340570000,-2118.713134400,-2259.203370000,-2385.165283300,
        -2497.007812200,-2595.190430000,-2680.222412200,-2752.658203300,-2813.093994400,-2862.164550000,-2900.537590000,-2928.907950000,
        -2947.991700000,-2958.517822200,-2961.222650000,-2956.839350000,-2946.091552200,-2929.683350000,-2908.292230000,-2882.560540000,
        -2853.090080000,-2820.434082200,-2785.094482200,-2747.517333300,-2708.092041100,-2667.150140000,-2624.968500000,-2581.769770000,
        -2537.729730000,-2492.980224400,-2447.618164400,-2401.711420000,-2355.307120000,-2308.439690000,-2261.137940000,-2213.432861100,
        -2165.363520000,-2116.981440000,-2068.355224400,-2019.572380000,-1970.740350000,-1921.984980000,-1873.449340000,-1825.289550000,
        -1777.670043300,-1730.759150000,-1684.721923300,-1639.715080000,-1595.880493300,-1553.340080000,-1512.190790000,-1472.501342200,
        -1434.309570000,-1397.621210000,-1362.410760000,-1328.622680000,-1296.175292200,-1264.964600000,-1234.869873300,-1205.759880000,
        -1177.499140000,-1149.954711100,-1123.002800000,-1096.534050000,-1070.459472200,-1044.713134400,-1019.255790000,-994.075744000,
        -969.188293440,-944.634338330,-920.477295000,-896.798462000,-873.692077000,-851.259216330,-829.601562440,-808.814880330,
        -788.983154220,-770.172973000,-752.429016110,-735.770691000,-720.190063440,-705.651245110,-692.091308000,-679.422546330,
        -667.536193000,-656.307312000,-645.600402000,-635.275817000,-625.196350000,-615.233886000,-605.275512000,-595.229004000,
        -585.027282000,-574.631591000,-564.033447220,-553.254821000,-542.346923000,-531.387756330,-520.477966330,-509.735595000,
        -499.290527330,-489.277618440,-479.830261220,-471.073608330,-463.118743000,-456.057190000,-449.956909110,-444.859253000,
        -440.777649000,-437.697479220,-435.577880000,-434.354522000,-433.944000220,-434.248993000,-435.164337110,-436.583465000,
        -438.404754000,-440.537933330,-442.909454330,-445.467254000,-448.184082000,-451.059661000,-454.121124220,-457.422180110,
        -461.040466330,-465.073761000,-469.634948000,-474.846222000,-480.832397440,-487.714386000,-495.602508000,-504.590667000,
        -514.750915000,-526.129760000,-538.745178220,-552.586059000,-567.612426000,-583.757812440,-600.933227000,-619.031982440,
        -637.936096110,-657.523071220,-677.673339000,-698.277160000,-719.241394000,-740.495056110,-761.993713330,-783.721740000,
        -805.693237330,-827.950134220,-850.558349000,-873.601989000,-897.174682000,-921.369873000,-946.269043000,-971.929199220,
        -998.369567000,-1025.558350000,-1053.400024400,-1081.723632200,-1110.273071100,-1138.698974400,-1166.554560000,-1193.292602200,
        -1218.267211100,-1240.738770000,-1259.881470000,-1274.795280000,-1284.520751100,-1288.056152200,-1284.377800000,-1272.461420000,
        -1251.305900000,-1219.956910000,-1177.530883300,-1123.238160000,-1056.406250000,-976.499206000,-883.136718000,-776.109497000,
        -655.391845000,-521.151001000,-373.752929000,-213.764404220,-41.951789800,140.723938000,333.115356440,533.900024440,
        741.594665000,954.571960440,1171.079711100,1389.261840000,1607.181152200,1822.842651100,2034.219230000,2239.276360000,
        2435.997802200,2622.412110000,2796.617180000,2956.805420000,3101.288080000,3228.517822200,3337.110590000,3425.865234400,
        3493.781000000,3540.073974400,3564.189453300,3565.812744400,3544.875244400,3501.559570000,3436.298820000,3349.772700000,
        3242.901850000,3116.834472200,2972.933350000,2812.756340000,2638.033691100,2450.644531100,2252.587640000,2045.952140000,
        1832.885253300,1615.559814400,1396.141350000,1176.755120000,959.454528000,746.191284110,538.787902000,338.913055440,
        148.060974110,-32.465728700,-201.568542440,-358.361267000,-502.173584000,-632.550109000,-749.244995110,-852.212768000,
        -941.595398000,-1017.705932200,-1081.010010000,-1132.105340000,-1171.699584400,-1200.587760000,-1219.629394400,-1229.727172200,
        -1231.805170000,-1226.789910000,-1215.592890000,-1199.095330000,-1178.135864400,-1153.499633300,-1125.911621100,-1096.030510000,
        -1064.447021100,-1031.682000000,-998.188537000,-964.353820000,-930.503479000,-896.906066000,-863.778503440,-831.291504000,
        -799.575500440,-768.725891110,-738.808532000,-709.864502000,-681.914245000,-654.961486000,-628.996582000,-603.999267000,
        -579.941101000,-556.787353000,-534.498901330,-513.033691440,-492.348114000,-472.398071220,-453.140289330,-434.533050000,
        -416.537292440,-399.117187440,-382.240600000,-365.879577000,-350.010192000,-334.612487000,-319.669769220,-305.167755110,
        -291.093566000,-277.434234000,-264.175262440,-251.299041000,-238.783386220,-226.600113000,-214.714141000,-203.082885000,
        -191.656234000,-180.377243000,-169.183487000,-158.009185000,-146.788055440,-135.456863440,-123.959488000,-112.251495330,
        -100.304824000,-88.112571711,-75.693458500,-63.095802300,-50.400745400,-37.724391933,-25.218734744,-13.071174622
    } ;

    static float f_decimatedDCSpectrum[] =
    {
        164622.859, 156185.187, 130679.648, 94457.766, 60629.086, 35816.664, 20260.646, 12169.420,
        9521.371, 9680.042, 10389.553, 14924.355, 29797.771, 54048.340, 79677.930, 100584.007,
        114916.859, 123130.789, 126640.937, 127108.539, 125978.539, 124264.656, 122565.765, 121160.593,
        120120.234, 119405.492, 118934.531, 118622.812, 118402.031, 118224.390, 118060.507, 117896.296,
        117727.086, 117552.828, 117375.304, 117196.453, 117017.570, 116837.726, 116656.039, 116473.492,
        116289.906, 116104.265, 115915.937, 115723.570, 115526.664, 115325.656, 115120.421, 114911.429,
        114697.828, 114476.804, 114247.992, 114013.179, 113771.867, 113522.140, 113264.578, 113002.109,
        112735.640, 112462.671, 112182.187, 111892.171, 111590.554, 111280.000, 110962.125, 110639.429,
        110312.679, 109975.781, 109623.398, 109255.132, 108878.789, 108498.742, 108109.039, 107706.664,
        107291.461, 106863.039, 106420.086, 105960.101, 105483.578, 104994.242, 104495.437, 103984.203,
        103457.976, 102917.992, 102363.515, 101794.343, 101214.031, 100629.211, 100040.492, 99439.859,
        98819.461, 98173.063, 97503.469, 96825.117, 96144.188, 95452.000, 94738.039, 93996.977,
        93228.984, 92435.734, 91625.875, 90815.219, 90004.086, 89177.555, 88331.219, 87471.523,
        86607.555, 85742.336, 84864.008, 83967.336, 83058.867, 82140.016, 81212.930, 80285.523,
        79358.758, 78424.992, 77494.000, 76579.289, 75662.469, 74731.563, 73808.039, 72908.133,
        72025.875, 71144.391, 70256.500, 69372.094, 68501.109, 67648.109, 66813.898, 65996.867,
        65195.641, 64414.758, 63665.418, 62947.723, 62251.930, 61577.820, 60930.938, 60310.703,
        59717.070, 59153.746, 58622.043, 58124.535, 57664.281, 57239.590, 56849.543, 56495.852,
        56177.289, 55892.000, 55643.016, 55431.777, 55257.840, 55124.254, 55029.434, 54971.840,
        54955.445, 54979.289, 55040.957, 55139.879, 55278.230, 55457.840, 55673.457, 55925.805,
        56220.500, 56554.961, 56925.184, 57330.996, 57773.441, 58253.074, 58770.402, 59326.258,
        59917.891, 60538.234, 61187.504, 61875.184, 62605.102, 63370.285, 64161.137, 64974.477,
        65816.297, 66691.320, 67592.148, 68503.984, 69422.414, 70364.266, 71343.883, 72351.906,
        73372.016, 74390.320, 75401.859, 76421.227, 77464.242, 78523.742, 79582.586, 80642.227,
        81714.758, 82804.367, 83906.422, 85006.211, 86092.281, 87169.992, 88245.125, 89310.359,
        90360.758, 91407.516, 92457.242, 93497.219, 94524.414, 95552.211, 96577.430, 97588.609,
        98588.539, 99575.063, 100543.234, 101498.179, 102434.992, 103344.476, 104234.867, 105114.281,
        105982.656, 106839.539, 107682.593, 108518.468, 109349.765, 110166.625, 110967.500, 111753.843,
        112524.437, 113279.453, 114016.820, 114732.929, 115433.171, 116129.640, 116820.101, 117493.351,
        118144.578, 118774.367, 119390.500, 119997.554, 120589.562, 121168.031, 121741.507, 122308.328,
        122861.468, 123400.164, 123925.187, 124437.125, 124944.039, 125448.539, 125940.484, 126416.187,
        126881.367, 127340.125, 127790.164, 128227.226, 128651.578, 129064.890, 129469.812, 129867.937,
        130259.703, 130648.023, 131031.523, 131406.046, 131770.140, 132124.406, 132471.937, 132814.875,
        133153.359, 133487.703, 133816.296, 134134.171, 134439.421, 134738.953, 135040.187, 135338.218,
        135628.468, 135915.687, 136198.750, 136474.656, 136746.062, 137011.234, 137270.937, 137531.343,
        137787.781, 138033.875, 138274.046, 138511.078, 138741.453, 138967.078, 139194.343, 139418.796,
        139635.859, 139849.718, 140059.359, 140263.546, 140464.187, 140661.343, 140856.937, 141051.328,
        141241.859, 141427.875, 141610.156, 141789.250, 141965.359, 142139.234, 142310.125, 142476.343,
        142639.187, 142799.046, 142954.359, 143106.234, 143255.687, 143402.343, 143547.250, 143690.609,
        143830.875, 143967.234, 144099.421, 144228.046, 144353.625, 144475.734, 144594.656, 144710.671,
        144823.093, 144931.390, 145036.046, 145138.921, 145240.593, 145339.421, 145434.562, 145525.906,
        145612.765, 145694.562, 145771.609, 145845.453, 145916.843, 145984.953, 146049.296, 146109.531,
        146165.484, 146216.843, 146262.718, 146304.031, 146341.515, 146373.906, 146400.953, 146423.250,
        146440.265, 146451.703, 146457.453, 146456.968, 146450.687, 146439.015, 146420.750, 146395.453,
        146364.703, 146329.000, 146286.328, 146235.890, 146179.187, 146116.390, 146046.562, 145970.250,
        145888.875, 145802.343, 145709.578, 145611.015, 145507.875, 145400.656, 145289.718, 145175.437,
        145058.906, 144941.578, 144823.593, 144704.640, 144585.046, 144464.984, 144345.546, 144229.687,
        144119.281, 144013.609, 143912.671, 143818.609, 143732.531, 143654.062, 143584.265, 143524.062,
        143472.843, 143430.656, 143398.421, 143376.796, 143365.031, 143361.171, 143364.421, 143375.312,
        143393.546, 143417.750, 143448.921, 143488.828, 143537.078, 143593.468, 143657.265, 143727.843,
        143806.734, 143893.125, 143984.609, 144081.359, 144184.625, 144293.546, 144406.218, 144524.500,
        144650.468, 144782.187, 144918.953, 145061.843, 145211.578, 145369.562, 145536.781, 145712.218,
        145895.156, 146085.937, 146284.156, 146491.125, 146708.437, 146934.328, 147168.796, 147413.734,
        147668.906, 147933.296, 148206.078, 148487.953, 148777.734, 149071.968, 149373.265, 149686.093,
        150007.109, 150328.703, 150648.953, 150969.625, 151287.671, 151603.171, 151921.078, 152239.421,
        152552.781, 152860.218, 153163.453, 153463.234, 153758.718, 154050.921, 154341.515, 154628.250,
        154907.796, 155179.656, 155444.531, 155703.500, 155957.656, 156205.265, 156446.359, 156685.718,
        156922.781, 157152.437, 157374.281, 157590.484, 157802.796, 158010.562, 158211.375, 158407.156,
        158600.015, 158787.093, 158967.703, 159144.312, 159317.187, 159486.218, 159652.437, 159816.218,
        159976.906, 160135.031, 160291.250, 160443.359, 160590.359, 160733.843, 160875.171, 161015.390,
        161154.515, 161291.843, 161427.000, 161559.953, 161689.953, 161815.781, 161939.203, 162063.375,
        162187.312, 162307.875, 162424.328, 162538.328, 162651.750, 162765.296, 162878.078, 162987.562,
        163093.531, 163199.171, 163305.312, 163410.593, 163514.609, 163617.453, 163720.437, 163824.093,
        163926.906, 164028.125, 164126.515, 164224.531, 164334.281, 164304.281, 163889.750, 164209.968
    } ;
   
    // We will be using the LCD so instantiate an object locally
    static LCD_DISCO_F429ZI st_lcd;
    
// ----------------------------------------------------------------------
// MainProcessDecimatedSpectrum()
//
// This function will:
//      o Search through the AC spectra and the DC spectra looking for
//        the smallest and the highest numeric values in both
//      o Convert the smallest number in to an absolute value by removing
//        the sign bit using the ANSI absolute function
//      o Add that now-positive smallest number to every sample in the
//        AC and DC spectrums so that each value in the spectrums are
//        positive values
//      o Add that smallest value to the known largest value so that it
//        agrees with the rest of the spectrum again
//      o Compute a ratio derived by dividing the ;argest known value 
//        by the "width" of the screen
//      o Goes through all 512 samples and averages them two by two,
//        compressing the spectrum in place (in the same array of values)
//        so that we end up with 256 samples which can fit on the LCD
//      o Divides each of the 256 samples by the radio so that each of
//        the samples will fit on the LCD
//
// ----------------------------------------------------------------------
static void MainProcessDecimatedSpectrum(void)
{
    uint16_t u16_searchLoop    = 0;
    uint32_t u32_sampleACRatio = 0;
    uint32_t u32_sampleDCRatio = 0;
    uint16_t u16_resultIndex   = 0;
    float    f_smallACNumber   = 10000000.0f;
    float    f_largeACNumber   = 0.0f;
    float    f_smallDCNumber   = 10000000.0f;
    float    f_largeDCNumber   = 0.0f;
    
    // Search through the AC spectrum for the smallest and the largest values
    for (u16_searchLoop = 0; u16_searchLoop < 512; u16_searchLoop++)
    {
        // First do the AC SPectrum
        if (f_decimatedACSpectrum[u16_searchLoop] < f_smallACNumber)
        {
            // This value is the new lowest number
            f_smallACNumber = f_decimatedACSpectrum[u16_searchLoop];
        }
        
        if (f_decimatedACSpectrum[u16_searchLoop] > f_largeACNumber)
        {
            // This value is the new highest number
            f_largeACNumber = f_decimatedACSpectrum[u16_searchLoop];
        }
        
        // Now do the DC spectrum
        if (f_decimatedDCSpectrum[u16_searchLoop] < f_smallDCNumber)
        {
            // This value is the new lowest number
            f_smallDCNumber = f_decimatedDCSpectrum[u16_searchLoop];
        }
        
        if (f_decimatedDCSpectrum[u16_searchLoop] > f_largeDCNumber)
        {
            // This value is the new highest number
            f_largeDCNumber = f_decimatedDCSpectrum[u16_searchLoop];
        }        
    }
    
    // Is the smallest AC number negative? We know that it is but we check
    if (f_smallACNumber < 0.0f)
    {
        // Make the value absolute so that it is positive
        f_smallACNumber = abs(f_smallACNumber);
    }
    
    // Is the smallest DC number negative?
    if (f_smallDCNumber < 0.0f)
    {
        // Make the value absolute so that it is positive
        f_smallDCNumber = abs(f_smallDCNumber);
    }
     
    // To make all values positive, add the newly-derived positive value.
    // This makes all numbers positive and ensures that the smallest
    // value in the spectra is zero.
    for (u16_searchLoop = 0; u16_searchLoop < 512; u16_searchLoop++)
    {
        // Do the AC spectrum 
        f_decimatedACSpectrum[u16_searchLoop] += f_smallACNumber;

        // Do the DC spectrum 
        f_decimatedDCSpectrum[u16_searchLoop] += f_smallDCNumber;
    }
    
    // Also add the newly-positive number to our largest known value
    f_largeACNumber += f_smallACNumber;
    f_largeDCNumber += f_smallDCNumber;
    
    // Determine the ratio to use, the LCD screen is 240 by 320 so we 
    // take the highest number value in the spectra and divide by 240
    // for the verticle plot, and we have 512 samples which will be
    // plotted across the 320 by averaging every 2 samples to yield
    // 256 horizontal positions. We plot with the screen laying on
    // its side
    u32_sampleACRatio = (f_largeACNumber / LCD_WIDTH);
    u32_sampleDCRatio = (f_largeDCNumber / LCD_WIDTH);
    
    // Go through each pairs of samples and average them and then
    // divide the result by the ratio. Step through in twos
    for (u16_searchLoop = 0; u16_searchLoop < 512; u16_searchLoop += 2, u16_resultIndex++)
    {
        // Take the next two AC samples and average them, putting the result
        // in to the next result index of the same buffer we're processing
        f_decimatedACSpectrum[u16_resultIndex] =
            (f_decimatedACSpectrum[u16_searchLoop] + f_decimatedACSpectrum[u16_searchLoop]) / 2;
    
        // Divide the sample value by the ratio
        f_decimatedACSpectrum[u16_resultIndex] /= u32_sampleACRatio;
        
        // Take the next two DC samples and average them, putting the result
        // in to the next result index of the same buffer we're processing
        f_decimatedDCSpectrum[u16_resultIndex] =
            (f_decimatedDCSpectrum[u16_searchLoop] + f_decimatedDCSpectrum[u16_searchLoop]) / 2;
    
        // Divide the sample value by the ratio
        f_decimatedDCSpectrum[u16_resultIndex] /= u32_sampleDCRatio;
    }
}

// ----------------------------------------------------------------------
// MainPlotFinalSpectrum()
//
// This function will step through the "normalized" and compressed AC
// and DC spectrums consisting of 256 samples and will plot a line from
// one sample to the next on the LCD
//
// ----------------------------------------------------------------------
static void MainPlotFinalSpectrum(void)
{
    uint16_t u16_searchLoop = 0;

    // Plot the spectrum values, plotting a line from point to point
    for (u16_searchLoop = 0; u16_searchLoop < ((512 / 2) - 1); u16_searchLoop++)
    {
        // Ensure that we actually fit. The AC sample should
        if (f_decimatedACSpectrum[u16_searchLoop] <= LCD_WIDTH)
        {
            st_lcd.DrawLine(f_decimatedACSpectrum[u16_searchLoop], u16_searchLoop, 
                f_decimatedACSpectrum[u16_searchLoop + 1], u16_searchLoop + 1);
        }
        else
        {
            // Mathematically we topped-out past the maximum some how
            st_lcd.DrawPixel(0, u16_searchLoop, LCD_COLOR_WHITE);
        }

        // Ensure that we actually fit. The DC sample should
        if (f_decimatedDCSpectrum[u16_searchLoop] <= LCD_WIDTH)
        {
            st_lcd.DrawLine(f_decimatedDCSpectrum[u16_searchLoop], u16_searchLoop, 
                f_decimatedDCSpectrum[u16_searchLoop + 1], u16_searchLoop + 1);
        }
        else
        {
            // Mathematically we topped-out past the maximum some how
            st_lcd.DrawPixel(0, u16_searchLoop, LCD_COLOR_WHITE);
        }
    }
}

// ----------------------------------------------------------------------
// MainInit()
//
// This function will:
//      o Clear and initialize the LCD
//      o Process the scanned decimated 2f AC and DC spectrums
//      o Plot the processed spectrums
//
// ----------------------------------------------------------------------
static void MainInit(void)
{
    // Bring the LCD up and set its various colors
    st_lcd.Clear(LCD_COLOR_BLACK);
    st_lcd.SetBackColor(LCD_COLOR_BLACK);
    st_lcd.SetTextColor(LCD_COLOR_WHITE);
    
    // Process the decimated spectrum and then plot it
    MainProcessDecimatedSpectrum();
    MainPlotFinalSpectrum();
}

// ----------------------------------------------------------------------
// MainThread()
//
// Called once a second.
//
// ----------------------------------------------------------------------
static void MainThread(void)
{
}

// ----------------------------------------------------------------------
// main()
//
// This is the main entry point which the operating system will load 
// and execute on power-up after the OS completes its own start-up
// functionality.
//
// The function will:
//      o Initialize this module's functionality
//      o Launch the MODBUS CPUM thread
//      o Launch the Modbus Tunnel thread
//      o Enter in to a forever loop which should never terminate,
//        waking up once a second to perform any additional desired
//        functionality
//
// ----------------------------------------------------------------------
int main(void)
{
    // Perform local module initialization, if any
    MainInit();

    // Initialize and launch the MODBUS CPUM thread
    ModbusInit();
    
    // Initialize and launch the MODBUS Tunnel thread
    TunnelInit();

    // Enter in to a forever loop which wakes up once a second
    while (true)
    {
        // Sleep for one second
        wait(1.0);
   
        MainThread();
    }
}

// End of file

