
// ----------------------------------------------------------------------
// SecurityUnlockDemo-Main.cpp
//
// Fredric L. Rice, June 2019
//
// This is a demonstration of how to build a keypad on the liquid 
// crystal display and then use the touch screen as a keypad to enter
// an access code which grants access to other functionality. Two
// levels of access is possible, with the more permissive access a
// longer series of digits than the lesser-permissive access code.
//
// The keypad is built using position information in a table with some
// fairly easy ways to move the keypad around on the display.
//
// The demonstration code also allows for the push button to be used
// to key in Moorse Code pulses which are timed and then checked to
// see if they match an expected access code for Moorse access.
//
// ----------------------------------------------------------------------

#include "mbed.h"                           // The mbed operating system
#include "LCD_DISCO_F429ZI.h"               // For controlling the LCD
#include "TS_DISCO_F429ZI.h"                // For controlling the touch screen
#include "SecurityUnlockDemo-Main.h"        // Always include our own header
#include "SecurityUnlockDemo-Keypad.h"      // For all of the keypad functionality
#include "SecurityUnlockDemo-Moorse.h"      // For all of the Moorse Code functionality
#include "SecurityUnlockDemo-Animation.h"   // For performing animation

// ----------------------------------------------------------------------
// Define global data storage which we will export
//
// ----------------------------------------------------------------------

    // We will be using the LCD so instantiate an object locally
    LCD_DISCO_F429ZI st_lcd;

    // We will be using the touch screen so instantiate an object
    TS_DISCO_F429ZI st_touchScreen;
    
// ----------------------------------------------------------------------
// Local data storage this module will hold
//
// ----------------------------------------------------------------------

    // Because the mbed randomizer is not very good, we maintain the
    // following counter which is incremented every time the main
    // thread executes so that we can use it to assist in randoming
    // our numbers for the animation
    static uint32_t u32_randomSeeder;
    
    // We keep track of what access level we may have granted
    static uint8_t u8_thisAccessLevel;
    
// ----------------------------------------------------------------------
// MainInit()
//
// This function will:
//      o Initialize the module's locally-held data
//      o Clear the LCD and set the display to WHITE
//      o Set the default character font size
//      o Initialize the touch screen by polling the size of the LCD
//      o Sets the push button to have no pull-up
//
// ----------------------------------------------------------------------
static void MainInit(void)
{
    // Initialize locally-held data
    u32_randomSeeder   = 0;
    u8_thisAccessLevel = 0;
    
    uint8_t u8_TouchScreenStatus = 0;
        
    // Bring the LCD up 
    st_lcd.Clear(LCD_COLOR_WHITE);
    
    // Set the default font size
    BSP_LCD_SetFont(&Font16);
    
    // Initialize the touch screen
    u8_TouchScreenStatus = st_touchScreen.Init(st_lcd.GetXSize(), st_lcd.GetYSize());
    
    if (TS_OK != u8_TouchScreenStatus)
    {
        st_lcd.DisplayStringAt(1, LINE(ENTERED_KEYS_LINE), 
            (uint8_t *)"Touch screen failed", CENTER_MODE);    
    }
}

// ----------------------------------------------------------------------
// MainGrantAccess()
//
// This function is invoked to display what access level has been 
// granted.
//
// ----------------------------------------------------------------------
void MainGrantAccess(uint8_t u8_accessLevel)
{
    uint8_t au8_reportString[21] = { 0 };
    
    // Clear the display 
    st_lcd.Clear(LCD_COLOR_WHITE);
    
    // Build a report to offer    
    (void)sprintf((char *)au8_reportString, "Level %u", u8_accessLevel);
    
    // Display the level of access that was granted
    st_lcd.DisplayStringAt(1, LINE(1), (uint8_t *)"Access granted", CENTER_MODE);
    st_lcd.DisplayStringAt(1, LINE(2), au8_reportString, CENTER_MODE);
    st_lcd.DisplayStringAt(1, LINE(3), (uint8_t *)"Press RESET to exit", CENTER_MODE);
    
    // Perform the screen animation
    AnimationPerformAnimation(u32_randomSeeder);
    
    // Flag the fact that we have granted access already
    u8_thisAccessLevel = u8_accessLevel;
}

// ----------------------------------------------------------------------
// MainThread()
//
// Called ten times a second.
//
// This function will:
//      o Get the state of the touch screen
//      o Check to see if a touch screen touch was detected
//      o Extract the X and Y coordinates of the screen touch
//      o Call a function which will handle the screen touch
//      o Call a function to monitor the push button to drive the
//        Moorse Code access functionality
//
// ----------------------------------------------------------------------
static void MainThread(void)
{
    uint16_t        u16_screenX          = 0;
    uint16_t        u16_screenY          = 0;
    TS_StateTypeDef st_touchScreenState;
    
    // Have we not offered access already?
    if (0 == u8_thisAccessLevel)
    {
        // Get the status of the Touch Screen interface
        st_touchScreen.GetState(&st_touchScreenState);
        
        // Has the screen been touched?
        if (st_touchScreenState.TouchDetected)
        {
            // It has been, extract the X and Y coordinates touched
            u16_screenX = st_touchScreenState.X;
            u16_screenY = st_touchScreenState.Y;
        
            // Call the routine which handles the key press
            KeypadHandleKeyPress(u16_screenX, u16_screenY);
        }
    
        // Drive the Moorse Code access functionality, scanning the
        // push button 10 times a second
        MoorseScanPushButton();
    }
    
    // Keep track of how many times this function has been called
    // so that we can attempt to see the random generator
    u32_randomSeeder++;
}

// ----------------------------------------------------------------------
// main()
//
// This is the main entry called by thembed operating system.
//
// This function will:
//      o Initialize this module
//      o Call the function that established the LCD screen and
//        draws the keypad
//      o Goes in to a forever loop that wakes up 10 times a second
//      o Calls a function which drives the main task which scans the
//        LCD touch screen for screen touches
//
// ----------------------------------------------------------------------
int main(void)
{
    // Perform local module initialization, if any
    MainInit();
    
    // Initialize the Keypad module
    KeypadInit();
    
    // Initialize the Moorse Code module
    MoorseInit();
    
    // Initialize the Animation module
    AnimationInit();
    
    // When we start up we will draw the touch screen keypad
    KeypadDrawKeypad();

    // Enter in to a forever loop which wakes up once a second
    while (true)
    {
        // Sleep for a tenth of a second
        wait(0.1);
   
        // Call the main process
        MainThread();
    }
}

// End of file

