
// ----------------------------------------------------------------------
// SecurityUnlockDemo-Moorse.cpp
//
// Fredric L. Rice, June 2019
//
// This module contains the code which performs the Moorse Code
// functionality. The push button is used to enter pulses in to the
// unlocking mechanism, and the LEDs are used to indicate when the
// pushbutton is down, and when a "dash" has been entered so that
// the user knows the dash was long enough to register as a dash.
//
// Once the operator stops entering dashes and dots, afterabout 3
// seconds the software will evaluate the entry to see if it matches
// the expected / desired characters and if they do, access is granted.
//
// ----------------------------------------------------------------------

#include "mbed.h"                       // The mbed operating system
#include "LCD_DISCO_F429ZI.h"           // For controlling the LCD
#include "TS_DISCO_F429ZI.h"            // For controlling the touch screen
#include "SecurityUnlockDemo-Main.h"    // Bring in the main module
#include "SecurityUnlockDemo-Moorse.h"  // Always include our own header

// ----------------------------------------------------------------------
// Describe data which is defined externally that we may access
//
// ----------------------------------------------------------------------

    // We may be accessing the LCD
    extern LCD_DISCO_F429ZI st_lcd;

    // We may be accessing the touch screen
    extern TS_DISCO_F429ZI st_touchScreen;

// ----------------------------------------------------------------------
// Define local data storage
//
// ----------------------------------------------------------------------

    // When the push button is used, we drive the lEDs
    static DigitalOut st_moorseLED(LED1);
    static DigitalOut st_moorseDash(LED2);

    // For the Moorse Code access, we describe the dots and dashes which
    // describe the letters C and Q. We do not care about testing for a
    // period of quiet beteen characters, we ignore silence, we only
    // test the length of when the key is held down
    static const uint8_t * ACCESS_MOORSE  = "-.-.--.-";

    // Instantiate a digitial input mapped as our push button
    static DigitalIn st_pushButton(PA_0);

    // To drive the Moorse Code access, we maintain counters which keep
    // trash of how long the push button has been detected to be held
    // down. Since the button is checked 10 times a second, the count
    // indicates about how many milliseconds the button was held down.
    // If it wraps, we don't care since that means the operator is holding
    // the button down for a very long time
    static uint16_t u16_buttonDownCount;
    
    // To determine whether the operator is finished with entering a
    // Moorse Code access code, we maintain a counter of "up" time, 
    // a.k.a. quiet time.
    static uint16_t u16_buttonUpCount;
    
    // When Moorse Code pulses are entered, we store the down time for
    // else pulse in this array
    static uint16_t au16_moorseCharacters[MAX_MOORSE_PULSES];
    
    // As Moorse Code pulses are entered with the push button, we keep
    // track of how many down presses there have been
    static uint8_t u8_moorsePulseCount;

// ----------------------------------------------------------------------
// MoorseInit()
//
// This function initializes the locally-held data. It also sets the
// push button to have no pull-up or pull-down resister
//
// ----------------------------------------------------------------------
void MoorseInit(void)
{
    // Initialize locall-held data in this module
    u16_buttonDownCount   = 0;
    u16_buttonUpCount     = 0;
    u8_moorsePulseCount   = 0;

    // Set the push button to not have an internal pull-up or down
    st_pushButton.mode(PullNone);
    
    // Ensure that both of the LEDs are turned OFF
    st_moorseLED = st_moorseDash = 0;
}

// ----------------------------------------------------------------------
// MoorseProcessButtonDownThenRelease()
//
// After the push button has been used to send a down pulse, this
// function is called to store the down time, if there is room to
// store it.
//
// ----------------------------------------------------------------------
static void MoorseProcessButtonDownThenRelease(void)
{
    // Make sure that we have a valid button down timer
    if (u16_buttonDownCount > 0)
    {
        // Is there room to store another pulse period?
        if (u8_moorsePulseCount < MAX_MOORSE_PULSES)
        {
            // There is, so store the down counter timer
            au16_moorseCharacters[u8_moorsePulseCount++] = u16_buttonDownCount;
        }
    }
}

// ----------------------------------------------------------------------
// MoorseProcessMoorseDigits()
//
// After the push button has been used to send Moore pulses, and after
// the button has not been pressed for 3 seconds, this function gets 
// called to process the pulses.
//
// We consider a value of 5 or higher to be 500 milliseconds down, or
// a "dash." Any down pulse shorter than that is considere to be a "dot."
//
// ----------------------------------------------------------------------
static void MoorseProcessMoorseDigits(void)
{
    uint8_t u8_testLoop          = 0;
    uint8_t au8_reportString[21] = { 0 };
    uint8_t au8_moorseString[21] = { 0 };
    
    // Since we have pulses, clear the display to get ready for a report
    st_lcd.Clear(LCD_COLOR_WHITE);

    for (u8_testLoop = 0; u8_testLoop < u8_moorsePulseCount; u8_testLoop++)
    {
        // Build a report of the times that we measured
        (void)sprintf((char *)au8_reportString, "Time: %u", 
            au16_moorseCharacters[u8_testLoop]);
        
        // Was the time down greater than 500 milliseconds?
        if (au16_moorseCharacters[u8_testLoop] > 5)
        {
            // It was, so we consider that to be a dash
            au8_moorseString[u8_testLoop] = '-';
        }
        else
        {
            // It was not so we consider that to be a dot
            au8_moorseString[u8_testLoop] = '.';
        }
        
        // Display the times that we measured
        st_lcd.DisplayStringAt(1, LINE(u8_testLoop + 1), au8_reportString, LEFT_MODE);
    }
    
    // Make sure that the string of Moorse Code pulses is NULL terminated
    au8_moorseString[u8_testLoop] = 0x00;
    
    // Build a report showing what the Moorse Code looks like    
    (void)sprintf((char *)au8_reportString, "Moorse code: %s", au8_moorseString);
    
    // Display what the Moorse Code looked like
    st_lcd.DisplayStringAt(1, LINE(u8_testLoop + 1), au8_moorseString, LEFT_MODE);

    // Is that enough pulses to satisfy all required Moorse pulses?
    if (u8_moorsePulseCount == strlen((char *)ACCESS_MOORSE))
    {
        // It was, are the pulses what we expect for Level 1 access?
        if (! memcmp(au8_moorseString, ACCESS_MOORSE, u8_moorsePulseCount))
        {
            st_lcd.DisplayStringAt(1, LINE(u8_testLoop + 2), 
                (uint8_t *)"Access granted", LEFT_MODE);
            
            // Leave everything up on the screen for 5 seconds
            wait(5.0);
            
            // Grant access level 1
            MainGrantAccess(1);            
        }
    }
}

// ----------------------------------------------------------------------
// MoorseScanPushButton()
//
// This function will check to see what the state of the push button
// is, and it will drive the Moorse Code entry access functionality.
//
// A push button held down for 500 milliseconds or longer is considered
// to be a "dash." And down time shorter than that is considered to be
// a "dot." After 3 seconds of quiet time after a down time has been
// detected, we consider the operator to have finished pulsing in all
// Moorse Code pulses.
//
// ----------------------------------------------------------------------
void MoorseScanPushButton(void)
{
    // Acquire / poll the state of the push button
    int i_buttonState = st_moorseLED = st_pushButton.read();
    
    // Is the button not being pressed?
    if (0 == i_buttonState)
    {
        // The button is not down, was the button down before?
        if (u16_buttonDownCount > 0)
        {
            // The push button was down and now it has been released.           
            // Process the button down time
            MoorseProcessButtonDownThenRelease();
        }     
   
        // Add another count to the time that the button has been up
        u16_buttonUpCount++;
        
        // Since the button is not being pressed, make sure that
        // the Dash LED is turned OFF
        st_moorseDash = 0;
        
        // Has the button been up for at least 3 seconds? Since
        // we are called 10 times a second, we check for 3 counts.
        // We only check if there has been at least 1 down pulse.
        if (u8_moorsePulseCount > 0 && u16_buttonUpCount >= 30)
        {
            // The operator appears to be finished with entering 
            // the Moorse digits so process what may have come in
            MoorseProcessMoorseDigits();
                
            // Discard all Moorse down pulses we may have accumulated
            u8_moorsePulseCount = 0;
        }
            
        // Discard the button down counter
        u16_buttonDownCount = 0;
     }
    else
    {
        // The button is down so count how long it is down
        u16_buttonDownCount++;
        
        // Discard any button on time
        u16_buttonUpCount = 0;
        
        // Is the button being held down long enough to indicate a dash?
        if (u16_buttonDownCount >= 5)
        {
            // It is, so indicate the fact by illuminating the Dash LED
            st_moorseDash = 1;
        }
    }
}

// End of file

