/**
  ******************************************************************************
  * @file    adxl355.h
  * @author  DUR
  * @version V1.0.0
  * @date    01-Aug-2017
  * @brief   Header file of a library for adxl355 interface.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2014 TD Group</center></h2>
  *
  * The information contained herein is property of TD Group S.p.A.
  *
  * Licensees are granted free, non-transferable use of the information. NO
  * WARRANTY of ANY KIND is provided. This heading must NOT be removed from
  * the file.
  * 
  ******************************************************************************
  */ 
  

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __ADXL355_H
#define __ADXL355_H


/* Includes ------------------------------------------------------------------*/
#include <stdint.h>
#include <string.h>



/* Exported define -----------------------------------------------------------*/
  
/** @brief   ID dell'accelerometro
  */
#define ADXL355_DEVICE_ID                0xED


/** @brief   Codice per effettuare il reset sw dell'accelerometro
  */
#define ADXL355_RESET_CODE               0x52
  

/** @brief   Start/stop self-test
  */
#define ADXL355_SELF_TEST_ENABLE         0x01
#define ADXL355_SELF_TEST_DISABLE        0x00  
    
    
/** @brief   Indirizzi registri accelerometro
  */
#define ADXL355_DEVID_AD                 0x00
#define ADXL355_DEVID_MST                0x01
#define ADXL355_PARTID                   0x02
#define ADXL355_REVID                    0x03
#define ADXL355_STATUS                   0x04
#define ADXL355_FIFO_ENTRIES             0x05
#define ADXL355_TEMP2                    0x06
#define ADXL355_TEMP1                    0x07
#define ADXL355_XDATA3                   0x08
#define ADXL355_XDATA2                   0x09
#define ADXL355_XDATA1                   0x0A
#define ADXL355_YDATA3                   0x0B
#define ADXL355_YDATA2                   0x0C
#define ADXL355_YDATA1                   0x0D
#define ADXL355_ZDATA3                   0x0E
#define ADXL355_ZDATA2                   0x0F
#define ADXL355_ZDATA1                   0x10
#define ADXL355_FIFO_DATA                0x11
#define ADXL355_OFFSET_X_H               0x1E
#define ADXL355_OFFSET_X_L               0x1F
#define ADXL355_OFFSET_Y_H               0x20
#define ADXL355_OFFSET_Y_L               0x21
#define ADXL355_OFFSET_Z_H               0x22
#define ADXL355_OFFSET_Z_L               0x23
#define ADXL355_ACT_EN                   0x24
#define ADXL355_ACT_THRESH_H             0x25
#define ADXL355_ACT_THRESH_L             0x26
#define ADXL355_ACT_COUNT                0x27
#define ADXL355_FILTER                   0x28
#define ADXL355_FIFO_SAMPLES             0x29
#define ADXL355_INT_MAP                  0x2A
#define ADXL355_SYNC                     0x2B
#define ADXL355_RANGE                    0x2C
#define ADXL355_POWER_CTL                0x2D
#define ADXL355_SELF_TEST                0x2E
#define ADXL355_RESET                    0x2F


/** @brief   Maschere di bit 
  */
#define ADXL355_READ_BYTE_MASK           0x01
#define ADXL355_WRITE_BYTE_MASK          0xFE  

#define ADXL355_20_BIT_MASK              0x000FFFFF
#define ADXL355_12_BIT_MASK              0x0FFF

#define ADXL355_RANGE_MASK               0x03    
#define ADXL355_ODR_MASK                 0x0F
#define ADXL355_INT_POL_MASK             0x40
#define ADXL355_INT_1_MASK               0x0F
#define ADXL355_INT_2_MASK               0xF0   
#define ADXL355_MEAS_MASK                0x03  
#define ADXL355_DRDY_OFF_MASK            0x04 
#define ADXL355_SELF_TEST_MASK           0x03 


/** @brief   Temperature parameters 
  */
#define ADXL355_TEMP_BIAS       (float)1852.0      /* Accelerometer temperature bias(in ADC codes) at 25 Deg C */
#define ADXL355_TEMP_SLOPE      (float)-9.05       /* Accelerometer temperature change from datasheet (LSB/degC) */



/* Exported typedef ----------------------------------------------------------*/
/** @brief   Enumerativo per le misure attive sul sensore in un determinato momento.
  */ 
typedef enum
{
    adxl355_acc = 0u,    //Sola misura di accelerazione attiva
    adxl355_both,        //Entrambe le misure attive (accelerazione e temperatura)
    adxl355_none         //Nessuna misura attiva
}adxl355_measure_enabled;  
 
 
/** @brief   Enumerativo per valori di range di misura ammissibili.
  */ 
typedef enum
{
    adxl355_2g = 1u,
    adxl355_4g,
    adxl355_8g
}adxl355_range_value; 


/** @brief   Enumerativo per valori di output data rate ammissibili.
  */
typedef enum
{
    adxl355_4000hz = 0u,
    adxl355_2000hz,
    adxl355_1000hz,
    adxl355_500hz,
    adxl355_250hz,
    adxl355_125hz,
    adxl355_62p5hz,
    adxl355_31p25hz,
    adxl355_15p625hz,
    adxl355_7p813hz,
    adxl355_3p906hz
}adxl355_output_datarate_value;



/** @brief   Enumerativo per i differenti eventi associabili ai 2 interrupt ammissibili.
  */ 
typedef enum
{
    adxl355_rdy_en = 0u, //DATA_RDY interrupt enable on INT1/2
    adxl355_full_en,     //FIFO_FULL interrupt enable on INT1/2
    adxl355_ovr_en,      //FIFO_OVR interrupt enable on INT1/2
    adxl355_act_en       //Activity interrupt enable on INT1/2
}adxl355_int_event; 




/** @brief   Struttura che setta i parametri degli interrupt sui 2 pin a disposizione.
  */ 
typedef struct adxl355_gpio_config_t
{
    adxl355_int_event   int1_evt;    //evento che scatena l'interrupt 1
    adxl355_int_event   int2_evt;    //evento che scatena l'interrupt 2
    bool                int_act_low; // se true gli interrupt sono attivi basso, altrimenti attivi alto
}adxl355_int_config;



/** @brief   Struttura che definisce i parametri di configurazione ed utilizzo 
  *          dell'accelerometro
  */ 
typedef struct adxl355_handler_t
{ 
    /* Range di misura */
    adxl355_range_value               measure_range;
    /* Data rate di acquisizione */
    adxl355_output_datarate_value     out_data_rate; 
    /* Pin di controllo dell'accelerometro - porre a NULL se non utilizzati */
    uint32_t                     drdy_pin;  //output
    uint32_t                     int1_pin;  //output
    uint32_t                     int2_pin;  //output
    /* Configurazione interrupt */
    adxl355_int_config           int_config;
    /* Valori grezzi della accelerazione */
    uint32_t                     raw_acc_x_value;   
    uint32_t                     raw_acc_y_value;   
    uint32_t                     raw_acc_z_value;
    /* Valore grezzo della temperatura */
    uint16_t                     raw_temp_value;        
}adxl355_handler;





/* Exported types ------------------------------------------------------------*/ 

/* Exported constants --------------------------------------------------------*/

/* Exported macro ------------------------------------------------------------*/


/* Exported function prototypes ----------------------------------------------*/
/**< Funzione che restituisce i parametri identificativi del driver */
void adxl355_driver_info(uint8_t * adxl355_driver_version, uint8_t * adxl355_driver_revision, uint8_t * adxl355_driver_patch);
/**< Funzione che restituisce l'ID dell'accelerometro */
uint16_t adxl355_who_am_i(uint8_t * adxl355_device_id);
/**< Funzione di inizializzazione dell'accelerometro */
uint16_t adxl355_init(adxl355_handler *adxl_init_handler);
/**< Funzione di start/stop delle acquisizioni */
uint16_t adxl355_start_acquisition(adxl355_measure_enabled adxl_start);
/**< Funzione per la lettura del/i dato/i acquisito/i */
uint16_t adxl355_get_data(adxl355_handler *adxl_data_handler);
/**< Funzione che restituisce il valore grezzo di accelerazione lungo l'asse X */
uint32_t adxl355_raw_x_acc(adxl355_handler *adxl_xdata_handler);
/**< Funzione che restituisce il valore grezzo di accelerazione lungo l'asse Y */
uint32_t adxl355_raw_y_acc(adxl355_handler *adxl_ydata_handler);
/**< Funzione che restituisce il valore grezzo di accelerazione lungo l'asse Z */
uint32_t adxl355_raw_z_acc(adxl355_handler *adxl_zdata_handler);
/**< Funzione che restituisce il valore grezzo di temperatura */
uint16_t adxl355_raw_temp(adxl355_handler *adxl_tdata_handler);
/**< Funzione per il reset sw */
uint16_t adxl355_sw_reset(void);
/**< Funzione per il self-test */
uint16_t adxl355_self_test(bool adxl_start);


/* Funzione di trasferimento su bus SPI */
extern uint16_t adxl355_spi_transfer(uint8_t * rbuffer, uint32_t rlen);

#endif 

/************************ (C) COPYRIGHT TD Group *****END OF FILE****/


