#include "mbed.h"
#include "sample_hardware.hpp"
#include "string.h"
#include <stdio.h>
#include <ctype.h>

#define SWITCH1_RELEASE 1

void thread1();
void thread2();
void switchISR();


//Threads
Thread samplingThread(osPriorityRealtime);
Thread t1;

//Class type
class message_t {
public:    
    float adcValue;    
    int sw1State;
    int sw2State;
    
    //Constructor
    message_t(float f, int s1, int s2) {
        adcValue = f;
        sw1State = s1;
        sw2State = s2;    
    }
};
 
//Memory Pool - with capacity for 16 message_t types
//MemoryPool<message_t, 16> mpool;

//Message queue - matched to the memory pool
//Queue<message_t, 16> queue;

//Mail queue
Mail<message_t, 16> mail_box;


// Call this on precise intervals
void adcISR() {
    //This is going to be brief
    samplingThread.signal_set(1);    
}
void doSample() {
    
    while (true) {
        //Waiting for ISR in WAITING state
        Thread::signal_wait(1);
                
        //Read sample - make a copy
        float sample = adcIn;
        //Grab switch state
        uint32_t switch1State = SW1;
        uint32_t switch2State = SW2;
        
        //Allocate a block from the memory pool
        message_t *message = mail_box.alloc();
        if (message == NULL) {
            //Out of memory
            printf("Out of memory\n\r");
            redLED = 1;
            return;   
        }
        
        //Fill in the data
        message->adcValue = sample;
        message->sw1State = switch1State;
        message->sw2State = switch2State;
        
        //Write to queue
        osStatus stat = mail_box.put(message);    //Note we are sending the "pointer"
        
        //Check if succesful
        if (stat == osErrorResource) {
            redLED = 1; 
            printf("queue->put() Error code: %4Xh, Resource not available\r\n", stat);   
            mail_box.free(message);
            return;
        }
        
    } //end whie    
}

//Normal priority thread (consumer)
void thread1() 
{      
    static int counter = 0;
    
    while (true) {
        //Block on the queue
        osEvent evt = mail_box.get();
        
        //Check status
        if (evt.status == osEventMail) {
            message_t *pMessage = (message_t*)evt.value.p;  //This is the pointer (address)
            //Make a copy
            message_t msg(pMessage->adcValue, pMessage->sw1State, pMessage->sw2State);
            //We are done with this, so give back the memory to the pool
            mail_box.free(pMessage);
            
            //Echo to the terminal
            printf("ADC Value: %.2f\t",    msg.adcValue);
            printf("SW1: %u\t",             msg.sw1State);
            printf("SW2: %u\n\r",             msg.sw2State);
            
            //Toggle green
            if ( (msg.sw1State == 1) && (msg.sw2State == 1)) {
                counter++;
                if (counter == 10) {
                    greenLED = !greenLED;
                    counter = 0;
                }   
            } else {
                counter = 0;   
            }            
            
        } else {
            printf("ERROR: %x\n\r", evt.status);   
        }  
                
    } //end while
}


// Main thread
int main() {
    //Power on self-test
    post();
           
    //Start message
    printf("Welcome\n");           
   
    //Hook up timer interrupt   
    Ticker timer; 
    timer.attach(&adcISR, 0.1);
               
    //Threads
    samplingThread.start(doSample); //High Priority
    t1.start(thread1); 
    
    printf("Main Thread\n");
    while (true) {
        Thread::wait(5000);
        puts("Main Thread Alive");
    }
}

   
