#include "mbed.h"
#include "Motor.h"
#include "QEI.h"

// Definitions
#define PI           (3.1415926)
#define LOG_FREQ     (100)      // (Hz) - data logging frequency
#define NUM_DATA_PTS (500)      // (#)  - number of data points at above LOG_FREQ;

// Classes and objects
QEI        enc(p17,p16,NC,1600); // Need to know PPR count of selected encoder
Serial     pc(USBTX, USBRX);
Motor m(p22,p21,p23); // pin 18 will not be used
Timer      t;
Ticker     ticker_timer;
DigitalOut mbed_LED(LED1);

// Create local file system under the name "mBed_drive"
LocalFileSystem mBed_drive("local");

// Global declarations
int   num_data_pts_rec, i;
char  c;
float theta;
float lv_time[NUM_DATA_PTS];
float lv_theta[NUM_DATA_PTS];


// timer_function
void timer_function()
{

//generate random data
//float min = 0.1 / 3.3;  //Define 0.1 V as ~03% from 3.3 V
//float max = 3.2 / 3.3;  //Define 3.2 V as ~97% from 3.3 V

//measure signal
//signal = min + ((float)rand()/RAND_MAX)*(max - min)*3.3;  // Generate random voltage output from 0.1 to 3.2
//printf("Voltage= %1.2f \n", signal);     // Debug

// Measure theta
    theta = (long)enc.getPulses()/(1600.0*2.0)*2.0*PI;  // The QEI uses 2X interpolation by default.
    if(num_data_pts_rec<NUM_DATA_PTS) {
        lv_time[num_data_pts_rec]  = t.read();
        lv_theta[num_data_pts_rec] = theta;
        num_data_pts_rec++;
    }
}

// main function
int main()
{
    // Serial port initializations
    pc.baud(9600);

    // Initialize encoder
    enc.reset();

    // Number of data points recorded
    num_data_pts_rec = 0;

    // Clear Terra Term screen
    for(i=1; i<80; i++)
        pc.printf("\n\r");

    // Prompt user to start recording data.
    pc.printf("Running datalog.cpp.\r\n");
    pc.printf("____________________________________________________________r\n");
    pc.printf("\r\nPress any key to start.\r\n");

    // Wait until user presses a key
    while(!pc.readable());
    c = pc.getc();
    c = '\0';

    // Start timer and tickers
    t.start();
    ticker_timer.attach(&timer_function,1.0/LOG_FREQ);

    // Measurement while loop
    while(num_data_pts_rec < NUM_DATA_PTS) {
        // Print data to screen
        pc.printf("%.2f %.2f; \n\r",t.read(),theta);
    }

    // Turn off ticker functions
    ticker_timer.detach();

    // File initialization
    pc.printf("Opening mBed_output.m                ... ");
    FILE *fp = fopen("/local/mBedOut.m", "w");
    pc.printf("done.\n");

    // Print measured data to local file.
    pc.printf("Printing measured data to local file ... ");
    fprintf(fp,"lv.time = [  \n");
    for(i=0; i<num_data_pts_rec; i++) {
        //fprintf(fp,"lv.time(%d,1) = [ %.5f;\n",i+1,lv_time[i]);
        fprintf(fp, "%.5f; \n",lv_time[i]);
    }
    fprintf(fp,"]; \n");
    fprintf(fp,"lv.theta = [  \n");
    for(i=0; i<num_data_pts_rec; i++) {
        fprintf(fp,"%.5f;\n",lv_theta[i]);
    }
    fprintf(fp,"]; \n");
    fprintf(fp,"plot(lv.time,lv.theta); \n");
    fprintf(fp,"xlabel('time (s)'); \n");
    fprintf(fp,"ylabel('radians'); \n");

    //fprintf(fp,"lv.units = '(volts)';");
    pc.printf("done.\n\r");

    // Close file
    fclose(fp);

    // Finished
    pc.printf("____________________________________________________________\r\n\r\n");
    pc.printf("\n\rProgram exited normally. End of line.\n\r");

}

