/*
 * CharValue.cpp
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "CharValue.h"
#include "NullValue.h"
#include <stdlib.h>
#include <string.h>
#include <ctype.h>


/*-------------------------------------------------------------------------*/
CharValue::CharValue(const char *str, bool copy)
{
	if ((str == NULL) || (*str == '\0'))
	{
		_alloc = false;
		_str = NULL;
	}
	else
	{
		_alloc = copy;
		if (copy)
		{
			_str = (const char*)malloc(strlen(str)*sizeof(char));
			if (_str == NULL)
				return;
			strcpy((char*)_str, str);
			_escape = escapeCheck();
		}
		else
		{
			_str = str;
			_escape = escapeCheck();
		}
	}
}
/*-------------------------------------------------------------------------*/
CharValue::~CharValue()
{
	if (_alloc)
		free((void*)_str);
}
/*-------------------------------------------------------------------------*/
uint8_t CharValue::valueType() const
{
	if (_str == NULL)
		return VALUE_NULL;
	return VALUE_CHARACTER;
}
/*-------------------------------------------------------------------------*/
long CharValue::integerValue() const
{
	return 0;
}
/*-------------------------------------------------------------------------*/
double CharValue::floatValue() const
{
	return 0.0;
}
/*-------------------------------------------------------------------------*/
const char * CharValue::characterValue() const
{
	return _str;
}
/*-------------------------------------------------------------------------*/
size_t CharValue::write(AbstractDataSink& sink) const
{
	size_t n;
	const char *ptr; char c;

	if (_str == NULL)
		return 0;

	n = 0;
	if (_escape)
		n += sink.write('"');

	for (ptr = _str; (c = *ptr) != '\0'; ptr++)
	{
		if ((_escape) && (c == '"'))
			n += sink.write('"');
		n += sink.write(c);
	}

	if (_escape)
		n += sink.write('"');
	return n;
}
/*-------------------------------------------------------------------------*/
size_t CharValue::length() const
{
	size_t n;
	const char *ptr; char c;

	if (_str == NULL)
		return 0;

	n = 0;
	if (_escape)
		n += 2;

	for (ptr = _str; (c = *ptr) != '\0'; ptr++)
	{
		if ((_escape) && (c == '"'))
			n++;
		n++;
	}

	return n;
}
/*-------------------------------------------------------------------------*/
Value* CharValue::copy() const
{
	if (_str == NULL)
		return new NullValue();
	return new CharValue(_str, true);
}
/*-------------------------------------------------------------------------*/
bool CharValue::escapeCheck()
{
	const char *ptr; char c;

	ptr = _str;
	while (*ptr != '\0')
	{
		c = *ptr;
		if ((isspace(c)) && ((ptr == _str) || (c != ' ')))
			return true;
		else if ((c == '"') || (c == ','))
			return true;
		ptr++;
	}

	if (isspace(c))
		return true;

	return false;
}
/*-------------------------------------------------------------------------*/
