#ifndef ABSTRACTSMARTREST_H
#define ABSTRACTSMARTREST_H

#include "config.h"
#include <stddef.h>
#include <stdint.h>
#include "DataGenerator.h"
#include "ParsedRecord.h"
#ifdef SMARTREST_TRANSACTIONAL
#include "Aggregator.h"
#endif

/** Return value indicating that no error occurred. */
#define SMARTREST_SUCCESS 0
/** Return value indicating that the connection has been closed during
 * data transmission. */
#define SMARTREST_CONNECTION_FAILED 1
/** Return value indicating an internal state error. */
#define SMARTREST_INTERNAL_ERROR 2
/** Return value indicating a transmission timeout. */
#define SMARTREST_TIMEOUT_ERROR 3
/** Return value indicating an end of response indicated by the
 * Content-Length header. */
#define SMARTREST_END_OF_RESPONSE 4
/** Return value indicating that the connection has been closed. */
#define SMARTREST_CONNECTION_CLOSED 5

/**
 * Abstract SmartRest facade class.
 * This class provides methods to send a request and receive a response
 * from the server.
 * 
 * Example:
 * @code
 * // given a concrete SmartRest implementation and a template
 * SmartRest client;
 * StaticData template;
 * 
 * // bootstrap
 * if (client.bootstrap(template) != SMARTREST_SUCCESS) {
 *     // error handling
 *     return;
 * }
 * 
 * if (client.send(StaticData("100,Hello")) != SMARTREST_SUCCESS) {
 *     // error handling
 * }
 * 
 * uint8_t ret;
 * while ((ret = client.receive(record)) == SMARTREST_SUCCESS) {
 *     // work with data
 * }
 * 
 * // error handling
 * 
 * // call after every request.
 * client.stop();
 * @encode
 */
class AbstractSmartRest
{
	public:
		virtual ~AbstractSmartRest() { };

		/**
		 * Sets the client authorization.
		 * @param username the authorization username
		 * @param password the authorization password
		 */
		virtual uint8_t setAuthorization(const char*, const char*) = 0;

#ifdef SMARTREST_TRANSACTIONAL
		/**
		 * Performs a SmartRest request without receiving any data from the
		 * server.
		 * @param generator the generator which will generate the data to be
		 *                  sent as a template.
		 * @param overrideIdentifier a device identifier which gets sent instead
		 *                           of the identifier specified in the
		 *                           constructor. If an empty string is
		 *                           specified, no identifier is sent at all.
		 * @return a non-zero value if and only if an error occured
		 */
		virtual uint8_t request(const DataGenerator&, const char* = NULL) = 0;

		/**
		 * Performs a SmartRest request.
		 * @param generator the generator which will generate the data to be
		 *                  sent as a template.
		 * @param aggregator the aggregator where all received records will
		 *                   be stored. The aggregator must be managed,
		 *                   otherwise an error will be returned.
		 * @param overrideIdentifier a device identifier which gets sent instead
		 *                           of the identifier specified in the
		 *                           constructor. If an empty string is
		 *                           specified, no identifier is sent at all.
		 * @return a non-zero value if and only if an error occured
		 */
		virtual uint8_t request(const DataGenerator&, Aggregator&, const char* = NULL) = 0;
#endif

		/*
		 * Initiates the SmartRest bootstrap process.
		 * When successful, the template identifier will be replaced by the
		 * global managed object ID in future requests.
		 * @param generator the generator which will generate the data to be
		 *                  sent as a template.
		 * @return a non-zero value if and only if an error occured
		 */
		virtual uint8_t bootstrap(const DataGenerator&) = 0;

		/**
		 * Sends a smart request.
		 * @param generator the generator which will generate the data to be
		 *                  sent.
		 * @param overrideIdentifier a device identifier which gets sent instead
		 *                           of the identifier specified in the
		 *                           constructor. If an empty string is
		 *                           specified, no identifier is sent at all.
		 * @return a non-zero value if and only if an error occured
		 */
		virtual uint8_t send(const DataGenerator&, const char* = NULL) = 0;

		/**
		 * Starts a SmartRest stream.
         * @param uri the stream endpoint URI
		 * @param record the record which will be sent to the endpoint
		 * @param overrideIdentifier a device identifier which gets sent instead
		 *                           of the identifier specified in the
		 *                           constructor. If an empty string is
		 *                           specified, no identifier is sent at all.
		 * @return a non-zero value if and only if an error occured
		 */
		virtual uint8_t stream(const char*, const Record&, const char* = NULL) = 0;

		/**
		 * Tries to receive a parsed response row.
		 * When the function succeeds, but the row pointer is NULL, there are
		 * no more rows to be read.
		 * @param record an instance to where the parsed row is written
		 * @return a non-zero value if and only if an error occured
		 */
		virtual uint8_t receive(ParsedRecord&) = 0;

		/*
		 * Closes the connection.
		 */
		virtual void stop() = 0;

		/*
		 * Retrieves the template identifier.
		 * @return template identifier
		 */
		virtual const char * getIdentifier() = 0;
};

#endif
