/*
 * ParsedRecord.cpp
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "ParsedRecord.h"
#include "ComposedRecord.h"
#include "ParsedValue.h"
#include <stdlib.h>
#include <string.h>


/*-------------------------------------------------------------------------*/
ParsedRecord::ParsedRecord(bool copy) :
	_copy(copy)
{
	_count = 0;
#ifdef SMARTREST_PARSED_DYNAMIC_ALLOC
	_values = NULL;
#endif
	_buffer = NULL;
}
/*-------------------------------------------------------------------------*/
ParsedRecord::~ParsedRecord()
{
	clear();

#ifdef SMARTREST_PARSED_DYNAMIC_ALLOC
	if (_values != NULL)
		free(_values);
#endif
}
/*-------------------------------------------------------------------------*/
size_t ParsedRecord::values() const
{
	return _count;
}
/*-------------------------------------------------------------------------*/
const Value& ParsedRecord::value(size_t index) const
{
	return *_values[index];
}
/*-------------------------------------------------------------------------*/
ParsedRecord* ParsedRecord::copy() const
{
	ParsedRecord *copy = new ParsedRecord(true);
	copy->set(_buffer, _count);
	return copy;
}
/*-------------------------------------------------------------------------*/
const char * ParsedRecord::rawValue(size_t index) const
{
	if (index >= _count)
		return NULL;

	const char *ptr = _buffer;
	for (size_t n = 0; n < index; n++)
	{
		ptr += strlen(ptr) + 1;
	}

	return ptr;
}
/*-------------------------------------------------------------------------*/
bool ParsedRecord::set(const char *buffer, size_t count)
{
	clear();

#ifdef SMARTREST_PARSED_DYNAMIC_ALLOC
	const Value **values = (const Value**)realloc(_values, count*sizeof(Value*));
	if (values == NULL)
		return false;
	_values = values;
#else
	if (count > SMARTREST_PARSED_FIXED_SIZE)
		count = SMARTREST_PARSED_FIXED_SIZE;
#endif

	if (_copy)
	{
		const char *ptr = buffer;
		for (size_t n = 0; n < count; n++)
			ptr += strlen(ptr) + 1;
		_buffer = (const char*)malloc(ptr-buffer);
		if (_buffer == NULL)
			return false;
		memcpy((char*)_buffer, buffer, ptr-buffer);
	}
	else
	{
		_buffer = buffer;
	}
	_count = count;

	const char *ptr = _buffer;
	for (size_t n = 0; n < count; n++)
	{
		_values[n] = new ParsedValue(ptr, false);
		ptr += strlen(ptr) + 1;
	}

	return true;
}
/*-------------------------------------------------------------------------*/
void ParsedRecord::clear()
{
	for (size_t n = 0; n < _count; n++)
		delete _values[n];
	if ((_copy) && (_buffer != NULL))
		free((char*)_buffer);
	_buffer = NULL;
	_count = 0;
}
/*-------------------------------------------------------------------------*/
