/*
 * ParsedRecord.h
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef PARSEDRECORD_H
#define PARSEDRECORD_H

#include "config.h"
#include "DataGenerator.h"
#include "Record.h"
#include "Value.h"

#ifndef SMARTREST_PARSED_DYNAMIC_ALLOC
#ifndef SMARTREST_PARSED_FIXED_SIZE
#define SMARTREST_PARSED_FIXED_SIZE 24
#endif
#endif

class Parser;

/**
 * This class provides accessors for values in a parsed record.
 * 
 * Example:
 * @code
 * // given a concrete SmartRest client instance
 * SmartRest client;

 * ParsedRecord record;
 * if (client.receive(record) == SMARTREST_SUCCESS) {
 *     for (size_ i = 0; i < record.values(); i++) {
 *         const Value& val = record.value(i);
 *         switch (val.valueType()) {
 *         case VALUE_NULL:
 *             printf("Null value.\n");
 *             break;
 *         case VALUE_CHARACTER:
 *             printf("Character value: %s\n", val.characterValue());
 *             break;
 *         case VALUE_INTEGER:
 *             printf("Integer value: %d\n", val.integerValue());
 *             break;
 *         case VALUE_FLOAT:
 *             printf("Float value: %lf\n", val.floatValue());
 *             break;
 *         }
 *     }
 * }
 * @encode
 */
class ParsedRecord : public Record
{
	public:
		ParsedRecord(bool = false);
		~ParsedRecord();

		size_t values() const;
		const Value& value(size_t) const;

		ParsedRecord* copy() const;

		/**
		 * Returns the raw value string by index
		 * @param index the index of the value
		 * @return a pointer to the raw value
		 */
		const char * rawValue(size_t) const;

	protected:
		bool set(const char*, size_t);
		void clear();

	private:
#ifdef SMARTREST_PARSED_DYNAMIC_ALLOC
		const Value **_values;
#else
		const Value *_values[SMARTREST_PARSED_FIXED_SIZE];
#endif
		const char *_buffer;
		size_t _count;
		bool _copy;

		friend class Parser;
};


#endif
