/*
 * Aggregator.h
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef AGGREGATOR_H
#define AGGREGATOR_H

#include "config.h"
#include <stddef.h>
#include "DataGenerator.h"
#include "Record.h"

#ifndef SMARTREST_AGGREGATOR_INITIAL_CAPACITY
#define SMARTREST_AGGREGATOR_INITIAL_CAPACITY 50
#endif
#ifndef SMARTREST_AGGREGATOR_MEMORY_INCREMENT
#define SMARTREST_AGGREGATOR_MEMORY_INCREMENT 25
#endif
//#define SMARTREST_AGGREGATOR_FIXED_SIZE 100

/**
 * An aggregator of records.
 * 
 * Example:
 * @code
 * // Given: A concrete SmartRest implementation.
 * SmartRest client;
 * 
 * // sets up a growing aggregator which copies all added objects
 * // to the heap
 * Aggregator buffer(25, true, true);
 * 
 * // random data collection
 * for (uint8_t i = 0; i < 32; ++i) {
 *     // Note: ComposedRecord is not copying any values into the heap
 *     // because the availability of its values in guaranteed during the
 *     // instance's lifetime.
 *     ComposedRecord record(false);
 *     
 *     CharValue msgId(100);
 *     IntegerValue number(i);
 *     record.add(msgId).add(number);
 *     
 *     // Note: ComposedRecord is now cloned into the heap to extend its
 *     // lifetime beyond this code block.
 *     buffer.add(record);
 * }
 * 
 * // send the bulk request
 * uint8_t ret = client.send(buffer);
 * if (ret != SMARTREST_SUCCESS) {
 *     // error handling
 * }
 * @encode
 */
class Aggregator : public DataGenerator
{
	public:
#ifndef SMARTREST_AGGREGATOR_FIXED_SIZE
		/**
		 * Creates a new Aggregator instance.
		 * @param capacity the initial capacity of the instance
		 * @param growing specifies the capability of this instance to grow
		 * @param managed specifies whether internal memory management shall be
		 *                used. If true, all added records will be copied to
		 *                the heap and freed accordingly.
		 */
		Aggregator(size_t = SMARTREST_AGGREGATOR_INITIAL_CAPACITY, bool = true, bool = false);
#else
		/**
		 * Creates a new Aggregator instance.
		 * @param managed specifies whether internal memory management shall be
		 *                used. If true, all added records will be copied to
		 *                the heap and freed accordingly.
		 */
		Aggregator(bool = false);
#endif
		~Aggregator();

		/**
		 * Adds a record to the aggregator.
		 * @param record the record to add
		 * @return true if added, false otherwise.
		 */
		bool add(const Record&);

		/**
		 * Retrieves the nth record from the aggregator.
		 * @param index the index of the nth element to retrieve
		 */
		const Record& get(size_t) const;

		/**
		 * Clears the aggregator. The capacity will shrink to it's initial
		 * size.
		 */
		void clear();

		/**
		 * Returns the number of records aggregated by this instance.
		 * @return the number of records aggregated
		 */
		size_t length() const;

		/**
		 * Returns whether the aggregator is full. If growing, this will
		 * always return false.
		 * @return whether the aggregator is full
		 */
		bool full() const;

		/**
		 * Returns the capacity of the aggregator. This will always return zero
		 * if the aggregator is growing.
		 */
		size_t capacity() const;

		/**
		 * Returns whether this aggregator is using internal memory management.
		 * @return whether internal memory management is being used
		 */
		bool managed() const;

		size_t writeTo(AbstractDataSink&) const;
		size_t writtenLength() const;
		Aggregator* copy() const;

	private:
#ifdef SMARTREST_AGGREGATOR_FIXED_SIZE
		const Record *_list[SMARTREST_AGGREGATOR_FIXED_SIZE];
#else
		const Record **_list;
#endif
		size_t _length;
#ifndef SMARTREST_AGGREGATOR_FIXED_SIZE
		size_t _capacity, _initial;
		bool _growing;
#endif
		bool _managed;
};

#endif
