/*
* 	FastECompass Library
*   Autor: Federico Pinna
*   Date: 29 de may.de 2016
*/
#include "mbed.h"
#include "arm_math.h"
#include "magnetic.h"
#include "MotionSensor.h"
#include "FastAtan2.h"


#ifndef FASTECOMPASS_FASTECOMPASS_H_
#define FASTECOMPASS_FASTECOMPASS_H_

/** FastECompass class.
*
* Example:
* @code
* #include "mbed.h"
* #include "MAG3110.h"
* #include "MMA8451Q.h"
* #include "FastECompass.h" 
*
* #define MMA8451_I2C_ADDRESS (0x1d<<1)
*
* MAG3110  mag( PTE25, PTE24);
* MMA8451Q acc( PTE25, PTE24, MMA8451_I2C_ADDRESS);
* DigitalOut lred(LED_RED,1);
*
* FastECompass ec(&mag,&acc);
* Serial pc(USBTX,USBRX);
*
* int main(){
*    
*   ec.enableSensors();
*   ec.calibrateSensors();
*    
*   lred=0;
*   
*   while(true){
*     ec.updateSensors();
*     pc.printf("Pitch: %d  Roll: %d Yaw: %d\n",ec.getPitch(),ec.getRoll(),ec.getYaw()); 
*     wait(0.5);
*   }    
*
* }
* @endcode
**/


class FastECompass{

	MotionSensor *magsensor;
	MotionSensor *accsensor;
	MotionSensorDataCounts mag_raw;
	MotionSensorDataCounts acc_raw;
	MagCalibration magcal;
	i16MagCalibration i16magcal;
	MagneticBuffer magbuf;
	int32_t pitch;
	int32_t roll;
	int32_t yaw;

	public:
  	/** Create and Initialize FastEComppas instance
   	 *@param magsensor Pointer to object Magnetometer sensor derived from the MotionSensor class
   	 *@param accsensor Pointer to object Accelerometer sensor derived from the MotionSensor class
   	 */ 
	FastECompass(MotionSensor *magsensor, MotionSensor *accsensor);
	
	/**Calibrates hard-iron magnetometer sensor.
	*  Takes a sample every 40 milliseconds.
	*/
	void calibrateSensors();
	/**
	*	Enables magnetometer and accelerometer sensors.
	*/
	void enableSensors();
	/**
	*	Update magnetometer and accelerometer sensors.
	*   Calculates pitch, roll and yaw using the Tilt-Compensated algorithm. 
	*/
	void updateSensors();
	/** Gets the value of the roll angle
	* @return An integer representing the roll angle in deg the range of -90 to 90.
	*/
	int32_t getRoll();
	/** Gets the value of the pitch angle
	* @return An integer representing the pitch angle in deg the range of -90 to 90.
	*/
	int32_t getPitch();
	/** Gets the value of the yaw angle
	* @return An integer representing the yaw angle in deg in the range of 0 to 360.
	*/
	int32_t getYaw();

};



#endif /* FASTECOMPASS_FASTECOMPASS_H_ */
