#ifndef ASYNC_ADC_H__
#define ASYNC_ADC_H__

#include "mbed.h"
#include "fsl_edma.h"
#include "fsl_adc16.h"
#include "fsl_pdb.h"

#define DMA_AUTO        (-1)

typedef void (*asyncISR)(uint32_t start, uint32_t end);

typedef enum {
    SUCCESS = 0,            // Return value on successful initialisation of async analog initiation
    E_ASYNC_ADC_ACTIVE,     // If an async analog read is initiated whilst another is on going
    E_DMA_IN_USE,           // If the channel is in use (Specified) or there are no free DMA channels (Auto).
    E_INVALID_BUFFER_SIZE   // If the provided buffer size is non-positive or for a non power of 2 circular buffer
} async_error_t;

/*!
 * @brief Computes the nearest achievable frequency to the target frequency.
 * 
 * The floating-point sampling frequency is required for fractional values.
 * 
 * @param targetFrequency The target frequency to achieve
 */
uint32_t approximateFrequency(uint32_t targetFrequency);

/*!
 * @brief Computes the nearest achievable frequency to the target frequency.
 * 
 * The integer sampling frequency is recommended for values over 10kHz.
 * 
 * @param targetFrequency The target frequency to achieve
 */
float approximateFrequency(float targetFrequency);

/*!
 * @brief Initiates an asynchronus analog read from 'source' to 'destination'
 * using the PDB to drive the Analog conversion and DMA to transfer the result
 * to the destination buffer. 
 * 
 * The read will fill the destination buffer and then terminate. A call to
 * terminateAsyncRead() can terminate the process early.
 * 
 * The integer sampling frequency is recommended for values over 10kHz.
 * 
 * @param source The source pin to read from. Must be Analog compatiable
 * @param destination A pointer to the destination array. Must be positive.
 * @param destSize The size of the destination array.
 * @param sampleFrequency The desired sampling frequency from the source pin
 * @param callback An optional callback function to be executed once the
 *          destination buffer is filled.
 * @param dmaChannel The optional desired DMA channel to use, or DMA_AUTO if the 
 *          first free dma channel is to be used. Defaults to DMA_AUTO.
 */
async_error_t asyncAnalogToBuffer(PinName source, uint16_t* destination, uint32_t destSize, uint32_t sampleFrequency, asyncISR callback = NULL, int16_t dmaChannel = DMA_AUTO);

/*!
 * @brief Initiates an asynchronus analog read from 'source' to 'destination'
 * using the PDB to drive the Analog conversion and DMA to transfer the result
 * to the destination buffer. 
 * 
 * The read will fill the destination buffer, then cycle back to the start of
 * the destination buffer and repeat continuously until explicitly terminated by
 * a call to terminateAsyncRead().
 * 
 * The integer sampling frequency is recommended for values over 10kHz.
 * 
 * @brief Initiates an asynchronus analog read from 'source' to 'destination'
 * using the PDB to drive the Analog conversion and DMA to transfer the result
 * to the destination buffer. The integer sampling frequency is recommended for
 * values over 10kHz.
 * @param source The source pin to read from. Must be Analog compatiable
 * @param destination A pointer to the destination array.
 * @param destination A pointer to the destination array. Must be a positive
 *          power of two.
 * @param sampleFrequency The desired sampling frequency from the source pin
 * @param callback An optional callback function to be executed after each half
 *          of the destination buffer is filled.
 * @param dmaChannel The optional desired DMA channel to use, or DMA_AUTO if the 
 *          first free dma channel is to be used. Defaults to DMA_AUTO.
 */
async_error_t asyncAnalogToCircularBuffer(PinName source, uint16_t* destination, uint32_t destSize, uint32_t sampleFrequency, asyncISR callback = NULL, int16_t dmaChannel = DMA_AUTO);

/*!
 * @brief Initiates an asynchronus analog read from 'source' to 'destination'
 * using the PDB to drive the Analog conversion and DMA to transfer the result
 * to the destination buffer. 
 * 
 * The read will fill the destination buffer and then terminate. A call to
 * terminateAsyncRead() can terminate the process early.
 * 
 * The floating-point sampling frequency is required for fractional values.
 * 
 * @param source The source pin to read from. Must be Analog compatiable
 * @param destination A pointer to the destination array.
 * @param destination A pointer to the destination array. Must be positive.
 * @param sampleFrequency The desired sampling frequency from the source pin
 * @param callback An optional callback function to be executed once the
 *          destination buffer is filled.
 * @param dmaChannel The optional desired DMA channel to use, or DMA_AUTO if the 
 *          first free dma channel is to be used. Defaults to DMA_AUTO.
 */
async_error_t asyncAnalogToBuffer_f(PinName source, uint16_t* destination, uint32_t destSize, float sampleFrequency, asyncISR callback = NULL, int16_t dmaChannel = DMA_AUTO);

/*!
 * @brief Initiates an asynchronus analog read from 'source' to 'destination'
 * using the PDB to drive the Analog conversion and DMA to transfer the result
 * to the destination buffer. 
 * 
 * The read will fill the destination buffer, then cycle back to the start of
 * the destination buffer and repeat continuously until explicitly terminated by
 * a call to terminateAsyncRead().
 * 
 * The floating-point sampling frequency is required for fractional values.
 * 
 * @param source The source pin to read from. Must be Analog compatiable
 * @param destination A pointer to the destination array.
 * @param destination A pointer to the destination array. Must be a positive
 *          power of two.
 * @param sampleFrequency The desired sampling frequency from the source pin
 * @param callback An optional callback function to be executed after each half
 *          of the destination buffer is filled.
 * @param dmaChannel The optional desired DMA channel to use, or DMA_AUTO if the 
 *          first free dma channel is to be used. Defaults to DMA_AUTO.
 */
async_error_t asyncAnalogToCircularBuffer_f(PinName source, uint16_t* destination, uint32_t destSize, float sampleFrequency, asyncISR callback = NULL, int16_t dmaChannel = DMA_AUTO);

/*!
 * @brief Terminates any on going async analog reads. Mainly used to terminate
 * the 
 */
void terminateAsyncRead();

#endif /* ASYNC_ADC_H__ */