
#include "mbed.h"

#include <stdint.h>
#include "bme280.h"

#if defined TARGET_LPC81X
    // SDA, SCL for Elektor CoCo-ri-Co!
    #define PIN_SDA  (P0_10)
    #define PIN_SCL  (P0_11)
#elif defined TARGET_NRF51_MICROBIT
    // SDA, SCL for BBC micro:bit
    #define PIN_SDA  I2C_SDA0 /* (P20) */
    #define PIN_SCL  I2C_SCL0 /* (P19) */
#endif

I2C i2c(PIN_SDA,PIN_SCL);
BME280 bme280;

DigitalOut led1(LED1); // Blue on LPC800-MAX, Pad0 on micro:bit
DigitalOut led2(LED2); // Green on LPC800-MAX, Pad1 on micro:bit
DigitalOut led3(LED3); // Red on LPC800-MAX, Pad2 on micro:bit

#if defined TARGET_LPC81X
    #define LED_ON  (0)
    #define LED_OFF  (1)
#elif defined TARGET_NRF51_MICROBIT
    #define LED_ON  (1)
    #define LED_OFF  (0)
#endif

// Note for LPC800-MAX users: check SJ1 and SJ4. 
// For the USB serial port to work pins 1&2 must be shorted, not pins 2&3.


void i2cWrite(uint8_t i2c_address, uint8_t *p_data, uint8_t data_size, uint8_t repeated_start)
{
    // mbed uses 8-bit addresses, always confusing.
    i2c.write(i2c_address<<1,(const char *)p_data,data_size,repeated_start);
}


void i2cRead(uint8_t i2c_address, uint8_t *p_data, uint8_t data_size)
{
    // mbed uses 8-bit addresses, always confusing.
    i2c.read(i2c_address<<1,(char *)p_data,data_size);
}


// Not using BME280 SPI interface, provide stubs.
void spiWrite(uint8_t *p_data, uint8_t data_size) {}
void spiRead(uint8_t *p_data, uint8_t data_size) {}


int main(void)
{
    led1 = LED_OFF;
    led2 = LED_OFF;
    led3 = LED_OFF;
    
    printf("Elektor project 150652\n");
    printf("BME280 weather sensor\n");
#if defined TARGET_LPC81X
    printf("for LPC800-MAX\n");
#elif defined TARGET_NRF51_MICROBIT
    printf("for BBC micro:bit\n");
#endif

#if BME280_ALLOW_FLOAT!=0
    printf("Using floating point maths\n");
#else
    printf("Using integer maths\n");
#endif

    if (bme280.begin(BME280_I2C_ADDRESS2)!=0)
    {
        printf("\n*** BME280 not found.\n");
    }
    printf("\n");
    
    // Configure for test purposes.
    bme280.writeConfigRegister(BME280_STANDBY_500_US,BME280_FILTER_OFF,0);
    bme280.writeControlRegisters(BME280_OVERSAMPLING_1X,BME280_OVERSAMPLING_1X,BME280_OVERSAMPLING_1X,BME280_MODE_NORMAL);

    while (1)
    {
        led1 = LED_ON;
 
        bme280.read();
#if BME280_ALLOW_FLOAT!=0
        printf("T=%0.1f degrees C, P=%0.1f mbar, RH=%0.1f%%\n",bme280.temperature(),bme280.pressure()/100,bme280.humidity());
#else
        printf("T=%d degrees C, P=%d mbar, RH=%d%%\n",bme280.temperature()/100,bme280.pressure()/100,bme280.humidity()>>10);
#endif

        // Worst case speed is about 1 Hz.
        // Keep flash short, it is terribly bright.
        wait_ms(5);
        led1 = LED_OFF;
        wait_ms(995);
    }
}
