/* mbed USBJoystick Library
 * Copyright (c) 2012, v01:  Initial version, WH,
 *                           Modified USBMouse code ARM Limited.
 *                           (c) 2010-2011 mbed.org, MIT License
 *               2016, v02:  Updated USBDevice Lib, Added waitForConnect, Updated 32 bits button
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, inclumosig without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUmosiG BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


#include "stdint.h"
#include "USBJoystick.h"

bool USBJoystick::update(int16_t x, int16_t y, int16_t b, int16_t f, int16_t r, int16_t t, uint8_t hat, uint32_t buttons)
{

    _x = x;
    _y = y;
    _b = b;
    _f = f;
    _r = r;
    _t = t;
    _hat = hat;
    _buttons = buttons;

    return update();
}

bool USBJoystick::update()
{
    uint8_t pos = 0;
    HID_REPORT report;

    // Fill the report according to the Joystick Descriptor
    report.data[pos] = (_x >>  0) & 0xff;

    report.data[++pos] = (_x >>  8) & 0xff;
    report.data[++pos] = _y & 0xff;
    report.data[++pos] = (_y >>  8) & 0xff;
    report.data[++pos] = _b & 0xff;
    report.data[++pos] = (_b >>  8) & 0xff;
    report.data[++pos] = _f & 0xff;
    report.data[++pos] = (_f >>  8) & 0xff;
    report.data[++pos] = _r & 0xff;
    report.data[++pos] = (_r >>  8) & 0xff;
    report.data[++pos] = _t & 0xff;
    report.data[++pos] = (_t >>  8) & 0xff;

#if (HAT4 == 1)
//Use 4 bit padding for hat4 or hat8
    report.data[++pos] = (_hat & 0x0f) ;
#endif

#if (HAT8 == 1)
//Use 4 bit padding for hat4 or hat8
    report.data[++pos] = (_hat & 0x0f) ;
#endif

#if (BUTTONS4 == 1)
//Use 4 bit padding for buttons
    report.data[++pos] = (_buttons & 0x0f) ;
    report.length = ++pos;
#endif

#if (BUTTONS8 == 1)
//Use 8 bits for buttons
    report.data[++pos] = (_buttons & 0xff) ;
    report.length = ++pos;
#endif

#if (BUTTONS32 == 1)
//No bit padding for 32 buttons
    report.data[++pos] = (_buttons >>  0) & 0xff;
    report.data[++pos] = (_buttons >>  8) & 0xff;
    report.data[++pos] = (_buttons >> 16) & 0xff;
    report.data[++pos] = (_buttons >> 24) & 0xff;
    report.length = ++pos;
#endif

    return send(&report);
}

bool USBJoystick::move(int16_t x, int16_t y)
{
    _x = x;
    _y = y;
    return update();
}

bool USBJoystick::diveBreak(int16_t b)
{
    _b = b;
    return update();
}

bool USBJoystick::flaps(int16_t f)
{
    _f = f;
    return update();
}

bool USBJoystick::rudder(int16_t r)
{
    _r = r;
    return update();
}

bool USBJoystick::throttle(int16_t t)
{
    _t = t;
    return update();
}

bool USBJoystick::hat(uint8_t hat)
{
    _hat = hat;
    return update();
}

bool USBJoystick::buttons(uint32_t buttons)
{
    _buttons = buttons;
    return update();
}



void USBJoystick::_init()
{
    _x = 32767;                                     // X-Axis
    _y = 32767;                                     // Y-Axis
    _b = 32767;                                     // Breaks
    _f = 32767;                                     // Wing Flaps
    _r = 32767;                                     // Rudder
    _t = 32767;                                     // Throttle
    _hat = 0x00;                                // POV Hat-Switches
    _buttons = 0x00000000;                      // Buttons
}


uint8_t * USBJoystick::reportDesc()
{
    static uint8_t reportDescriptor[] = {

        USAGE_PAGE(1), 0x01,                    // Generic Desktop
        USAGE(1), 0x04,                         // Usage (Joystick)
        COLLECTION(1), 0x01,                    // Application
        
//******* Joystick Axis and Sliders *******
            USAGE_PAGE(1), 0x01,                // Generic Desktop
            USAGE(1), 0x01,                     // Usage (Pointer) - (Joystick = 0x04)
            COLLECTION(1), 0x00,                // Physical                                     Usage in main.cpp
                USAGE(1), 0x30,                 // X                                            X
                USAGE(1), 0x31,                 // Y                                            Y
                USAGE(1), 0x32,                 // Z                                            Breaks
                USAGE(1), 0x33,                 // Rx                                           Flaps
//                USAGE(1), 0x34,               // Ry
//                USAGE(1), 0x35,               // Rz (Rudder on Usage Page 0x02)
//                USAGE(1), 0x36                // Slider (Throttle on Usage Page 0x02)
                LOGICAL_MINIMUM(2), 0xff, 0x08, // war 0, 0x00, 0x80
                LOGICAL_MAXIMUM(2), 0x00, 0x08, // war 65536, 0xff 0x7f
                REPORT_SIZE(1), 0x10,
                REPORT_COUNT(1), 0x04,
                INPUT(1), 0x02,                 // Data, Variable, Absolute        
            END_COLLECTION(0),

            USAGE_PAGE(1), 0x02,                // Simulation Controls
            USAGE(1), 0xBA,                     // Usage (Rudder)
            USAGE(1), 0xBB,                     // Usage (Throttle)
            LOGICAL_MINIMUM(2), 0xff, 0x08,     // 0
            LOGICAL_MAXIMUM(2), 0x00, 0x08,     // 65536
            REPORT_SIZE(1), 0x10,
            REPORT_COUNT(1), 0x02,
            INPUT(1), 0x02,                     // Data, Variable, Absolute

//******* Hat Switches *******
            USAGE_PAGE(1), 0x01,                // Generic Desktop
            USAGE(1), 0x39,                     // Usage (Hat switch)
#if (HAT4 == 1 && HAT4_8 == 0)
// 4 Position Hat Switch
            LOGICAL_MINIMUM(1), 0x00,           // 0
            LOGICAL_MAXIMUM(1), 0x03,           // 3
            PHYSICAL_MINIMUM(1), 0x00,          // Physical_Minimum (0)
            PHYSICAL_MAXIMUM(2), 0x0E, 0x01,    // Physical_Maximum (270)
            //UNIT_EXPONENT(1), 0x00,           // Unit Exponent (0)
            UNIT(1), 0x14,                      // Unit (Degrees)
#endif
#if (HAT8 == 1) || (HAT4 == 1 && HAT4_8 == 1)
// 8 Position Hat Switch
            LOGICAL_MINIMUM(1), 0x00,          // 0
            LOGICAL_MAXIMUM(1), 0x07,          // 7
            PHYSICAL_MINIMUM(1), 0x00,         // Physical_Minimum (0)
            PHYSICAL_MAXIMUM(2), 0x3B, 0x01,   // Physical_Maximum (315)
            //UNIT_EXPONENT(1), 0x00,          // Unit Exponent (0)
            UNIT(1), 0x14,                     // Unit (Degrees)
#endif
            REPORT_SIZE(1), 0x04,
            REPORT_COUNT(1), 0x01,
            INPUT(1), 0x42,                    // Data, Variable, Absolute, Null State
// Padding 4 bits
            REPORT_SIZE(1), 0x01,
            REPORT_COUNT(1), 0x04,
            INPUT(1), 0x01,                    // Constant
            
//******* Buttons *******
            USAGE_PAGE(1), 0x09,               // Buttons
#if (BUTTONS4 == 1)
// 4 Buttons
            USAGE_MINIMUM(1), 0x01,            // 1
            USAGE_MAXIMUM(1), 0x04,            // 4
            LOGICAL_MINIMUM(1), 0x00,          // 0
            LOGICAL_MAXIMUM(1), 0x01,          // 1
            REPORT_SIZE(1), 0x01,
            REPORT_COUNT(1), 0x04,
            //UNIT_EXPONENT(1), 0x00,       // Unit_Exponent (0)
            UNIT(1), 0x00,                  // Unit (None)
            INPUT(1), 0x02,                 // Data, Variable, Absolute
// Padding 4 bits
            REPORT_SIZE(1), 0x01,
            REPORT_COUNT(1), 0x04,
            INPUT(1), 0x01,                 // Constant
#endif
#if (BUTTONS8 == 1)
// 8 Buttons
            USAGE_MINIMUM(1), 0x01,         // 1
            USAGE_MAXIMUM(1), 0x08,         // 8
            LOGICAL_MINIMUM(1), 0x00,       // 0
            LOGICAL_MAXIMUM(1), 0x01,       // 1
            REPORT_SIZE(1), 0x01,
            REPORT_COUNT(1), 0x08,
            //UNIT_EXPONENT(1), 0x00,       // Unit_Exponent (0)
            UNIT(1), 0x00,                  // Unit (None)
            INPUT(1), 0x02,                 // Data, Variable, Absolute
#endif
#if (BUTTONS32 == 1)
// 32 Buttons
            USAGE_MINIMUM(1), 0x01,         // 1
            USAGE_MAXIMUM(1), 0x20,         // 32
            LOGICAL_MINIMUM(1), 0x00,       // 0
            LOGICAL_MAXIMUM(1), 0x01,       // 1
            REPORT_SIZE(1), 0x01,
            REPORT_COUNT(1), 0x20,
            //UNIT_EXPONENT(1), 0x00,       // Unit_Exponent (0)
            UNIT(1), 0x00,                  // Unit (None)
            INPUT(1), 0x02,                 // Data, Variable, Absolute
#endif

        END_COLLECTION(0)
    };

    reportLength = sizeof(reportDescriptor);
    return reportDescriptor;
}

