#include "morse.h"
#include "mbed.h"

/**
 *  @file       morse.cpp
 *  Project     morse code handling Library
 *  @brief      morse code handling library for mbed
 *  @version    1.0
 *  @author     Chuck Timber
 *  @date       07/08/2014
 */
/** class to generate Morse code tone and keying signal.
 *         sound with a buzzer, based on a PwmOut, 
 *         keying signal base on a DigitalOut
 *
 * Example:
 * @code
 * // Morse sample
 * #include "mbed.h"
 * #include "morse.h"
 *
 * char* codes[] = { ".-", "-...", "-.-.", "-..", ".",
 *                   "..-.", "--.", "....", "..", ".---",
 *                   "-.-", ".-..", "--", "-.", "---",
 *                   ".--.", "--.-", ".-.", "...", "-",
 *                   "..-", "...-", ".--", "-..-", "-.--", "--..", " "
 *                 };
 *
 * Morse morse;
 * //Morse morse(dp24, LED1);
 * //Morse morse(dp24, LED1, 0.1, 800);
 * 
 * int main()
 * {
 *     int i;
 * 
 *     while(1) {
 *         for (i = 0; i < 27; i++) {
 *             morse.code(codes[i]);
 *         }
 *     }
 * }
 * @endcode
 */

/**
 * 0to127-value vs CW_tick table
 *   =0.03*5^(1-i/128)
 */
const float Morse::table[] = { 0.15000000  ,0.14812575  ,0.14627491  ,0.14444721  ,0.14264234  ,0.14086002  ,0.13909997  ,0.13736192  ,
                               0.13564558  ,0.13395068  ,0.13227697  ,0.13062416  ,0.12899201  ,0.12738026  ,0.12578864  ,0.12421691  ,
                               0.12266482  ,0.12113212  ,0.11961857  ,0.11812393  ,0.11664797  ,0.11519045  ,0.11375115  ,0.11232983  ,
                               0.11092626  ,0.10954024  ,0.10817153  ,0.10681993  ,0.10548521  ,0.10416717  ,0.10286560  ,0.10158029  ,
                               0.10031105  ,0.09905766  ,0.09781993  ,0.09659767  ,0.09539068  ,0.09419877  ,0.09302176  ,0.09185945  ,
                               0.09071166  ,0.08957822  ,0.08845894  ,0.08735364  ,0.08626216  ,0.08518431  ,0.08411993  ,0.08306885  ,
                               0.08203091  ,0.08100593  ,0.07999376  ,0.07899424  ,0.07800720  ,0.07703250  ,0.07606998  ,0.07511948  ,
                               0.07418086  ,0.07325397  ,0.07233866  ,0.07143479  ,0.07054221  ,0.06966079  ,0.06879037  ,0.06793084  ,
                               0.06708204  ,0.06624385  ,0.06541613  ,0.06459875  ,0.06379159  ,0.06299452  ,0.06220740  ,0.06143012  ,
                               0.06066255  ,0.05990457  ,0.05915606  ,0.05841690  ,0.05768698  ,0.05696618  ,0.05625439  ,0.05555149  ,
                               0.05485737  ,0.05417193  ,0.05349505  ,0.05282663  ,0.05216656  ,0.05151474  ,0.05087106  ,0.05023543  ,
                               0.04960773  ,0.04898788  ,0.04837578  ,0.04777132  ,0.04717442  ,0.04658497  ,0.04600289  ,0.04542809  ,
                               0.04486046  ,0.04429993  ,0.04374640  ,0.04319979  ,0.04266001  ,0.04212697  ,0.04160059  ,0.04108079  ,
                               0.04056749  ,0.04006060  ,0.03956004  ,0.03906574  ,0.03857761  ,0.03809558  ,0.03761958  ,0.03714952  ,
                               0.03668534  ,0.03622695  ,0.03577430  ,0.03532730  ,0.03488588  ,0.03444998  ,0.03401953  ,0.03359445  ,
                               0.03317469  ,0.03276017  ,0.03235083  ,0.03194661  ,0.03154744  ,0.03115325  ,0.03076399  ,0.03037959  };

/** beeping function
 * this is a private funciton / internal use only
 *
 * @param k - beep duration factor; k times cw_tick
 */
void Morse::beep (int k)
{
    _pwm.write(0.5);
    _pout = 1;
    wait(cw_tick * (float)k);
    _pwm.write(0.0);
    _pout = 0;
}

/** spacing function
 * this is a private funciton / internal use only
 *
 * @param k - spacing factor; k times cw_tick
 */
void Morse::space(int k)
{
    wait(cw_tick * (float)k);
}

/** set morse oscillation frequency
 *
 * @param f - a morse code speed tick [0.03-0.3] (sec)
 */
void Morse::setfreq(float f)
{
    freq = f;
    _pwm.period(1.0/freq);
    _pwm.write(0.0);
}

/** get morse oscillation frequency
 *
 * @return morse oscillation frequency (Hz) 
 */
float Morse::getfreq()
{
    return freq;
}

/** set morse code speed
 *
 * @param t - a morse code speed tick [0.03-0.3] (sec)
 */
void Morse::settick(float t)
{
    if ((t > 0.03) && (t < 0.3))
        cw_tick = t;
}

/** get morse code speed tick
 *
 * @return a morse code speed tick
 */
float Morse::gettick()
{
    return cw_tick;
}

/** set morse code speed index
 *
 * @param idx - a morse code speed index [0-127]
 */
void Morse::setidx(int idx)
{
    tick_idx = idx;
    settick(table[tick_idx]);
}

/** get morse code speed index
 *
 * @return a morse code speed index [0-127]
 */
int Morse::getidx(void)
{
    return tick_idx;
}

/** increment morse code speed index
 *
 */
void Morse::incidx(void)
{
    tick_idx++;
    if (tick_idx > 127)
        tick_idx = 127;

    settick(table[tick_idx]);
}

/** decrement morse code speed index
 *
 */
void Morse::decidx(void)
{
    tick_idx--;
    if (tick_idx < 0)
        tick_idx = 0;

    settick(table[tick_idx]);
}


/** Coding function
 *
 * @param code - a pointer to code strings
 */
void Morse::code(char* code)
{
    char* ptr;

    for (ptr = code; *ptr; ptr++, space(1) ) {
        switch (*ptr) {
            case '-':
                beep(3); // dash();
                break;
            case '.':
                beep(1); // dot();
                break;
            case ' ':
                space(7);
                break;
            default:
                continue;
        }
    }
    space(2);
}

 // constructors
 /** Create a Morse object connected to the specified PwmOut pin and DigitalOut pin
  *
  * @param pwm - PwmOut pin to connect to (default dp24)
  * @param led - DigitalOut pin to connect to (default LED1)
  * @param t   - float t, cw tick (default 0.1sec) 
  * @param f   - float f, oscillation frequency (default 800Hz) 
  */

Morse::Morse() : _pout( LED1 ), _pwm( dp24 )
{
    cw_tick = 0.1;
    freq = 800;
    _pwm.period(1.0/freq);
    _pwm.write(0.0);
    _pout = 0;
}

Morse::Morse(PinName pwm, PinName led) : _pout( led ), _pwm( pwm )
{
    cw_tick = 0.1;
    freq = 800;
    _pwm.period(1.0/freq);
    _pwm.write(0.0);
    _pout = 0;
}

Morse::Morse(PinName pwm, PinName led, float t) : _pout( led ), _pwm( pwm )
{
    cw_tick = t;
    freq = 800;
    _pwm.period(1.0/freq);
    _pwm.write(0.0);
    _pout = 0;
}

Morse::Morse(PinName pwm, PinName led, float t, float f) : _pout( led ), _pwm( pwm )
{
    cw_tick = t;
    freq = f;
    _pwm.period(1.0/freq);
    _pwm.write(0.0);
    _pout = 0;
}
