/**
 *  @file       REnc.h
 *  Project     Rotary Encoder handling Library
 *  @brief      Rotary Encoder handling library for mbed
 *  @version    1.0
 *  @author     Chuck Timber
 *  @date       12/08/2014
 */

#ifndef MBED_RENC_H
#define MBED_RENC_H

#include "mbed.h"

/**
 *  @define
 *
 */
#define RENC_SAMPLING_PERIOD 0.002
#define RENC_EXECUTION_DELAY 300

namespace mbed {

    enum { STOP, CLOCKWISE, COUNTERCLOCKWISE };
//    enum { STOP, CC,        CCW              };

/** Class: REnc
 *  A class to handle a rotary encoder 
 *   This class uses DigitalIn and Ticker
 *
 * Refered to: http://elm-chan.org/docs/tec/te04.html
 *
 * Example:
 * @code
 * // REnc sample
 * #include "mbed.h"
 * #include "REnc.h"
 * void proc_REnc_right(void);
 * void proc_REnc_left(void);
 * 
 * // void proc_REnc_right(void) { }
 * // void proc_REnc_left(void)  { }
 * 
 * REnc renc(dp1, dp2);
 * 
 * int main()
 * {
 * 
 *     // renc.setHandleRight(&proc_REnc_right);
 *     // renc.setHandleLeft(&proc_REnc_left);
 *  
 *     while(1) {
 *         if (renc.CMD == CLOCKWISE) {
 *             proc_REnc_right();
 *             renc.CMD = STOP;
 *         } else if (renc.CMD == COUNTERCLOCKWISE) {
 *             proc_REnc_left();
 *             renc.CMD = STOP;
 *         }
 *         if (renc.STABLE) {
 *             // do_something
 *         }
 *     }
 * }
 * @endcode
 */
class REnc {

public:
    unsigned char CMD;    /*!< CMD represents the state of rotation
                           *   @retval STOP
                           *   @retval CLOCKWISE
                           *   @retval COUNTERCLOCKWISE
                           */
    unsigned char STABLE; /*!< STABLE represents the rotation stability,
                           *   raises RENC_EXECUTION_DELAY counts after REnc STOPping
                           *   @retval 1 - stable
                           */ 

    /** set callback function to Clockwise TURN */
    void setHandleCC(void (*fptr)(void));
    /** set callback function to Counterclockwise TURN */
    void setHandleCCW(void (*fptr)(void));

    /// constructor
    REnc(PinName pina, PinName pinb);
    /// destructor
    virtual ~REnc() { };

private:
    void sample_encoder(void);
    void (*mCCCallback)(void);
    void (*mCCWCallback)(void);

    DigitalIn _pina;
    DigitalIn _pinb;
    Ticker _tick;

}; /* end of REnc class definition */

} /* end of namespace mbed */
#endif
