#include "mbed.h"

#ifndef PIDControl_h
#define PIDControl_h

/*! Diese Klasse erstellt einen PID-Regler. die einzelnen Regeranteile können bei der Initialisierung seperat einsgestellt werden.
    Bei der Initialisierung des Reglers bzw. der Klasse wird der Statusvektor resettet und die einzelnen Anteile werden auf null gesetzt und gesperrt.
    Solange die Variable Lock nicht wahr ist, kann über die Funktionen setKP(), setKI() und setKD() der Regler parametriert werden.
    Es ist zudem möglich die Regelgrenzen des Regelers festzulegen. Bei Überschreiten der Regelgrenzen wird ein Eintrag in dem Statusvektor erzeugt.
    Bei einer Änderung im Statusvektor kann eine externe Funktion aufgerufen werden, die dann den Statusvektor auswertet. 
    Der Regeltask ctrltask sollte in regelmäßigen Abständen aufgerufen werden. Wichtig dabei ist eine Stänige Übergabe des Ist und Soll-Werts sowie die Regelzeit.
    \image html http://developer.mbed.org/media/uploads/ChrisselH/pid-controller.jpg "Schematischer Regleraufbau ohne Statusvektoren und Fehlermeldungen"
 * @code
 * #include "mbed.h"
 * #include "PIDControl.h"        
 *
 * int16_t ist_value;                                   // Ist-Wert der Strecke
 * int16_t soll_value;                                  // Soll-Wert der Strecke
 * int16_t stell_value;                                 // Stell-Wert der Strecke
 *
 * uint8_t statusvektor;                                // Statusvektor des Reglers
 *
 * int main() {
 *    
 *     PIDControl PIregler(1,50.8,1,0.5,0,0);          // PI-Regler mit KP = 50,8 und KI = 0,5
 *
 *     PIregler.setIlimits(1,6000,1000);                // Aktivierung der maximale und minimale Größe des I-Anteils
 *
 *     PIregler.setAWU(5);                              // Die Differenz zwischen I-Anteil und Limit wird mit 
 *                                                      // Verstärkung zurückgeführt
 *
 *     PIregler.setUlimits(7000,500);                   // Limitierung der Stellgröße
 *
 *     PIregler.lock();                                 // Sperrt weitere Einstellung an dem Regler
 * 
 *      while(1){
 *      
 *          stell = PIregler.ctrltask(ist, soll, 1000); // Die Schleife wiederholt sich alle 1kHz  
 * 
 *          statusvektor = PIregler.getStatus();        // Status des Reglers wird ausgelesen
 *
 *        wait(0.001);                                  // Regler wird alle 1kHz aufgerufen
 *      }
 *    }
 *
 * @endcode
 */

class PIDControl
{
    public:

        /*! Konstruktor zum Erstellen einer Instanz mit einer vom User festgelegten Größe */
        /*!
            \param P <b>P-Anteil</b> Aktivert den P-Anteil, wenn der Wert 1 ist
            \param KP <b>P-Anteil</b> Verstärkung des P-Anteils
            \param I <b>I-Anteil</b> Aktivert den I-Anteil, wenn der Wert 1 ist
            \param KI <b>I-Anteil</b> Verstärkung des I-Anteils
            \param D <b>D-Anteil</b> Aktivert den D-Anteil, wenn der Wert 1 ist
            \param KD <b>D-Anteil</b> Verstärkung des D-Anteils
        */      
        PIDControl(bool P_usr, float KP_usr, bool I_usr, float KI_usr, bool D_usr, float KD_usr);
        
        /*! Destruktor entfernt den Regler */
        ~PIDControl(){};
            
        /*! Reglerprozess. Rückgabewert ist die Führungsgröße */
        /*!
            \param usoll <b>SOLL-Wert</b> Soll-Wert der Regelstrecke
            \param yist <b>IST-Wert</b> Ist-Wert der Regelstrecke
            \param time <b>Zeitkonstatnte</b> vergangene Zeit seit dem letzen Regeltask in ganzen Mirkosekunden (1ms = 1000µs)
        */ 
        int16_t ctrltask (int16_t wsoll, int16_t yist, uint16_t time);
        
    
        /*! Ermöglich Zugriff auf die Reglerparameter des P-Anteils. Rückgabe Wert ist eins, wenn die Reglerwerte gespeichert worden sind.*/
        /*!
            \param P <b>P-Anteil</b> Aktivert den P-Anteil, wenn der Wert 1 ist
            \param KP <b>P-Anteil</b> Verstärkung des P-Anteils
        */
        bool setKP(bool P_usr, float KP_usr);
    
        /*! Ermöglich Zugriff auf die Reglerparameter des I-Anteils. Rückgabe Wert ist eins, wenn die Reglerwerte gespeichert worden sind.*/
        /*!
            \param I <b>I-Anteil</b> Aktivert den I-Anteil, wenn der Wert 1 ist
            \param KI <b>I-Anteil</b> Verstärkung des I-Anteils
        */
        bool setKI(bool I_usr, float KI_usr);
        
        /*! Ermöglich Zugriff auf die Reglerparameter des D-Anteils. Rückgabe Wert ist eins, wenn die Reglerwerte gespeichert worden sind.*/
        /*!
            \param D <b>D-Anteil</b> Aktivert den D-Anteil, wenn der Wert 1 ist
            \param KD <b>D-Anteil</b> Verstärkung des D-Anteils
        */
        bool setKD(bool D_usr, float KD_usr);    
        
        /*! Ermöglich Zugriff auf den Anti-Windup zum begrenzen des I-Anteils*/
        /*!
            \param awu_gain <b>Anti-Windup Verstärkung</b> Rückführung der Differenz zwischen Stellgröße und Begrenzung mit einer Verstärkung in den I-Anteil
        */
        bool setAWU(float awu_gain_usr);
                
        /*! Setzt die Limits der Stellgröße U bzw. des Ausgangs des Reglers*/
        /*!
            \param u_max <b>maximales Limit</b> maximale Ausgangsgröße
            \param u_min <b>minimales Limit</b> minimale Ausgangsgröße
        */
        void setUlimits(int16_t u_max_usr, int16_t u_min_usr);
        
        /*! Setzt die Limits des Integralanteils */
        /*!
            \param integrallimiter <b>Aktiviert den Limiter</b> aktiviert eine seperate Begrenzung des I-Anteils
            \param i_max <b>maximales Limit</b> maximaler I-Anteil
            \param i_min <b>minimsler Limit</b> minimaler I-Anteil
        */
        void setIlimits(bool integrallimiter_usr, int16_t i_max_usr, int16_t i_min_usr);
        
        /*! Sperrt den Zugriff auf die Reglerparameter */
        void lock();
        
        /*! Erlaubt den Zugriff auf die Reglerparameter */
        void unlock();
        
        /*! Liest den Status des Reglers aus */    
        uint8_t getStatus();
        
        /*! Setzt eine Routine die verwendet wird, wenn ein schwerwiegender Fehler im Statusvektor gesetzt wird.*/ 
        /*!
            \param BUFFER_HANDLER Adresse zur weiteren Routine
         */ 
        void setERFFCT(void (*CTRL_HANDLER)(void));  
           
        private: 
        /*! Setzt ein Bit im Statusvektor. Wenn der Rückgabewert True ist, würde der Wert übernommen*/
        /*!
            \param bit  Welches Bit der Zelle gesetzt werden soll
        */     
        void setStatus(int bit);
     
        /*!ResSetzt ein Bit im Statusvektor. Wenn der Rückgabewert True ist, würde der Wert übernommen*/
        /*!
            \param bit  Welches Bit der Zelle zurück gesetzt werden soll
        */     
        void resetStatus(int bit);
        
        //! Zeitwert der letztes Berechnung des Verschiebungswerts
        int16_t time_last;
        //! Dazugehöriger Verschiebungswert
        int16_t time_last_result;
         //! Führungsgröße und Rückgabe des Reglertasks
        int16_t u; 
        //! Freigabe zur Verwendung externer Funktionen 
        bool extfct;
        //! Externe Funktion bei schwerwiegenden Fehlern
        void (*CTRL_HANDLER)(void);
        //! Reglereinstellungen können nicht mehr verändert werden.
        bool lockctrl;
        //! Limits für den Integralanteil aktiv
        bool integrallimiter;
        //! Vorheriger Ístwert des Regelfehlers
        int16_t e_last;
        //! Integral über alle bisherigern Werte
        int16_t e_sum;
        //! Antiwindup Wert mit Verstärkung awu_gain
        int16_t awu_value; 
        //! Regleranteil des P-Regler
        int16_t P_var;
        //! Regleranteil des I-Regler
        int16_t I_var;
        //! Regleranteil des D-Regler
        int16_t D_var;  
    
        //! Funktion zum bestimmen der zweier Potenz
        /*!
            \param a  16-Bit Integer bei der die Zweierpotenz bestimmt werden soll
        */
        int16_t log2(int16_t a);
        
    
    protected:
    
        //! Aktiviert den Proportional-Anteil
        bool P;
        //! Aktiviert den Integral-Anteil
        bool I;
        //! Aktiviert den Differnetial-Anteil
        bool D;  
        
        //! Verstärkung für den Proportional-Anteil
        float KP;
        //! Verstärkung für den Integral-Anteil
        float KI;
        //! Verstärkung für den Differnetial-Anteil
        float KD;
        
        //! Maximale Führungsgröße
        int16_t u_max;
        //! Minimale Führungsgröße
        int16_t u_min;
        //! Maximaler Integralanteil
        int16_t i_max;
        //! Minimaler Integralanteil
        int16_t i_min;
        //! Anti-Windup Verstärkung
        float awu_gain;
        

        /*! 8-Bit Statusvektor:
                -  [7] Regelgrenze max überschritten (<b>EXTFCT</b>)<br>
                -  [6] Regelgrenze min unterschritten (<b>EXTFCT</b>)<br>
                -  [5] Fehler bei der Berechnung (<b>EXTFCT</b>)<br>
                -  [4] aktueller IST-Wert < Soll-Wert<br>
                -  [3] aktueller IST-Wert > Soll-Wert<br>
                -  [2] Regelabweichung ist null<br>
                -  [1] I-Anteil Limit erreicht<br>
                -  [0] Reglertask aktiv<br> 
            Bei schwerwiegenden Fehlern wird eine externe Routine (<b>EXTFCT</b>) aufgerufen, falls diese angegeben wurde.          
        */
        uint8_t status;
 
    };
    
    


#endif
