#include "mbed.h"

#ifndef RingBuffer_h
#define RingBuffer_h

/*! Diese Klasse erstellt einen 32 Bit Ringbuffer mit einen vom Nutzer vorgegebenen Größenbereich. In dem Ringbuffer können 
entsprechend 32 Bit Daten für Messaufzeichnungen genutzt werden. Zudem gibt es 8 Bit in einem Statusregister die frei definiert
werden können. Es gibt für jedes einzelne Bit des Statusregisters eine Setter und Getter Methode. Nachdem der Ringbuffer komplett 
beschrieben wurde, wird wieder der erste Wert überschrieben. Wenn der Abstand zwischen Lese- und Schreibezeiger zu gering wird,
kann eine vom User definierte Funktion aufgerufen werden.
 * @code
 * #include "mbed.h"
 * #include "RingBuffer.h"
 *
 * int Event_Routine(){ .... }                        // Irgendeine Funktion die bei fast vollem Fifo aufgerufen wird           
 *
 * bool write;
 * uint8_t status;
 *
 * int main() {
 *    
 *     Ringbuffer adcbuffer(250);                       // Erstellen einer Instanz mit 250 Werten
 * 
 *     write = adcbuffer.writeBuffer(16658);            // Schreibt einen Wert in den Buffer. Wenn Rückgabe write
 *                                                      // true ist wurde dieser erfolgreich beschrieben.
 *
 *     adcbuffer.setBF_METHODE(Event_Routine);          // Bei fast vollem Fifo wird die Routine aufgerufen
 *
 *     adcbuffer.setBufferStatus(4);                    // Setzt das 4. Bit des Statusregisters
 *
 *     adcbuffer.resetBufferStatus(6);                  // Löscht das 6. Bit des Statusregisters
 *
 *     status = adcbuffer.getBufferStatus();            // Liest das Statusregister aus
 *
 *      while(1){
 *        printf("Value: %i\r\n", adcbuffer.readBuffer());   // Ausgabe der geschriebenen Werte
 *      }
 *    }
 *
 * @endcode
 */


class RingBuffer
{

    public:
    /*! Konstruktor zum Erstellen einer Instanz mit einer vom User festgelegten Größe */
    /*!
        \param size <b>Größe</b> Anzahl der maximal gespeicherten Werte im Buffer
    */     
    RingBuffer(const int size);
    
    /*! Destruktor entfernt den RingBuffer */
    ~RingBuffer(){};
    
    /*! Schreibt Daten in die aktuelle Zelle. Wenn die Rückgabe true ist, dann wurde der Wert geschrieben. Wenn nicht, ist der Buffer voll.  */
    /*!
        \param inputdata <b>Daten</b> 32 Bit Integer Daten für die aktuelle Zelle
    */     
    bool writeBuffer(uint32_t inputdata);

    /*! Setzt ein Bit im Statusvektor des Buffers. Wenn der Rückgabewert True ist, würde der Wert übernommen*/
    /*!
        \param bit  Welches Bit der Zelle gesetzt werden soll
    */     
    bool setBufferStatus(int bit);
 
    /*!ResSetzt ein Bit im Statusvektor des Buffers. Wenn der Rückgabewert True ist, würde der Wert übernommen*/
    /*!
        \param bit  Welches Bit der Zelle zurück gesetzt werden soll
    */     
    bool resetBufferStatus(int bit);
    
    
    /*! Liest den Status aus der aktuelle Zelle und gibt diesen zurück. */    
    uint8_t getBufferStatus();
    
 
    /*! Liest Daten aus der aktuelle Zelle. Wenn die Werte nicht ausgelesen werden konnten, dann ist der Rückgabewert 0 */    
    uint32_t readBuffer();
      
     /*! Setzt eine weitere Routine die verwendet werden kann, wenn der Buffer fast voll ist. Um die Daten z.B. zu übernehmen */ 
     /*!
           \param BUFFER_HANDLER Adresse zur weiteren Routine
     */ 
    void setBF_METHODE(void (*BUFFER_HANDLER)(void));   
    
    protected:   
    //! Größe des Buffers
    int buffer_size;
    //! Lesezeiger der auf die nächste zulesenede Stelle zeigt
    uint32_t read_pointer;
    //! Schreibzeiger der auf die nächste zubeschreibene Stelle zeigt
    uint32_t write_pointer; 
    //! Buffer ist voll
    bool bufferfull;
    //! Buffer ist leer
    bool bufferempty;
    //! Freigabe zur Verwendung externer Funktionen 
    bool extfct;
    //! Externe Funktion bei fast vollem Buffer aufrufen
    void (*BUFFER_HANDLER)(void);  
    
    private:
    //! Eigentlicher Datenspeicher mit einem 32Bit Daten- und einem 8Bit Statusvektor    
    struct buffer          
    {       uint32_t *data;
            uint8_t *status;   
     };
    //Erstellt eine Instanz des Buffers
    buffer buffer;       
            
};
    

#endif
