#include "mbed.h"
#include "PIDControl.h"

//====================================================================================================================================
//                                                  Konstruktor und Regelprozess
//====================================================================================================================================

PIDControl::PIDControl(bool P_usr, float KP_usr, bool I_usr, float KI_usr, bool D_usr, float KD_usr){
    
      awu_gain = 0;                                                 // Anti-Windup deaktivieren
      u_max = 32760;                                                // Maximale Stellgröße limitiert auf max. Integer Wert
      u_min = -32760;                                               // Minimale Stellgröße limitiert auf min. Integer Wert
      lockctrl = false;                                             // Regler frei parametrierbar
      
      e_last = 0;                                                   // Letzter Regelfehler ist 0
      e_sum = 0;                                                    // Integralteil ist 0
      u = 0;                                                        // Führungsgröße ist 0
      awu_value = 0;                                                // Atin-Windup größe ist 0
      status = 0;                                                   // Statusregister bisher nicht gefüllt
      extfct = false;                                               // Keine Externe Funktion festgelegt
      
      P = P_usr;                                                    // P-Anteil on/off
      KP = KP_usr;                                                  // P-Anteil Verstärkung
      I = I_usr;                                                    // I-Anteil on/off
      KI = KI_usr;                                                  // I-Anteil Verstärkung
      D = D_usr;                                                    // D-Anteil on/off
      KD = KD_usr;                                                  // D-Anteil Verstärkung
     
    }


int16_t PIDControl::ctrltask (int16_t wsoll, int16_t yist, uint16_t time){
    
        setStatus(0);                                               // Regeltask aktiv - Flag aktivieren
    
        int16_t e = wsoll - yist;                                   // Regelfehler bestimmen             
        e_sum = e_sum + e;                                          // Integralanteil bestimmen
        
        if(e > 0){                                                  // Regelfehler positiv
            setStatus(4);                                           // Regelfehler positiv - Flag aktivieren
            resetStatus(3);                                         // Regelfehler negativ - Flag deaktivieren
            resetStatus(2);                                         // Regelfehler null - Flag deaktivieren            
        }
        
        if(e < 0){                                                  // Regelfehler negativ
            resetStatus(4);                                         // Regelfehler positiv - Flag deaktivieren
            setStatus(3);                                           // Regelfehler negativ - Flag aktivieren
            resetStatus(2);                                         // Regelfehler null - Flag deaktivieren            
        }
                       
        if(e == 0){                                                 // Regelfehler null
            resetStatus(4);                                         // Regelfehler positiv - Flag deaktivieren
            resetStatus(3);                                         // Regelfehler negativ - Flag deaktivieren
            setStatus(2);                                           // Regelfehler null - Flag aktivieren
        }
        
        int16_t divisor = log2(time);                               // Bestimmung der Zeller um die das geschoben werden soll
       
        if(P){                                                      // P-Anteil bestimmen
            P_var = (KP * e);                                       // u = Kp * e   
        }
        else{
            P_var = 0;                                            // kein P-Anteil vorhanden
        }
            
        if(D){                                                      // D-Anteil bestimmen
            D_var = KD * time * (e - e_last);                       // u = Kd * Ta * (e - e_last) + P-Anteil 
            D_var = D_var << 10;                                    // Zeitanpassung, da eingabe in µs  
        }
        else{
            D_var = 0;                                              // kein D-Anteil vorhanden
        }
            
        if(I){                                                      // I-Anteil bestimmen
            I_var = (KI * (e_sum - awu_value));                     // u = Ki / Ta * (e_sum - awu_value) + P-Anteil + D-Anteil  
            I_var = I_var  << divisor;                              // Division durch die Zeit
        }
        else{
            I_var = 0;                                              // kein I-Anteil vorhanden
        }               

        resetStatus(1);                                             // I-Anteil an Grenze geraten

        if(integrallimiter){                                        // Limiteriung des I-Anteils erforderlich?
            if(e_sum >= i_max){                                     // Limitierung der Integralsumme zur oberen Schwelle
                setStatus(1);                                       // Fehler I-Anteil über Grenzwerte
                e_sum = i_max;                                      // Limitierung der Summe
                awu_value = (e_sum - i_max) * awu_gain;             // Anti-Windup Berechnung
            }

            if(e_sum <= i_min){                                     // Limitierung  der Integralsumme zur unteren Schwelle
                setStatus(1);                                       // Fehler I-Anteil über Grenzwerte
                e_sum = i_min;                                      // Limitierung der Summe
                awu_value = (e_sum - i_min) * awu_gain;             // Anti-Windup Berechnung
            }
        }  
   
        u = P_var + D_var + I_var;                                  // Reglerausgang definieren
        
        resetStatus(6);                                             // Regelgrenze max überschritten zurücksetzen
        resetStatus(7);                                             // Regelgrenze min überschritten zurücksetzen  
    
        if(u >= u_max){                                             // Limitierung des Regelausgangs zur oberen Schwelle
            setStatus(7);                                           // Fehler Regelgrenze max überschritten
            u = u_max;                                              // Ausgabe des Stellgrößen Limits 
        }

        if(u <= u_min){                                             // Limitierung des Regelausgangs zur unteren Schwelle
            setStatus(6);                                           // Fehler Regelgrenze min überschritten
            u = u_min;                                              // Ausgabe des Stellgrößen Limits 
        }      
            
        e = e_last;                                                 // Regelfehler als letzten Regelfehler übernehmen
        resetStatus(0);                                             // Regeltask aktiv - Flag deaktivieren
        
    return u;                                                       // Ausgabe der Stellgröße  
   
    }

//====================================================================================================================================
//                                                   Setter und Getter Funktionen
//====================================================================================================================================
   
bool PIDControl::setKP(bool P_usr, float KP_usr){                   // Setzen des P-Anteils
    
    if(!lockctrl){
        P = P_usr;                                                  // P-Anteil on/off
        KP = KP_usr;                                                // P-Anteil Verstärkung
        
        return true;                                                // Setzen der Werte erfolgreich     
        }
    else{ 
        return false;                                               // Setzen der Werte nicht erfolgreich
        }   
    } 

    
bool PIDControl::setKI(bool I_usr, float KI_usr){                   // Setzen des I-Anteils
    
    if(!lockctrl){
        I = I_usr;                                                  // I-Anteil on/off
        KI = KI_usr;                                                // I-Anteil Verstärkung
        
        return true;                                                // Setzen der Werte erfolgreich     
        }
    else{ 
        return false;                                               // Setzen der Werte nicht erfolgreich
        }   
    } 


bool PIDControl::setKD(bool D_usr, float KD_usr){                   // Setzen des D-Anteils
    
    if(!lockctrl){
        D = D_usr;                                                  // D-Anteil on/off
        KD = KD_usr;                                                // D-Anteil Verstärkung
        
        return true;                                                // Setzen der Werte erfolgreich     
        }
    else{ 
        return false;                                               // Setzen der Werte nicht erfolgreich
        }   
    } 


bool PIDControl::setAWU(float awu_gain_usr){                        // Setzen des Anti-Windups
    
    if(!lockctrl){
        awu_gain = awu_gain_usr;                                    // Anti-Windup Verstärkung
        
        return true;                                                // Setzen des Wertes erfolgreich     
        }
    else{ 
        return false;                                               // Setzen des Wertes nicht erfolgreich
        }   
    }  

void PIDControl::setIlimits(bool integrallimiter_usr, int16_t i_max_usr, int16_t i_min_usr){ // Setzen des Integrallimits

    if(!lockctrl){
        integrallimiter = integrallimiter_usr;                      // Integrallimits aktivieren
        i_max = i_max_usr;                                          // maximaler I-Anteil 
        i_min = i_min_usr;                                          // minimaler I-Anteil   
    }
}

void PIDControl::setUlimits(int16_t u_max_usr, int16_t u_min_usr){  // Setzen des Ausgangsgrößenlimits

    if(!lockctrl){
        u_max = u_max_usr;                                          // maximale Stellgröße
        u_min = u_min_usr;                                          // minimale Stellgröße  
    }
}
        
 
    
void PIDControl::lock(){
         lockctrl = true;                                           //  Einstellung der Regelparamieter sperren             
    }


void PIDControl::unlock(){
          lockctrl = false;                                         // Einstellung der Regelparamter freigeben 
    } 
 
    
uint8_t PIDControl::getStatus(){ 
    return status;                                                  // Rückgabe des aktuellen Statusvektors
}


void PIDControl::setERFFCT(void (*EXTERN_CTRL_HANDLER)(void)){         // Adresse zur externen Funktion übergeben und Freigabe setzen
          extfct = true;                                               // Externe Funktion vorhanden. Freigabe setzen.
          CTRL_HANDLER = EXTERN_CTRL_HANDLER;                          // Funktionspointer der Funktion übernehmen     
    } 

//====================================================================================================================================
//                                                   Interne Funktionen
//====================================================================================================================================


int16_t PIDControl::log2(int16_t a){                                // Bestimmt die nächstkleinere Zweierpotenz der Zahl a
  
        if(a == time_last){
            return time_last_result;                                // Aktueller Wert entspricht letztem Wert
            }               
        
         a = time_last;                                             // Speichert aktuellen Wert
  
         static const int DeBruijnFolge[32] = { 0, 9, 1, 10, 13, 21, 2, 29, 11, 14, 16, 18, 22, 25, 3, 30, 8, 12, 20, 28, 15, 17, 24, 7, 19, 27, 23, 6, 26, 5, 4, 31};

            a |= a >> 1;                                            // Runden auf die kleinste Stelle
            a |= a >> 2;
            a |= a >> 4;
            a |= a >> 8;
            a |= a >> 16;
           
           time_last_result = DeBruijnFolge[(uint32_t)(a * 0x07C4ACDDU) >> 27];     // Ausgabe der passenden Zweierpotenz
           
    return time_last_result; 
  }


void PIDControl::setStatus(int bit){ 
    
    status |= (1 << bit);                                           // Setzt das vorher ausgewählte Bit    
    
    if((extfct) && (bit >= 5 )){                                    // Wenn externe Funktionen aktiviert sind und ein schwerwiegender Fehler abgelegt wird
        (*CTRL_HANDLER)();                                          // Aufruf der Externen Funktion   
    }  
}


void PIDControl::resetStatus(int bit){ 

    int temp = 0xFF;                                                // Maske erstellen
    temp ^= (1 << bit);                                             // Maske anpassen 
    status &= temp;                                                 // Resetzen des entsprechenden Bits
                                     
}
