#include "mbed.h"

#ifndef AD5664_h
#define AD5664_h

/*! Diese Klasse dient zur Ansteuerung des Digital-Analog-Wandlers AD5664 der Firma Analog Devices. Der DAC ist ein 16-Bit Wandler mit 4 Kanälen, Versorgungsspannung von 2,7V bis 5,5V. 
Die Maximale Settling Time beträgt 7µs (AD5664), und die Genauigkeit liegt bei +-12 LSBs maximum. Angesteuert wird er mittels SCLK, CS und MOSI mit maximal 50 MHz. Zuvor muss die SPI-Schnittstelle definiert werden. Die 
Chip Select Schnittstelle kann durch die Initialisierung mit Übergeben werden. Es ist darauf zu achten das der Chip bzw die Library nur den SPI-Modus "1" (CLK Standart auf 0 und übernahme mit fallender Flanke) mit einer Datenbreite von 8-Bit unterstützt.
 
 * @code
 * #include "mbed.h"
 * #include "AD5664.h"
 *
 * SPI interface(PB_5, PB_4, PB_3);         // SPI Interface und Pins (MISO PB.5 , MOSI PB.4, SCLK PB.3) festlegen
 *
 * int main() {
 *    
 *     interface.format(8,1);               // Interface Format festlegen: 8-Bit, SPI-Modus 1
 *     interface.frequency(1000000);        // Übertragungsgeschwindigkeit 1MHz 
 *      
 *     AD5664 device(interface, PB_2);     // Eine Instanz des DACs erzeugen, Chipselect ist PB.2
 *            
 *     device.loadDAC('D', 0x11FF);         // Den Wert 0x11FF in den Kanal D laden
 *  
 *     device.updateDAC('D');               // Den geladenen Wert am Kanal D ausgeben
 *
 *     device.writeDAC('A', 0x89BD);        // Den Wert 0x89BD an dem Kanal A direkt ausgeben
 *   
 *      }
 *
 * @endcode
 */
 
class AD5664     
{   
    public:
    
     /*! Konstruktor zum Erstellen einer Instanz für einen vorher spezifizierten SPI-Port und Chip Select-Pin. Nach dem Erstellen dieser Instanz werden alle Ausgänge des DACs auf 0 gesetzt. */
     /*!
        \param spi Entsprechner SPI-Port (Datenbreite 8 Bit, SPI Modus 1)
        \param daccs Pin für Chipselect
     */                   
    AD5664(SPI _spi, PinName _daccs);
     
     
        
     /*! Funktion zum Beschreiben des Chips mit je 8-Bit. Für genauere Details bitte das Datenblatt lesen*/
     /*!
        \param instruction <b>Direkte Anweisungen für den AD5664</b><br>
                            Die 8 Instruction-Bits sind wie folgt aufgebaut:<br> <br>
                            <CENTER> X X C C C A A A </CENTER><br>
                            - Die ersten zwei Bits sind don't cares<br> 
                            - Die <EM>Befehle C</EM> sind wie fogt aufgebaut<br>
                            <ul>  
                            <li> 000   Schreibe Input register N</li> 
                            <li> 001   Update DAC Register N</li>
                            <li> 010   Schreibe Inputregister N und Update alle DAC Register</li>
                            <li> 011   Schreibe Inputregister N und Update DAC Register N</li>
                            <li> 100   Power Down DAC (siehe Datenblatt)</li>
                            <li> 101   Reset (siehe Datenblatt)</li>
                            <li> 110   Funktionsregister LDAC (siehe Datenblatt)</li>
                            </ul>
                            - Die <EM>Adresse A</EM> zum Register N sind wie fogt aufgebaut<br>
                            <ul>  
                            <li> 000   DAC A</li> 
                            <li> 001   DAC B</li>
                            <li> 010   DAC C</li>
                            <li> 011   DAC B</li>
                            <li> 111   All DACs</li>
                            </ul>
                            
        \param data1 <b>Datenbits</b><br>
                            ersten 8 Datenbits, bzw. oberen zwei Nibble
        \param data2 <b>Datenbits</b><br>
                            zweiten 8 Datenbits, bzw. unteren zwei Nibble        
     */         
    void sendDAC(int instruction, int data1, int data2); 
    
    /*! Destruktor der Klasse */        
    virtual ~AD5664(){};
    
    /*! Setzt den Chipselect-Pin des Wandlers auf "select" bzw. wählt diesen aus. */
    void SelectCS();
    
    /*! Setzt den Chipselect-Pin des Wandlers auf "deselect" bzw. wählt diesen ab. */
    void DeselectCS();
    
    
    
    /*! Schreibt ein 16-Bit Wert in den eingestellten Kanal. Der Wert wird direkt in das DAC Register geschrieben und der Wert wird direkt ausgegen. */
    /*!
        \param channel <b>Einstellung des Ausgewählten Kanals</b><br>
                            <ul>  
                            <li> A   Beschreibt Kanal A</li> 
                            <li> B   Beschreibt Kanal B</li>
                            <li> C   Beschreibt Kanal C</li>
                            <li> D   Beschreibt Kanal D</li>
                            <li> F   Beschreibt alle DACs </li>
                            </ul>
                            Standartwert, falls ein falscher Kanal angegeben wird ist "F" 
        \param value <b>16-Bit Wert</b><br>  Wird Automatisch in zwei 8 Bit Werte angepasst.  
     */        
    void writeDAC(char channel, int value);    
 
 
 
     /*! Lädt ein 16-Bit Wert in den eingestellten Kanal. Der Wert wird direkt in das Inputregister geschrieben und kann mit der updateDAC-Funktion in das DAC-Register geschrieben werden */
    /*!
        \param channel <b>Einstellung des Ausgewählten Kanals</b><br>
                            <ul>  
                            <li> A   Lädt Kanal A</li> 
                            <li> B   Lädt Kanal B</li>
                            <li> C   Lädt Kanal C</li>
                            <li> D   Lädt Kanal D</li>
                            <li> F   Lädt alle DACs </li>
                            </ul>
                            Standartwert, falls ein falscher Kanal angegeben wird ist "F" 
        \param value <b>16-Bit Wert</b><br>  Wird Automatisch in zwei 8 Bit Werte angepasst.  
     */        
    void loadDAC(char channel, int value); 
    
    
    /*! Schreibt zuvor eingestellten Wert in das DAC Register (Laden mittels loadDAC-Funktion). Dieser Wert ist dann am Ausgang des entsprechenden DAC-Kanals zu messen. */
    /*!
        \param channel <b>Einstellung des Ausgewählten Kanals</b><br>
                            <ul>  
                            <li> A   Beschreibt Kanal A</li> 
                            <li> B   Beschreibt Kanal B</li>
                            <li> C   Beschreibt Kanal C</li>
                            <li> D   Beschreibt Kanal D</li>
                            <li> F   Beschreibt alle DACs </li>
                            </ul>
                            Standartwert, falls ein falscher Kanal angegeben wird ist "F"   
     */        
    void updateDAC(char channel);     
               
    protected:      
        //! SPI Schnittstelle
        SPI spi;    
        //! Chipselect Pin     
        DigitalOut daccs;                                  
   
    };

#endif
