#ifndef ADJDs311_h
#define ADJDs311_h


//Includes

#include "mbed.h"

// ADJD-S311's I2C address, don't change
#define WRITE_ADDRESS   0xE8    // Address for write
#define READ_ADDRESS    0xE9    // Address for read


// ADJD-S311's register list
#define CTRL            0x00
#define CONFIG          0x01
#define CAP_RED         0x06
#define CAP_GREEN       0x07
#define CAP_BLUE        0x08
#define CAP_CLEAR       0x09
#define INT_RED_LO      0xA
#define INT_RED_HI      0xB
#define INT_GREEN_LO    0xC
#define INT_GREEN_HI    0xD
#define INT_BLUE_LO     0xE
#define INT_BLUE_HI     0xF
#define INT_CLEAR_LO    0x10
#define INT_CLEAR_HI    0x11
#define DATA_RED_LO     0x40
#define DATA_RED_HI     0x41
#define DATA_GREEN_LO   0x42
#define DATA_GREEN_HI   0x43
#define DATA_BLUE_LO    0x44
#define DATA_BLUE_HI    0x45
#define DATA_CLEAR_LO   0x46
#define DATA_CLEAR_HI   0x47
#define OFFSET_RED      0x48
#define OFFSET_GREEN    0x49
#define OFFSET_BLUE     0x4A
#define OFFSET_CLEAR    0x4B

/**
 * A structure contain info about the red, green, blue and clear channel
 */
struct RGBC{
    int red;
    int blue;
    int green;
    int clear;
};

/**
 * A ADJD-S311 color sensor class 
 */
class ADJDs311{
 public:

/** 
 * Create a color sensor interface
 *
 * @param sda       Pin connected to sda of color sensor
 * @param scl       Pin connected to scl of color sensor
 * @param led       Pin connected to on board led of the sensor
 */
  ADJDs311(PinName sda, PinName scl, PinName led);
  
  
/**
 * Calibrate the capacitance and integration time slot so that the current
 * readings are as close to 1000 as possible and the difference between RGB
 * readings are as small as possible
 */
  void calibrate();
  
/**
 * Turn the on board LED on/off.
 *
 * @param ledOn     Whether to turn the LED on.
 */
  void ledMode(bool ledOn);

/**
 * Get the current offset stored in offset registers
 *
 * @return          Current offset stored in offset registers
 */
 RGBC getOffset();
 
/**
 * Use the current light condition to set the offset
 *
 * @param useOffset Wether to use the offset
 * @return          The offset set
 */
 RGBC setOffset(bool useOffset = true);
 
 
/**
 * Use the offset registers to automatically subtract offset from the readings
 *
 * @param useOffset Wether to use the offset
 */
  void offsetMode(bool useOffset);

/**
 * Read in the color value from the sensor
 *
 * @return          Structure containing the value of red, green, blue and clear
 */
  RGBC read();
  
/**
 * Get the gain of number of capacitor for each channel, in the range of 0 to
 * 15. Less capacitor will give higher sensitivity.
 *
 * @return          Structure containing the gain of number of capacitor for each 
 *                  channel.
 */
  RGBC getColorCap();
  
/**
 * Get the gain of number of integration time slot, in the range of 0 to 4095.
 * More integration time slot will give higher sensitivity.
 *
 * @return          Structure containing the gain of number of integration time 
 *                  slot for each channel.
 */
  RGBC getColorInt();
  
/**
 * Set the gain of number of capacitor for each channel, in the range of 0 to
 * 15. Less capacitor will give higher sensitivity.
 *
 * @param red       gain value for red
 * @param green     gain value for green
 * @param blue      gain value for blue
 * @param clear     gain value for clear
 */
  void setColorCap(int red, int green, int blue, int clear); 
  
/**
 * Set the gain of number of integration time slot, in the range of 0 to 4095.
 * More integration time slot will give higher sensitivity.
 *
 * @param red       gain value for red
 * @param green     gain value for green
 * @param blue      gain value for blue
 * @param clear     gain value for clear
 */
  void setColorInt(int red, int green, int blue, int clear);   
  
private:

    // fields
    I2C _i2c;
    DigitalOut _led;
    
    RGBC colorCap;
    RGBC colorInt;
    RGBC colorOffset;
    
// private memeber functions
    
/** Write a byte of data to ADJD-S311 register
 *
 * @param data      byte data to write to the register
 * @param regAddr   address of the register to write
 */
    void writeRegister(char data, char regAddr);
   
/** Read a byte of data from ADJD-S311 register
 *
 * @param regAddr   address of the register to write
 * @retrun          byte data read from the register
 */
    char readRegister(char regAddr);

/** Read 2 bytes of data from ADJD-S311 registers and return as an integer
 *
 * @param loRegAddr low register address
 * @return          value from registers as an int
 */
  int readInt(char loRegAddr);
  
/** Write an integer data of 2 bytes to ADJD-S311 registers
 * 
 * @param loRegAddr low register address
 * @param data      integer value to write
 */
   void writeInt(int data, char loRegAddr);  
   
/**
 * Tell the color sensor to perform measurement and store data to the color and
 * clear registers. Must be called before reading
 * color values
 */
  void performMeasurement();
    
/**
 * Calibrate the clear integration time slot so that current reading of the 
 * clear channel will be as close to 1000 as possible     
 */  
  void calibrateClearInt();
  
/**
 * Calibrate the color integration time slots so that the max current reading 
 * of the RGB channels will be as close to 1000 as possible
 */  
  void calibrateColorInt();
  
/**
 * Calibrate the color capacitors so the difference of reading of different
 * channels are as small as possible 
 */  
  void calibrateCapacitors();
};

#endif