#include "mbed.h"
#include "rtos.h"

#include "ChipSelector.h" 
#include "CAT25160.h"
#include "Global.h"

static char trace[400];

CAT25160   CAT25160::sSingleton;

CAT25160::CAT25160()
        : _spi(PTD2, PTD3, PTD1)    // mosi, miso, sclk     // SPI connecteur mbed J2
//        : _spi(PTD6, PTD7, PTD5)  // mosi, miso, sclk     // SPI connecteur mbed J6
        
{
    _initComplete = false;
}

int CAT25160::init()
{  
    if ( _initComplete == true)
    {
        if (ENABLE_TRACE) {sprintf(trace,"Error CAT25160::init already initialized !!!\r\n"); DiplayTrace(trace);}
        return -1;
    }
    
    // Setup the spi for 8 bits data, 
    _spi.format(8,3);
    _spi.frequency(1000000); //1000Khz pas fiable
    //_spi.frequency(500000); //500Khz semble fonctionner
    //_spi.frequency(250000); //250Khz 
    //_spi.frequency(10000);
  
    _initComplete = true;
  
    return 0;
}

uint8_t CAT25160::getStatusRegister(ChipSelector::ChipToSelect cs)
{
    command(EEPROM_CAT25_COMMAND_RDSR, 0, cs);
    uint8_t ret = _spi.write(EEPROM_CAT25_DUMMY_BYTE);
    //if (ENABLE_TRACE) {sprintf(trace,"CAT25160::getStatusRegister: 0x%02x\r\n",ret); DiplayTrace(trace);}

    ChipSelector::get()->setCS(cs,1);
    
    return ret;
}

bool CAT25160::isReady(ChipSelector::ChipToSelect cs)
{
    if ((getStatusRegister(cs) & EEPROM_CAT25_RDY_Msk) == EEPROM_CAT25_RDY_BUSY) 
    {
        if (ENABLE_TRACE) {sprintf(trace,"CAT25160::isReady FALSE\r\n"); DiplayTrace(trace);}
        return false;
    }
    
    //if (ENABLE_TRACE) {sprintf(trace,"CAT25160::isReady True\r\n"); DiplayTrace(trace);}
    return true;
}

void CAT25160::enableWrite(ChipSelector::ChipToSelect cs)
{
    if (ENABLE_TRACE) {sprintf(trace,"EEPROM_CAT25_COMMAND_WREN\r\n"); DiplayTrace(trace);}
    command(EEPROM_CAT25_COMMAND_WREN, 0, cs);
    ChipSelector::get()->setCS(cs,1);
}

void CAT25160::disableWrite(ChipSelector::ChipToSelect cs)
{
    if (ENABLE_TRACE) {sprintf(trace,"EEPROM_CAT25_COMMAND_WRDI\r\n"); DiplayTrace(trace);}
    command(EEPROM_CAT25_COMMAND_WRDI, 0, cs);
    ChipSelector::get()->setCS(cs,1);
}


uint8_t CAT25160::protectAll(ChipSelector::ChipToSelect cs)
{
    if (ENABLE_TRACE) {sprintf(trace,"EEPROM_CAT25_COMMAND_WRSR 0x0c (protectAll)\r\n"); DiplayTrace(trace);}
    command(EEPROM_CAT25_COMMAND_WRSR, 0x00, cs);
    _spi.write(0x0c);
    //if (ENABLE_TRACE) {sprintf(trace,"CAT25160::write 0x0c\r\n"); DiplayTrace(trace);}

    ChipSelector::get()->setCS(cs,1);

    return 0;
}

uint8_t CAT25160::unprotectAll(ChipSelector::ChipToSelect cs)
{
    enableWrite(cs);

    if (ENABLE_TRACE) {sprintf(trace,"EEPROM_CAT25_COMMAND_WRSR 0x00 (unprotectAll)\r\n"); DiplayTrace(trace);}
    command(EEPROM_CAT25_COMMAND_WRSR, 0x00, cs);
    _spi.write(0x00);
    //if (ENABLE_TRACE) {sprintf(trace,"CAT25160::write 0x00\r\n"); DiplayTrace(trace);}
    ChipSelector::get()->setCS(cs,1);

    return 0;
}

uint8_t CAT25160::readByte(const uint32_t address,ChipSelector::ChipToSelect cs)
{
    uint8_t wait_count = 0;
    
    if (address >= EEPROM_CAPACITY_CAT25160) 
        return 0;

    while (!isReady(cs)) 
    {
        wait_count++;
        if (wait_count >= MAX_WAIT_COUNT)
        {       
            if (ENABLE_TRACE) {sprintf(trace,"CAT25160::readByte: MAX_WAIT_COUNT reached !!!\r\n"); DiplayTrace(trace);}
            return 0;
        }
        thread_sleep_for(10);
    }
    
    command(EEPROM_CAT25_COMMAND_READ, address, cs);
    uint8_t ret = _spi.write(EEPROM_CAT25_DUMMY_BYTE);
    //if (ENABLE_TRACE) {sprintf(trace,"CAT25160::readByte: value:0x%02x\r\n",ret); DiplayTrace(trace);}
 
    ChipSelector::get()->setCS(cs,1);

    return ret;
}

uint8_t CAT25160::writeByte(const uint32_t address, const uint8_t byte,ChipSelector::ChipToSelect cs)
{
    uint8_t wait_count = 0;

    if (address >= EEPROM_CAPACITY_CAT25160) 
        return 0;

    while (!isReady(cs)) 
    {
        wait_count++;
        if (wait_count >= MAX_WAIT_COUNT)
        {
            if (ENABLE_TRACE) {sprintf(trace,"CAT25160::readByte: MAX_WAIT_COUNT reached !!!\r\n"); DiplayTrace(trace);}
            return 0;
        }
        thread_sleep_for(10);
    }

    enableWrite(cs);
    command(EEPROM_CAT25_COMMAND_WRITE, address, cs);
    _spi.write(byte);
    //if (ENABLE_TRACE) {sprintf(trace,"CAT25160::writeByte: value:0x%02x\r\n",byte); DiplayTrace(trace);}

    ChipSelector::get()->setCS(cs,1);

    return 1;
}

uint16_t CAT25160::readBlock(const uint32_t address, const uint16_t length, void *buffer,ChipSelector::ChipToSelect cs)
{
    uint8_t wait_count = 0;

    if (length == 0 || (length + address) > EEPROM_CAPACITY_CAT25160) 
        return 0;

    while (!isReady(cs)) 
    {
        wait_count++;
        if (wait_count >= MAX_WAIT_COUNT)
        {
            if (ENABLE_TRACE) {sprintf(trace,"CAT25160::readByte: MAX_WAIT_COUNT reached !!!\r\n"); DiplayTrace(trace);}
            return 0;
        }
        thread_sleep_for(10);
    }

    char dummy = 0xff;
    
    command(EEPROM_CAT25_COMMAND_READ, address,cs);
    _spi.write(&dummy, 1, (char*)buffer, length);

    ChipSelector::get()->setCS(cs,1);

    return length;
}

uint16_t CAT25160::writeBlock(uint32_t address, const uint16_t length, void * buffer,ChipSelector::ChipToSelect cs)
{
    uint8_t *buf = reinterpret_cast<uint8_t *>(buffer);
    uint16_t len = length;

    if (length == 0 || (length + address) > EEPROM_CAPACITY_CAT25160) 
        return 0;


    uint16_t remainderFirstPage = (EEPROM_PAGE_SIZE_CAT25160 - (address % EEPROM_PAGE_SIZE_CAT25160));
    if (len < remainderFirstPage) 
        remainderFirstPage = len;


    if (remainderFirstPage) 
    {
        writePage(address, remainderFirstPage, buf, cs);
        buf += remainderFirstPage;
        len -= remainderFirstPage;
        address += remainderFirstPage;
    }

    while (len > EEPROM_PAGE_SIZE_CAT25160) 
    {
        writePage(address, EEPROM_PAGE_SIZE_CAT25160, buf, cs);
        buf += EEPROM_PAGE_SIZE_CAT25160;
        len -= EEPROM_PAGE_SIZE_CAT25160;
        address += EEPROM_PAGE_SIZE_CAT25160;
    }
    writePage(address, len, buf,cs);

    return length;
}

uint16_t CAT25160::writePage(const uint32_t address, const uint16_t length, void * buffer,ChipSelector::ChipToSelect cs)
{
    uint8_t wait_count = 0;
    uint8_t *buf = reinterpret_cast<uint8_t *>(buffer);
    uint16_t len = length;

    if (address >= EEPROM_CAPACITY_CAT25160 || length == 0 || length > (EEPROM_PAGE_SIZE_CAT25160 - (address % EEPROM_PAGE_SIZE_CAT25160))) 
        return 0;

    while (!isReady(cs)) 
    {
        wait_count++;
        if (wait_count >= MAX_WAIT_COUNT)
        {
            if (ENABLE_TRACE) {sprintf(trace,"CAT25160::readByte: MAX_WAIT_COUNT reached !!!\r\n"); DiplayTrace(trace);}
            return 0;
        }
        thread_sleep_for(10);
    }

    enableWrite(cs);
    command(EEPROM_CAT25_COMMAND_WRITE, address, cs);
    
    while (len--) 
    {
        _spi.write(*buf);
        buf++;
    }
    ChipSelector::get()->setCS(cs,1);

    return length;
}

void CAT25160::command(uint8_t command, const uint32_t address,ChipSelector::ChipToSelect cs)
{
    ChipSelector::get()->setCS(cs,0);
    thread_sleep_for(1);
    
    _spi.write(command);
    //if (ENABLE_TRACE) {sprintf(trace,"CAT25160::command: 0x%02x\r\n",command); DiplayTrace(trace);}

    if (command == EEPROM_CAT25_COMMAND_READ || command == EEPROM_CAT25_COMMAND_WRITE) 
    {
        // if (ENABLE_TRACE) {sprintf(trace,"CAT25160:: command adr:0x%08x\r\n",address); DiplayTrace(trace);}
        // envoi l'adresse
        _spi.write((uint8_t)((address >> 8) & 0xFF));
        _spi.write((uint8_t)(address & 0xFF));
    }
}

// Commandes de haut niveaux
int16_t CAT25160::checkAccess(ChipSelector::ChipToSelect cs)
{
    if (_initComplete == false)
        init();   

    // On part du principe que l'adresse 0x0 contient 0xaa
    // si on trouve 0xaa on a bien le chipset
    // si on ne le trouve pas, on tente une ecriture puis relecture
    if (readByte(0x0,cs) == 0xaa)   
        return 0;

    writeByte(0x0,0xaa,cs);
    if (readByte(0x0,cs) == 0xaa)
        return 0;
    
    return -1;
}

uint8_t * CAT25160::readConfig(ChipSelector::ChipToSelect cs, uint16_t *crc_out)
{
    // on partitionne la EEPROM de la manière suivante :
    // 0x00 => octet de presence (0xaa)
    // 0x01 => octet entree valide (0xaa si ok)
    // 0x02-0x03 : longueur des données
    // 0x04-0x05 : checksum données
    // 0x06 : début des données

    if (_initComplete == false)
        init(); 

    if (checkAccess(cs) == -1)
    {
        printf("EEPROM not found !!!\r\n");
        return NULL;
    }
    
    if (readByte(0x01,cs) != 0xaa)
    {
        printf("No EEPROM valid entry !!!\r\n");
        return NULL;
    }
    
    uint16_t config_len = 0;
    uint16_t config_crc = 0;
    
    readBlock(0x02,2,&config_len,cs);
    readBlock(0x04,2,&config_crc,cs);

    //if (ENABLE_TRACE) {sprintf(trace,"Config len : 0x%04x\r\n",config_len); DiplayTrace(trace);}
    //if (ENABLE_TRACE) {sprintf(trace,"Config CRC : 0x%04x\r\n",config_crc); DiplayTrace(trace);}
    
    uint8_t *buffer = (uint8_t *)malloc( config_len );
    if (buffer == NULL)
    {
        if (ENABLE_TRACE) {sprintf(trace,"Config malloc error (try to allocate %d !!!\r\n",config_len); DiplayTrace(trace);}
        return NULL;
    }

    readBlock(0x06, config_len, buffer, cs);
    
    if (ENABLE_TRACE) {sprintf(trace,"----------EEPROM config-----------\r\n%s",buffer); DiplayTrace(trace);}
    if (ENABLE_TRACE) {sprintf(trace,"\r\n----------------------------------\r\n"); DiplayTrace(trace);}
    
    uint16_t compute_config_crc = 0;
    for (int i=0; i < config_len; i++)
        compute_config_crc += buffer[i];
    
    if (compute_config_crc != config_crc)
    {
        if (ENABLE_TRACE) {sprintf(trace,"CRC Error. Computed config crc is 0x%04x and EEPROM CRC is 0x%04x\r\n",compute_config_crc, config_crc); DiplayTrace(trace);}
        free(buffer);
        return NULL;
    }
    
    if (crc_out != NULL)
        *crc_out = compute_config_crc;

    return buffer;   
}

int8_t CAT25160::writeConfigCheck(uint8_t *buffer, uint16_t buffer_len,ChipSelector::ChipToSelect cs)
{
    unprotectAll(cs);
    
    uint16_t crc_buffer=0;
    if (writeConfig(buffer,buffer_len,cs,&crc_buffer) != 0)
        return -1;
    
    uint16_t crc_datas_eeprom=0;
    uint8_t * tmp_buffer = NULL;
    tmp_buffer = readConfig(cs, &crc_datas_eeprom);
    if (tmp_buffer == NULL)
        return -1;

    free(buffer);
    
    if (  crc_buffer !=  crc_datas_eeprom )
    {
         if (ENABLE_TRACE) {sprintf(trace,"writeConfigCheck : Error on CRC 0x%04x != 0x%04x !!!\r\n",crc_buffer,crc_datas_eeprom); DiplayTrace(trace);}
         return -1;
    }
    return 0;
}

int8_t CAT25160::writeConfig(uint8_t *buffer, uint16_t buffer_len,ChipSelector::ChipToSelect cs, uint16_t *crc_out)
{
    if (_initComplete == false)
        init();   

    if (checkAccess(cs) == -1)
    {
        if (ENABLE_TRACE) {sprintf(trace,"EEPROM not found !!!\r\n"); DiplayTrace(trace);}
        return -1;
    }

    uint16_t config_crc = 0;
    for (int i=0; i < buffer_len; i++)
        config_crc += buffer[i];
    
    *crc_out = config_crc;
    if (ENABLE_TRACE) {sprintf(trace,"config crc is 0x%04x\r\n",config_crc); DiplayTrace(trace);}
    
    // positionne l'entree en dirty
    writeByte(0x01,0x00,cs);

    // write len
    writeByte(0x02,(buffer_len ) & 0xff,cs);
    writeByte(0x03,(buffer_len >> 8) & 0xff,cs);

    // write crc
    writeByte(0x04,(config_crc  ) & 0xff,cs);
    writeByte(0x05,(config_crc >> 8) & 0xff,cs);
    
    // write datas
    writeBlock(0x06, buffer_len, buffer, cs);

    // positionne l'entree en OK
    writeByte(0x01,0xaa,cs);
    
    return 0;
}

