//
//  Picaso_4DGL-32PTU is a class to drive 4D Systems TFT touch screens with PICASO processor
//  Tested with NUCLEO L152RE development board
//  Copyright (C) <2016> Rihards Balass <rihards.balass@gmail.com>
//
// Picaso_4DGL-32PTU is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Picaso_4DGL-32PTU is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You can see GNU General Public License at <http://www.gnu.org/licenses/>.
//

#include "mbed.h"
#include "Picaso_4DGL-32PTU.h"

//**************************************************************************
// The Clear Screen command clears the screen using the current background color.
// This command brings some of the settings back to default; such as,
//  Transparency turned OFF
//  Outline color set to BLACK
//  Opacity set to OPAQUE
//  Pen set to OUTLINE
//  Line patterns set to OFF
//  Right text margin set to full width
//  Text magnifications set to 1
//  All origins set to 0:0
// The alternative to maintain settings and clear screen is 
// to draw a filled rectangle with the required background color.
//**************************************************************************
void PICASO_4DGL :: cls() {  // clear screen

    char command[2] = "";
    
    command[0] = (CLS >> (8*1)) & 0xff;
    command[1] = (CLS >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 2);
    getResponse(1);
}

//**************************************************************************
// The Change Color command changes all oldColor pixels to newColor 
// within the clipping window area.
//**************************************************************************
void PICASO_4DGL :: changeColor(short oldColor, short newColor) {
    
    char command[6] = "";
    
    command[0] = (CHANGE_COLOR >> (8*1)) & 0xff;
    command[1] = (CHANGE_COLOR >> (8*0)) & 0xff;
    command[2] = (oldColor >> (8*1)) & 0xff;
    command[3] = (oldColor >> (8*0)) & 0xff;
    command[4] = (newColor >> (8*1)) & 0xff;
    command[5] = (newColor >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 6);
    getResponse(1);
}

//**************************************************************************
// The Draw Circle command draws a circle with centre point x, y 
// with radius r using the specified color.
//**************************************************************************
void PICASO_4DGL :: drawCircle(short x, short y, short r, short color) {
    
    char command[10] = "";
    
    command[0] = (DRAW_CIRCLE >> (8*1)) & 0xff;
    command[1] = (DRAW_CIRCLE >> (8*0)) & 0xff;
    command[2] = (x >> (8*1)) & 0xff;
    command[3] = (x >> (8*0)) & 0xff;
    command[4] = (y >> (8*1)) & 0xff;
    command[5] = (y >> (8*0)) & 0xff;
    command[6] = (r >> (8*1)) & 0xff;
    command[7] = (r >> (8*0)) & 0xff;
    command[8] = (color >> (8*1)) & 0xff;
    command[9] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 10);
    getResponse(1);
}

//**************************************************************************
// The Draw Circle command draws a solid circle with centre point x1, y1
// with radius r using the specified color.
// The outline color can be specified with the “Outline Color” command.
// If “Outline Color” is set to 0, no outline is drawn.
//**************************************************************************
void PICASO_4DGL :: drawFilledCircle(short x, short y, short r, short color) {
    
    char command[10] = "";
    
    command[0] = (CIRCLE_FILLED >> (8*1)) & 0xff;
    command[1] = (CIRCLE_FILLED >> (8*0)) & 0xff;
    command[2] = (x >> (8*1)) & 0xff;
    command[3] = (x >> (8*0)) & 0xff;
    command[4] = (y >> (8*1)) & 0xff;
    command[5] = (y >> (8*0)) & 0xff;
    command[6] = (r >> (8*1)) & 0xff;
    command[7] = (r >> (8*0)) & 0xff;
    command[8] = (color >> (8*1)) & 0xff;
    command[9] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 10);
    getResponse(1);
}

//**************************************************************************
// The Draw Line command draws a line from x1, y1 to x2, y2 
// using the specified color.
//**************************************************************************
void PICASO_4DGL :: drawLine(short x1, short y1, short x2, short y2, short color) {
    
    char command[12] = "";
    
    command[0] = (DRAW_LINE >> (8*1)) & 0xff;
    command[1] = (DRAW_LINE >> (8*0)) & 0xff;
    command[2] = (x1 >> (8*1)) & 0xff;
    command[3] = (x1 >> (8*0)) & 0xff;
    command[4] = (y1 >> (8*1)) & 0xff;
    command[5] = (y1 >> (8*0)) & 0xff;
    command[6] = (x2 >> (8*1)) & 0xff;
    command[7] = (x2 >> (8*0)) & 0xff;
    command[8] = (y2 >> (8*1)) & 0xff;
    command[9] = (y2 >> (8*0)) & 0xff;
    command[10] = (color >> (8*1)) & 0xff;
    command[11] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 12);
    getResponse(1);
}

//**************************************************************************
// The Draw Rectangle command draws a rectangle from x1, y1 to x2, y2
// using the specified color. 
// The line may be tessellated with the “Line Pattern” command.
//**************************************************************************
void PICASO_4DGL :: drawRectangle(short x1, short y1, short x2, short y2, short color) {
    
    char command[12] = "";
    
    command[0] = (DRAW_RECTANGLE >> (8*1)) & 0xff;
    command[1] = (DRAW_RECTANGLE >> (8*0)) & 0xff;
    command[2] = (x1 >> (8*1)) & 0xff;
    command[3] = (x1 >> (8*0)) & 0xff;
    command[4] = (y1 >> (8*1)) & 0xff;
    command[5] = (y1 >> (8*0)) & 0xff;
    command[6] = (x2 >> (8*1)) & 0xff;
    command[7] = (x2 >> (8*0)) & 0xff;
    command[8] = (y2 >> (8*1)) & 0xff;
    command[9] = (y2 >> (8*0)) & 0xff;
    command[10] = (color >> (8*1)) & 0xff;
    command[11] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 12);
    getResponse(1);
}

//**************************************************************************
// The Draw Filled Rectangle command draws a solid rectangle from x1, y1 to x2, y2
// using the specified color. 
// The line may be tessellated with the “Line Pattern” command.
// The outline color can be specified with the “Outline Color” command.
// If “Outline Color” is set to 0, no outline is drawn.
//**************************************************************************
void PICASO_4DGL :: drawFilledRectangle(short x1, short y1, short x2, short y2, short color) {
    
    char command[12] = "";
    
    command[0] = (RECTANGLE_FILLED >> (8*1)) & 0xff;
    command[1] = (RECTANGLE_FILLED >> (8*0)) & 0xff;
    command[2] = (x1 >> (8*1)) & 0xff;
    command[3] = (x1 >> (8*0)) & 0xff;
    command[4] = (y1 >> (8*1)) & 0xff;
    command[5] = (y1 >> (8*0)) & 0xff;
    command[6] = (x2 >> (8*1)) & 0xff;
    command[7] = (x2 >> (8*0)) & 0xff;
    command[8] = (y2 >> (8*1)) & 0xff;
    command[9] = (y2 >> (8*0)) & 0xff;
    command[10] = (color >> (8*1)) & 0xff;
    command[11] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 12);
    getResponse(1);
}


//**************************************************************************
// The Draw Polyline command plots lines between points specified by a pair of arrays
// using the specified color. 
// The lines may be tessellated with the “Line Pattern” command.
// The “Draw Polyline” command can be used to create complex raster graphics
// by loading the arrays from serial input or from MEDIA with very little code requirement.
// 
// n - Specifies the number of elements in the x and y arrays specifying the vertices for the polyline.
// vx, vy - Specifies the array of elements for the x/y coordinates of the vertices.
// Vx1, vx2, …, vxN, vy1, vy2, …, vyN
//**************************************************************************
void PICASO_4DGL :: drawPolyline(short n, short *vx, short *vy, short color) {
    
    int size = 6 + (n*4);
    int i, j = 4;
    char *command;
    command = (char *)malloc(sizeof(char) * size);
    for(i = 0; i < size; i++) command[i] = 0;
    
    command[0] = (DRAW_POLYLINE >> (8*1)) & 0xff;
    command[1] = (DRAW_POLYLINE >> (8*0)) & 0xff;
    command[2] = (n >> (8*1)) & 0xff;
    command[3] = (n >> (8*0)) & 0xff;
    for (i = 0; i < n; i++) {
        command[j] = (vx[i] >> (8*1)) & 0xff;
        j++;
        command[j] = (vx[i] >> (8*0)) & 0xff;
        j++;
    }
    for (i = 0; i < n; i++) {
        command[j] = (vy[i] >> (8*1)) & 0xff;
        j++;
        command[j] = (vy[i] >> (8*0)) & 0xff;
        j++;
    }
    command[j] = (color >> (8*1)) & 0xff;
    command[j+1] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, size);
    getResponse(1);
    free(command);
}

//**************************************************************************
// The Draw Polygon command plots lines between points specified by a pair of arrays
// using the specified color. 
// The last point is drawn back to the first point, completing the polygon.
// The lines may be tessellated with the “Line Pattern” command.
// The Draw Polygon command can be used to create complex raster graphics
// by loading the arrays from serial input or from MEDIA with very little code requirement.
// 
// n - Specifies the number of elements in the x and y arrays specifying the vertices for the polygon.
// vx, vy - Specifies the array of elements for the x/y coordinates of the vertices.
// Vx1, vx2, …, vxN, vy1, vy2, …, vyN
//**************************************************************************
void PICASO_4DGL :: drawPolygon(short n, short *vx, short *vy, short color) {
    
    int size = 6 + (n*4);
    int i, j = 4;
    char *command;
    command = (char *)malloc(sizeof(char) * size);
    for(i = 0; i < size; i++) command[i] = 0;
    
    command[0] = (DRAW_POLYGON >> (8*1)) & 0xff;
    command[1] = (DRAW_POLYGON >> (8*0)) & 0xff;
    command[2] = (n >> (8*1)) & 0xff;
    command[3] = (n >> (8*0)) & 0xff;
    for (i = 0; i < n; i++) {
        command[j] = (vx[i] >> (8*1)) & 0xff;
        j++;
        command[j] = (vx[i] >> (8*0)) & 0xff;
        j++;
    }
    for (i = 0; i < n; i++) {
        command[j] = (vy[i] >> (8*1)) & 0xff;
        j++;
        command[j] = (vy[i] >> (8*0)) & 0xff;
        j++;
    }
    command[j] = (color >> (8*1)) & 0xff;
    command[j+1] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, size);
    getResponse(1);
    free(command);
}

//**************************************************************************
// The Draw Filled Polygon command draws a solid Polygon between specified vertices:
// x1, y1 x2, y2, .... , xn, yn using the specified color.
// The last point is drawn back to the first point, completing the polygon.
// Vertices must be a minimum of 3 and can be specified in any fashion.
// 
// n - Specifies the number of elements in the x and y arrays specifying the vertices for the polygon.
// vx, vy - Specifies the array of elements for the x/y coordinates of the vertices.
// Vx1, vx2, …, vxN, vy1, vy2, …, vyN
//**************************************************************************
void PICASO_4DGL :: drawFilledPolygon(short n, short *vx, short *vy, short color) {
    
    if (n >= 3) {
        int size = 6 + (n*4);
        int i, j = 4;
        char *command;
        command = (char *)malloc(sizeof(char) * size);
        for(i = 0; i < size; i++) command[i] = 0;
        
        command[0] = (POLYGON_FILLED >> (8*1)) & 0xff;
        command[1] = (POLYGON_FILLED >> (8*0)) & 0xff;
        command[2] = (n >> (8*1)) & 0xff;
        command[3] = (n >> (8*0)) & 0xff;
        for (i = 0; i < n; i++) {
            command[j] = (vx[i] >> (8*1)) & 0xff;
            j++;
            command[j] = (vx[i] >> (8*0)) & 0xff;
            j++;
        }
        for (i = 0; i < n; i++) {
            command[j] = (vy[i] >> (8*1)) & 0xff;
            j++;
            command[j] = (vy[i] >> (8*0)) & 0xff;
            j++;
        }
        command[j] = (color >> (8*1)) & 0xff;
        command[j+1] = (color >> (8*0)) & 0xff;
        
        writeCOMMAND(command, size);
        getResponse(1);
        free(command);
    }
    else pc.printf("\n\r    ERROR_POLYGON_FILLED: There has to be at least 3 vertices!\n\r");
}

//**************************************************************************
// The Draw Triangle command draws a triangle outline between vertices
// x1,y1 , x2,y2 and x3,y3 using the specified color.
// The line may be tessellated with the “Line Pattern” command.
//**************************************************************************
void PICASO_4DGL :: drawTriangle(short x1, short y1, short x2, short y2, short x3, short y3, short color) {
    
    char command[16] = "";
    
    command[0] = (DRAW_TRIANGLE >> (8*1)) & 0xff;
    command[1] = (DRAW_TRIANGLE >> (8*0)) & 0xff;
    command[2] = (x1 >> (8*1)) & 0xff;
    command[3] = (x1 >> (8*0)) & 0xff;
    command[4] = (y1 >> (8*1)) & 0xff;
    command[5] = (y1 >> (8*0)) & 0xff;
    command[6] = (x2 >> (8*1)) & 0xff;
    command[7] = (x2 >> (8*0)) & 0xff;
    command[8] = (y2 >> (8*1)) & 0xff;
    command[9] = (y2 >> (8*0)) & 0xff;
    command[10] = (x3 >> (8*1)) & 0xff;
    command[11] = (x3 >> (8*0)) & 0xff;
    command[12] = (y3 >> (8*1)) & 0xff;
    command[13] = (y3 >> (8*0)) & 0xff;
    command[14] = (color >> (8*1)) & 0xff;
    command[15] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 16);
    getResponse(1);
}

//**************************************************************************
// The Draw Filled Triangle command draws a solid triangle between vertices
// x1, y1, x2, y2 and x3, y3 using the specified color.
//**************************************************************************
void PICASO_4DGL :: drawFilledTriangle(short x1, short y1, short x2, short y2, short x3, short y3, short color) {
    
    char command[16] = "";
    
    command[0] = (TRIANGLE_FILLED >> (8*1)) & 0xff;
    command[1] = (TRIANGLE_FILLED >> (8*0)) & 0xff;
    command[2] = (x1 >> (8*1)) & 0xff;
    command[3] = (x1 >> (8*0)) & 0xff;
    command[4] = (y1 >> (8*1)) & 0xff;
    command[5] = (y1 >> (8*0)) & 0xff;
    command[6] = (x2 >> (8*1)) & 0xff;
    command[7] = (x2 >> (8*0)) & 0xff;
    command[8] = (y2 >> (8*1)) & 0xff;
    command[9] = (y2 >> (8*0)) & 0xff;
    command[10] = (x3 >> (8*1)) & 0xff;
    command[11] = (x3 >> (8*0)) & 0xff;
    command[12] = (y3 >> (8*1)) & 0xff;
    command[13] = (y3 >> (8*0)) & 0xff;
    command[14] = (color >> (8*1)) & 0xff;
    command[15] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 16);
    getResponse(1);
}

//**************************************************************************
// The Calculate Orbit command calculates the x, y coordinates of a distant point relative
// to the current origin, where the only known parameters are the angle and the distance
// from the current origin. The new coordinates are calculated and then placed in the
// destination variables Xdest and Ydest.
//**************************************************************************
void PICASO_4DGL :: calculateOrbit(short angle, short distance) {
    
    char command[6] = "";
    
    command[0] = (CALCULATE_ORBIT >> (8*1)) & 0xff;
    command[1] = (CALCULATE_ORBIT >> (8*0)) & 0xff;
    command[2] = (angle >> (8*1)) & 0xff;
    command[3] = (angle >> (8*0)) & 0xff;
    command[4] = (distance >> (8*1)) & 0xff;
    command[5] = (distance >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 6);
    calculateOrbitResponse();
}

//**************************************************************************
// The Put Pixel command draws a pixel at position x, y using the specified color.
//**************************************************************************
void PICASO_4DGL :: putPixel(short x, short y, short color) {
    
    char command[8] = "";
    
    command[0] = (PUT_PIXEL >> (8*1)) & 0xff;
    command[1] = (PUT_PIXEL >> (8*0)) & 0xff;
    command[2] = (x >> (8*1)) & 0xff;
    command[3] = (x >> (8*0)) & 0xff;
    command[4] = (y >> (8*1)) & 0xff;
    command[5] = (y >> (8*0)) & 0xff;
    command[6] = (color >> (8*1)) & 0xff;
    command[7] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 8);
    getResponse(1);
}

//**************************************************************************
// The Move Origin command moves the origin to a new position, which is suitable for
// specifying the location for both graphics and text.
//**************************************************************************
void PICASO_4DGL :: moveOrigin(short x, short y) {
    
    char command[6] = "";
    
    command[0] = (MOVE_ORIGIN >> (8*1)) & 0xff;
    command[1] = (MOVE_ORIGIN >> (8*0)) & 0xff;
    command[2] = (x >> (8*1)) & 0xff;
    command[3] = (x >> (8*0)) & 0xff;
    command[4] = (y >> (8*1)) & 0xff;
    command[5] = (y >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 6);
    getResponse(1);
}

//**************************************************************************
// The Draw Line & Move Origin command draws a line from the current origin 
// to a new position. The Origin is then set to the new position. 
// The line is drawn using the current object colour, using the 
// “Set Graphics Parameters” – “Object Colour” command. The line may be 
// tessellated with the “Line Pattern” command.
// 
// Note: this command is mostly useful with the “Calculate Orbit” command, 
// and usually the “Draw Line” command would be used
//**************************************************************************
void PICASO_4DGL :: lineTo(short x, short y) {
    
    char command[6] = "";
    
    command[0] = (LINE_TO >> (8*1)) & 0xff;
    command[1] = (LINE_TO >> (8*0)) & 0xff;
    command[2] = (x >> (8*1)) & 0xff;
    command[3] = (x >> (8*0)) & 0xff;
    command[4] = (y >> (8*1)) & 0xff;
    command[5] = (y >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 6);
    getResponse(1);
}

//**************************************************************************
// The Set Clip Window command specifies a clipping window region on the screen such
// that any objects and text placed onto the screen will be clipped and displayed only
// within that region. For the clipping window to take effect, the clipping setting must be
// enabled separately using the “Clipping” command
//**************************************************************************
void PICASO_4DGL :: setClipWindow(short x1, short y1, short x2, short y2) {
    
    char command[10] = "";
    
    command[0] = (SET_CLIP_WINDOW >> (8*1)) & 0xff;
    command[1] = (SET_CLIP_WINDOW >> (8*0)) & 0xff;
    command[2] = (x1 >> (8*1)) & 0xff;
    command[3] = (x1 >> (8*0)) & 0xff;
    command[4] = (y1 >> (8*1)) & 0xff;
    command[5] = (y1 >> (8*0)) & 0xff;
    command[6] = (x2 >> (8*1)) & 0xff;
    command[7] = (x2 >> (8*0)) & 0xff;
    command[8] = (y2 >> (8*1)) & 0xff;
    command[9] = (y2 >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 10);
    getResponse(1);
}

//**************************************************************************
// The Clipping command Enables or Disables the ability for Clipping to be used. 
// The clipping points are set with “Set Clip Window” and must be set first.
//**************************************************************************
void PICASO_4DGL :: clipping(short value) {
    
    char command[4] = "";
    
    command[0] = (CLIPPING >> (8*1)) & 0xff;
    command[1] = (CLIPPING >> (8*0)) & 0xff;
    command[2] = (value >> (8*1)) & 0xff;
    command[3] = (value >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 4);
    getResponse(1);
}

//**************************************************************************
// The Extend Clip Region command forces the clip region to the extent 
// of the last text that was printed, or the last image that was shown.
//**************************************************************************
void PICASO_4DGL :: extendClipRegion() {
    
    char command[2] = "";
    
    command[0] = (EXTEND_CLIP >> (8*1)) & 0xff;
    command[1] = (EXTEND_CLIP >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 2);
    getResponse(1);
}

//**************************************************************************
// The Draw Ellipse command plots a colored Ellipse on the screen at centre
// x, y with x-radius = xrad and y-radius = yrad.
//**************************************************************************
void PICASO_4DGL :: drawElipse(short x, short y, short xRad, short yRad, short color) {
    
    char command[12] = "";
    
    command[0] = (DRAW_ELIPSE >> (8*1)) & 0xff;
    command[1] = (DRAW_ELIPSE >> (8*0)) & 0xff;
    command[2] = (x >> (8*1)) & 0xff;
    command[3] = (x >> (8*0)) & 0xff;
    command[4] = (y >> (8*1)) & 0xff;
    command[5] = (y >> (8*0)) & 0xff;
    command[6] = (xRad >> (8*1)) & 0xff;
    command[7] = (xRad >> (8*0)) & 0xff;
    command[8] = (yRad >> (8*1)) & 0xff;
    command[9] = (yRad >> (8*0)) & 0xff;
    command[10] = (color >> (8*1)) & 0xff;
    command[11] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 12);
    getResponse(1);
}

//**************************************************************************
// The Draw Filled Ellipse command plots a solid colored Ellipse on the screen at centre
// x, y with x-radius = xrad and y-radius = yrad.
//**************************************************************************
void PICASO_4DGL :: drawFilledElipse(short x, short y, short xRad, short yRad, short color) {
    
    char command[12] = "";
    
    command[0] = (ELIPSE_FILLED >> (8*1)) & 0xff;
    command[1] = (ELIPSE_FILLED >> (8*0)) & 0xff;
    command[2] = (x >> (8*1)) & 0xff;
    command[3] = (x >> (8*0)) & 0xff;
    command[4] = (y >> (8*1)) & 0xff;
    command[5] = (y >> (8*0)) & 0xff;
    command[6] = (xRad >> (8*1)) & 0xff;
    command[7] = (xRad >> (8*0)) & 0xff;
    command[8] = (yRad >> (8*1)) & 0xff;
    command[9] = (yRad >> (8*0)) & 0xff;
    command[10] = (color >> (8*1)) & 0xff;
    command[11] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 12);
    getResponse(1);
}

//**************************************************************************
// The Draw Button command draws a 3 dimensional Text Button at screen location 
// defined by x, y parameters (top left corner). The size of the button depends
// on the font, width, height and length of the text. The button can contain 
// multiple lines of text by having the \n character embedded in the string 
// for the end of line marker. In this case, the widest text in the string sets 
// the overall width, and the height of the button is set by the number of 
// text lines. In the case of multiple lines, each line is left justified. 
// If you wish to centre or right justify the text, you will need to prepare 
// the text string according to your requirements.
// 
// state - Appearance of button, 0 = Button depressed; 1 = Button raised.
// x, y - Specifies the top left corner position of the button on the screen.
//**************************************************************************
void PICASO_4DGL :: drawButton(short state, short x, short y, short btnColor, short txtColor, short font, short txtW, short txtH, char *txt) {
    
    int size = (19 + (strlen(txt)));
    int i, j = 18;
    char *command;
    command = (char *)malloc(sizeof(char) * size);
    for(i = 0; i < size; i++) command[i] = 0;
    
    command[0] = (DRAW_BUTTON >> (8*1)) & 0xff;
    command[1] = (DRAW_BUTTON >> (8*0)) & 0xff;
    command[2] = (state >> (8*1)) & 0xff;
    command[3] = (state >> (8*0)) & 0xff;
    command[4] = (x >> (8*1)) & 0xff;
    command[5] = (x >> (8*0)) & 0xff;
    command[6] = (y >> (8*1)) & 0xff;
    command[7] = (y >> (8*0)) & 0xff;
    command[8] = (btnColor >> (8*1)) & 0xff;
    command[9] = (btnColor >> (8*0)) & 0xff;
    command[10] = (txtColor >> (8*1)) & 0xff;
    command[11] = (txtColor >> (8*0)) & 0xff;
    command[12] = (font >> (8*1)) & 0xff;
    command[13] = (font >> (8*0)) & 0xff;
    command[14] = (txtW >> (8*1)) & 0xff;
    command[15] = (txtW >> (8*0)) & 0xff;
    command[16] = (txtH >> (8*1)) & 0xff;
    command[17] = (txtH >> (8*0)) & 0xff;
    for (i = 0; i < strlen(txt); i++) {
        command[j] = txt[i];
        j++;
    }
    command[j] = 0;
    
    writeCOMMAND(command, size);
    getResponse(1);
    free(command);
}

//**************************************************************************
// The Draw Panel command draws a 3 dimensional rectangular panel at a screen 
// location defined by x, y parameters (top left corner). The size of the panel 
// is set with the width and height parameters. The colour is defined by colour. 
// The state parameter determines the appearance of the panel, 0 = recessed, 1 = raised.
//**************************************************************************
void PICASO_4DGL :: drawPanel(short state, short x, short y, short width, short height, short color) {
    
    char command[14] = "";
    
    command[0] = (DRAW_PANEL >> (8*1)) & 0xff;
    command[1] = (DRAW_PANEL >> (8*0)) & 0xff;
    command[2] = (state >> (8*1)) & 0xff;
    command[3] = (state >> (8*0)) & 0xff;
    command[4] = (x >> (8*1)) & 0xff;
    command[5] = (x >> (8*0)) & 0xff;
    command[6] = (y >> (8*1)) & 0xff;
    command[7] = (y >> (8*0)) & 0xff;
    command[8] = (width >> (8*1)) & 0xff;
    command[9] = (width >> (8*0)) & 0xff;
    command[10] = (height >> (8*1)) & 0xff;
    command[11] = (height >> (8*0)) & 0xff;
    command[12] = (color >> (8*1)) & 0xff;
    command[13] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 14);
    getResponse(1);
}

//**************************************************************************
// The Draw Slider command draws a vertical or horizontal slider bar on the screen.
// The Draw Slider command has several different modes of operation.
// In order to minimise the amount of graphics functions we need, all modes of operation
// are selected naturally depending on the parameter values.
// Selection rules:
// 1a) if x2-x1 > y2-y1 slider is assumed to be horizontal 
//     (ie: if width > height, slider is horizontal)
// 1b) if x2-x1 <= y2-y1 slider is assumed to be vertical 
//     (ie: if height <= width, slider is horizontal)
// 2a) If value is positive, thumb is set to the position that is the proportion 
//     of value to the scale parameter.(used to set the control to the actual value of a variable)
// 2b) If value is negative, thumb is driven to the graphics position set by the 
//     ABSolute of value. (used to set thumb to its actual graphical position (usually by touch screen)
// 3) The thumb colour is determine by the “Set Graphics Parameters” – 
//    “Object Colour” command, however, if the current object colour is BLACK, 
//    a darkened shade of the colour parameter is used for the thumb .
//
// mode - mode = 0 : Slider Indented, mode = 1 : Slider Raised, mode 2, Slider Hidden (background colour).
// Scale - scale = n : sets the full scale range of the slider for the thumb from 0 to n.
// Value - If value positive, sets the relative position of the thumb on the slider bar, else set thumb to ABS position of the negative number.
//**************************************************************************
void PICASO_4DGL :: drawSlider(short state, short x1, short y1, short x2, short y2, short color, short scale, short value) {
    
    char command[18] = "";
    
    command[0] = (DRAW_SLIDER >> (8*1)) & 0xff;
    command[1] = (DRAW_SLIDER >> (8*0)) & 0xff;
    command[2] = (state >> (8*1)) & 0xff;
    command[3] = (state >> (8*0)) & 0xff;
    command[4] = (x1 >> (8*1)) & 0xff;
    command[5] = (x1 >> (8*0)) & 0xff;
    command[6] = (y1 >> (8*1)) & 0xff;
    command[7] = (y1 >> (8*0)) & 0xff;
    command[8] = (x2 >> (8*1)) & 0xff;
    command[9] = (x2 >> (8*0)) & 0xff;
    command[10] = (y2 >> (8*1)) & 0xff;
    command[11] = (y2 >> (8*0)) & 0xff;
    command[12] = (color >> (8*1)) & 0xff;
    command[13] = (color >> (8*0)) & 0xff;
    command[14] = (scale >> (8*1)) & 0xff;
    command[15] = (scale >> (8*0)) & 0xff;
    command[16] = (value >> (8*1)) & 0xff;
    command[17] = (value >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 18);
    getResponse(1);
}

//**************************************************************************
// The Screen Copy Paste command copies an area of a screen from xs, ys of size given
// by width and height parameters and pastes it to another location determined by xd, yd.
//**************************************************************************
void PICASO_4DGL :: screenCopyPaste(short xs, short ys, short xd, short yd, short width, short height) {
    
    char command[14] = "";
    
    command[0] = (SCREEN_CP >> (8*1)) & 0xff;
    command[1] = (SCREEN_CP >> (8*0)) & 0xff;
    command[2] = (xs >> (8*1)) & 0xff;
    command[3] = (xs >> (8*0)) & 0xff;
    command[4] = (ys >> (8*1)) & 0xff;
    command[5] = (ys >> (8*0)) & 0xff;
    command[6] = (xd >> (8*1)) & 0xff;
    command[7] = (xd >> (8*0)) & 0xff;
    command[8] = (yd >> (8*1)) & 0xff;
    command[9] = (yd >> (8*0)) & 0xff;
    command[10] = (width >> (8*1)) & 0xff;
    command[11] = (width >> (8*0)) & 0xff;
    command[12] = (height >> (8*1)) & 0xff;
    command[13] = (height >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 14);
    getResponse(1);
}

//**************************************************************************
// The Bevel Shadow command changes the graphics “Draw Button” commands bevel shadow depth
// value - 0 = No Bevel Shadow 1-4 = Number of Pixels Deep (Default = 3)
//**************************************************************************
bool PICASO_4DGL :: bevelShadow(short value) {
    
    if (value <= 4) {
        char command[4] = "";
        
        command[0] = (BEVEL_SHADOW >> (8*1)) & 0xff;
        command[1] = (BEVEL_SHADOW >> (8*0)) & 0xff;
        command[2] = (value >> (8*1)) & 0xff;
        command[3] = (value >> (8*0)) & 0xff;
        
        writeCOMMAND(command, 4);
        getResponse(3);
        return true;
    }
    else {
        pc.printf("\n\r    ERROR_BEVEL_SHADOW: Value has to be (0 - 4)\n\r");
        return false;
    }
}
    
//**************************************************************************
// The Bevel Width command changes the graphics “Draw Button” commands bevel width
//**************************************************************************
bool PICASO_4DGL :: bevelWidth(short value) {
    
    if (value <= 15) {
        char command[4] = "";
        
        command[0] = (BEVEL_WIDTH >> (8*1)) & 0xff;
        command[1] = (BEVEL_WIDTH >> (8*0)) & 0xff;
        command[2] = (value >> (8*1)) & 0xff;
        command[3] = (value >> (8*0)) & 0xff;
        
        writeCOMMAND(command, 4);
        getResponse(3);
        return true;
    }
    else {
        pc.printf("\n\r    ERROR_BEVEL_WIDTH: Value has to be (0 - 15)\n\r");
        return false;
    }
}
    
//**************************************************************************
// The Background Colour command sets the screen background colour
//**************************************************************************
void PICASO_4DGL :: bgColor(short color) {
    
    char command[4] = "";
    
    command[0] = (BG_COLOR >> (8*1)) & 0xff;
    command[1] = (BG_COLOR >> (8*0)) & 0xff;
    command[2] = (color >> (8*1)) & 0xff;
    command[3] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 4);
    getResponse(3);
}

//**************************************************************************
// The Outline Colour command sets the outline colour for rectangles and circles.
//**************************************************************************
void PICASO_4DGL :: outlineColor(short color) {
    
    char command[4] = "";
    
    command[0] = (OUTLINE_COLOR >> (8*1)) & 0xff;
    command[1] = (OUTLINE_COLOR >> (8*0)) & 0xff;
    command[2] = (color >> (8*1)) & 0xff;
    command[3] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 4);
    getResponse(3);
}

//**************************************************************************
// The Contrast Command sets the contrast of the display, or turns it On/Off 
// depending on display model
// Contrast 0 = display OFF, non-zero = display ON
// EXCEPTION:
// uLCD-43 supports Contrast values from 1-15 and 0 to turn the Display off.
// 3202X-P1 supports Contrast values from 1 to 9 and 0 to turn the Display off.
// Note: Does not apply to uVGA-II/III modules.
//**************************************************************************
void PICASO_4DGL :: contrast(short contrast) {
    
    char command[4] = "";
    
    command[0] = (CONTRAST >> (8*1)) & 0xff;
    command[1] = (CONTRAST >> (8*0)) & 0xff;
    command[2] = (contrast >> (8*1)) & 0xff;
    command[3] = (contrast >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 4);
    getResponse(3);
}

//**************************************************************************
// The Frame Delay command sets the inter frame delay for the “Media Video” command
// 0-255 milliseconds
//**************************************************************************
bool PICASO_4DGL :: frameDelay(short msec) {
    
    if (msec <= 255) {
        char command[4] = "";
        
        command[0] = (FRAME_DELAY >> (8*1)) & 0xff;
        command[1] = (FRAME_DELAY >> (8*0)) & 0xff;
        command[2] = (msec >> (8*1)) & 0xff;
        command[3] = (msec >> (8*0)) & 0xff;
        
        writeCOMMAND(command, 4);
        getResponse(3);
        return true;
    }
    else {
        pc.printf("\n\r    ERROR_FRAME_DELAY: Value has to be (0 - 255)\n\r");
        return false;
    }
}

//**************************************************************************
// The Line Pattern command sets the line draw pattern for line drawing. 
// If set to zero, lines are solid, else each '1' bit represents a pixel
// that is turned off.
//**************************************************************************
void PICASO_4DGL :: linePatern(short patern) {
    
    char command[4] = "";
    
    command[0] = (LINE_PATERN >> (8*1)) & 0xff;
    command[1] = (LINE_PATERN >> (8*0)) & 0xff;
    command[2] = (patern >> (8*1)) & 0xff;
    command[3] = (patern >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 4);
    getResponse(3);
}

//**************************************************************************
// The Screen Mode command alters the graphics orientation LANDSCAPE, 
// LANDSCAPE_R, PORTRAIT, PORTRAIT_R
//**************************************************************************
void PICASO_4DGL :: screenMode(char c) {   // select screen orientation
    char command[4] = "";
    command[0] = (ORIENTATION >> (8*1)) & 0xff;
    command[1] = (ORIENTATION >> (8*0)) & 0xff;
    
    switch (c) {
        case 1 :
            command[2] = (LANDSCAPE >> (8*1)) & 0xff;
            command[3] = (LANDSCAPE >> (8*0)) & 0xff;
            currentMode = 1;
            break;
        case 2 :
            command[2] = (LANDSCAPE_R >> (8*1)) & 0xff;
            command[3] = (LANDSCAPE_R >> (8*0)) & 0xff;
            currentMode = 2;
            break;
        case 3 :
            command[2] = (PORTRAIT >> (8*1)) & 0xff;
            command[3] = (PORTRAIT >> (8*0)) & 0xff;
            currentMode = 3;
            break;
        case 4 :
            command[2] = (PORTRAIT_R >> (8*1)) & 0xff;
            command[3] = (PORTRAIT_R >> (8*0)) & 0xff;
            currentMode = 4;
            break;
    }
    
    writeCOMMAND(command, 4);
    getResponse(3);
}

//**************************************************************************
// The Transparency command turns the transparency ON or OFF. Transparency 
// is automatically turned OFF after the next image or video command.
//**************************************************************************
void PICASO_4DGL :: transparency(short mode) {
    
    char command[4] = "";
    
    command[0] = (TRANSPARENCY >> (8*1)) & 0xff;
    command[1] = (TRANSPARENCY >> (8*0)) & 0xff;
    command[2] = (mode >> (8*1)) & 0xff;
    command[3] = (mode >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 4);
    getResponse(3);
}

//**************************************************************************
// The Transparent Colour command alters the colour that needs to be made transparent.
//**************************************************************************
void PICASO_4DGL :: transparentColor(short color) {
    
    char command[4] = "";
    
    command[0] = (TRANSPARENT_COLOR >> (8*1)) & 0xff;
    command[1] = (TRANSPARENT_COLOR >> (8*0)) & 0xff;
    command[2] = (color >> (8*1)) & 0xff;
    command[3] = (color >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 4);
    getResponse(3);
}

//**************************************************************************
// Sets various graphics parameters:
// Function = 18 Object Colour -        Sets the Object colour used in various functions 
//                                      such as Draw Slider and Draw Line & Move Origin (0 – 65535 or 0 - 0xFFFF)
//
// Function = 32 Screen Resolution -    Set VGA Screen resolution. Applies to uVGA-II and uVGA-III only 
//                                      (0 for 320x240, 1 for 640 x 480, 2 for 800 x 480)
//
// Function = 33 Page Display -         Choose Page to be displayed. Value depends on the resolution set. 
//                                      Applies to uVGA-II, uVGA-III and uLCD-43 range only.
//                                      e.g. 0-4 for 320x240 resolution on a uVGA-II and uVGA-III
//
// Function = 34 Page Read -            Choose the Page to be read. Value depends on the resolution set. 
//                                      Applies to uVGA-II, uVGA-III and uLCD-43 range only..
//                                      e.g. 0-4 for 320x240 resolution on a uVGA-II and uVGA-III
//
// Function = 35 Page Write -           Choose the Page to be written. Value depends on the resolution set. 
//                                      Applies to uVGA-II, uVGA-III and uLCD-43 range only.
//                                      e.g. 0-4 for 320x240 resolution on a uVGA-II and uVGA-III
//                                      
//**************************************************************************
void PICASO_4DGL :: setGraphics(short function, short value) { // set graphics parameters

    char command[6] = "";
    
    command[0] = (SET_GRAPHICS >> (8*1)) & 0xff;
    command[1] = (SET_GRAPHICS >> (8*0)) & 0xff;
    command[2] = (function >> (8*1)) & 0xff;
    command[3] = (function >> (8*0)) & 0xff;
    command[4] = (value >> (8*1)) & 0xff;
    command[5] = (value >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 6);
    getResponse(1);
}

//**************************************************************************
// Returns various graphics parameters to the caller.
// mode = 0 : Current orientations maximum X value (X_MAX)
// mode = 1 : Current orientations maximum Y value (Y_MAX)
// mode = 2 : Left location of last Object
// mode = 3 : Top location of Object
// mode = 4 : Right location of last Object
// mode = 5 : Bottom location of Object
// 
//**************************************************************************
short PICASO_4DGL :: getGraphics(short mode) { // set graphics parameters

    short answer = 0;
    char command[4] = "";
    
    command[0] = (GET_GRAPHICS >> (8*1)) & 0xff;
    command[1] = (GET_GRAPHICS >> (8*0)) & 0xff;
    command[2] = (mode >> (8*1)) & 0xff;
    command[3] = (mode >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 4);
    answer = getGraphicsResponse();
    pc.printf("\n\ranswer = %i\n\r", answer);
    return answer;
}



















