//
//  Picaso_4DGL-32PTU is a class to drive 4D Systems TFT touch screens with PICASO processor
//  Tested with NUCLEO L152RE development board
//  Copyright (C) <2016> Rihards Balass <rihards.balass@gmail.com>
//
// Picaso_4DGL-32PTU is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Picaso_4DGL-32PTU is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You can see GNU General Public License at <http://www.gnu.org/licenses/>.
//

#include "mbed.h"
#include "Picaso_4DGL-32PTU.h"

//**************************************************************************
// The Media Init command initialises a uSD/SD/SDHC memory card for further operations.
// The SD card is connected to the SPI (serial peripheral interface) of the PICASO-GFX2 chip.
//**************************************************************************
short PICASO_4DGL :: media_Init() {

    char command[2] = "";
    
    command[0] = (MEDIA_INIT >> (8*1)) & 0xff;
    command[1] = (MEDIA_INIT >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 2);
    short success = mediaInitResponse();
#ifdef DEBUGMODE
    pc.printf("\n\r DEBUG: uSD card INIT: %i\n\r", success);
#endif
    return success;
}

//**************************************************************************
// The Set Byte Address command sets the media memory internal 
// Address pointer for access at a non-sector aligned byte address.
//**************************************************************************
bool PICASO_4DGL :: media_SetAdd(int address) {

    char command[6] = "";
    
    command[0] = (MEDIA_SET_ADD >> (8*1)) & 0xff;
    command[1] = (MEDIA_SET_ADD >> (8*0)) & 0xff;
    command[2] = (address >> (8*3)) & 0xff;
    command[3] = (address >> (8*2)) & 0xff;
    command[4] = (address >> (8*1)) & 0xff;
    command[5] = (address >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 6);
    bool success = getResponse(1);
#ifdef DEBUGMODE
    pc.printf("\n\r DEBUG: Set byte address: %i\n\r", success);
#endif
    return success;
}

//**************************************************************************
// The Set Sector Address command sets the media memory internal Address pointer for sector access.
//**************************************************************************
bool PICASO_4DGL :: media_SetSector(int address) {

    char command[6] = "";
    
    command[0] = (MEDIA_SET_SECTOR >> (8*1)) & 0xff;
    command[1] = (MEDIA_SET_SECTOR >> (8*0)) & 0xff;
    command[2] = (address >> (8*3)) & 0xff;
    command[3] = (address >> (8*2)) & 0xff;
    command[4] = (address >> (8*1)) & 0xff;
    command[5] = (address >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 6);
    bool success = getResponse(1);
#ifdef DEBUGMODE
    pc.printf("\n\r DEBUG: Set sector address: %i\n\r", success);
#endif
    return success;
}

//**************************************************************************
// The Read Sector command reads and returns 512 bytes (256 words) 
// pointed to by the internal Sector pointer, determined by the 
// “Set Sector Address” command. 
// After the read the Sector pointer is automatically incremented by 1.
// Answer = acknowledge (byte) , status (word), block (sector) = 1 + 2 + 512 = 515 bytes
//**************************************************************************
bool PICASO_4DGL :: media_RdSector() {

    char command[2] = "";
    
    command[0] = (MEDIA_READ_SECTOR >> (8*1)) & 0xff;
    command[1] = (MEDIA_READ_SECTOR >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 2);
    bool success = readSectorResponse(515);
#ifdef DEBUGMODE
    pc.printf("\n\r DEBUG: Read sector: %i\n\r", success);
#endif
    return success;
}

//**************************************************************************
// The Write Sector command writes 512 bytes (256 words) from a source memory 
// block into the uSD card. After the write the Sect pointer is automatically incremented by 1.
// Response = acknowledge (byte) , status (word)
//**************************************************************************
bool PICASO_4DGL :: media_WrSector(char *block) {

    char command[514] = "";
    bool success = false;
    int j;
    
    command[0] = (MEDIA_WRITE_SECTOR >> (8*1)) & 0xff;
    command[1] = (MEDIA_WRITE_SECTOR >> (8*0)) & 0xff;
    
#if DEBUGMODE
    pc.printf("\n\r     DEBUG: string length = %i\n\r", strlen(block));
#endif
    if (strlen(block) <= 512) {
        j = 513 - strlen(block);
        for (int i = 0; i < strlen(block); i++) {
            command[j++] = block[i];
        }
        success = media_WrData(command, 514);
    }
    else { // data is bigger than one block
        j = 513 - (strlen(block) % 512); // set the first block pointer
        for (int i = 0; i < strlen(block); i++) {
            if (j == 513) {
                success = media_WrData(command, 514);
                #if DEBUGMODE
                    pc.printf("\n\r     DEBUG: Block send = %i", success);
                    //if (success) puts("\n\r Sector send: OK");
                    //else puts("\n\r Sector send: FAIL");
                #endif
                j = 2;
                command[j++] = block[i];
            }
            else command[j++] = block[i];
        }
    }
    return success;
}

bool PICASO_4DGL :: media_WrData(char *block, int size) {

#if DEBUGMODE
    pc.printf("\n\r     DEBUG: Write block =");
    for (int k = 2; k < size; k++) {
        pc.printf(" %02X", block[k]);
    }
    pc.printf("\n\r");
#endif
    writeCOMMAND_2(block, size);
    bool success = writeSectorResponse(3);
    return success;
}

//**************************************************************************
// The Read Byte command returns the byte value from the current media address,
// set by the “Set Byte Address” command.
// The internal byte address will then be internally incremented by one.
//**************************************************************************
bool PICASO_4DGL :: media_ReadByte() {

    char command[2] = "";
    
    command[0] = (MEDIA_READ_BYTE >> (8*1)) & 0xff;
    command[1] = (MEDIA_READ_BYTE >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 2);
    bool success = readResponse();
#ifdef DEBUGMODE
    pc.printf("\n\r DEBUG: Read byte: %i\n\r", success);
#endif
    return success;
}

//**************************************************************************
// The Read Word command returns the word value (2 bytes) from the current media
// address, set by the “Set Byte Address” command.
// The internal byte address will then be internally incremented by one.
// If the address is not aligned, the word will still be read correctly.
//**************************************************************************
bool PICASO_4DGL :: media_ReadWord() {

    char command[2] = "";
    
    command[0] = (MEDIA_READ_WORD >> (8*1)) & 0xff;
    command[1] = (MEDIA_READ_WORD >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 2);
    bool success = readResponse();
#ifdef DEBUGMODE
    pc.printf("\n\r DEBUG: Read Word: %i\n\r", success);
#endif
    return success;
}

//**************************************************************************
// Writes a byte to the current media address that was initially set with the 
// “Set Sector Address” command.
// 
// Note: Writing bytes or words to a media sector must start from the beginning 
// of the sector. All writes will be incremental until the “Flush Media” command 
// is executed, or the sector address rolls over to the next sector. 
// When the “Flush Media” command is called, any remaining bytes in the sector 
// will be padded with 0xFF, destroying the previous contents. 
// An attempt to use the “Set Byte Address” command will result in the 
// lower 9 bits being interpreted as zero. If the writing rolls over to the 
// next sector, the “Flush Media” command is issued automatically internally.
//**************************************************************************
bool PICASO_4DGL :: media_WriteByte(short value) {

    char command[4] = "";
    
    command[0] = (MEDIA_WRITE_BYTE >> (8*1)) & 0xff;
    command[1] = (MEDIA_WRITE_BYTE >> (8*0)) & 0xff;
    command[2] = (value >> (8*1)) & 0xff;
    command[3] = (value >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 4);
    bool success = writeByteResponse();
#ifdef DEBUGMODE
    pc.printf("\n\r DEBUG: Write Byte: %i\n\r", success);
#endif
    return success;
}

//**************************************************************************
// Writes a word to the current media address that was initially set with the 
// “Set Sector Address” command.
// 
// Note: Writing bytes or words to a media sector must start from the beginning
// of the sector. All writes will be incremental until the “Flush Media” command 
// is executed, or the sector address rolls over to the next sector. 
// When the “Flush Media” command is called, any remaining bytes in the sector 
// will be padded with 0xFF, destroying the previous contents. 
// An attempt to use the “Set Byte Address” command will result in the 
// lower 9 bits being interpreted as zero. If the writing rolls over to the 
// next sector, the “Flush Media” command is issued automatically internally.
//**************************************************************************
bool PICASO_4DGL :: media_WriteWord(short value) {

    char command[4] = "";
    
    command[0] = (MEDIA_WRITE_WORD >> (8*1)) & 0xff;
    command[1] = (MEDIA_WRITE_WORD >> (8*0)) & 0xff;
    command[2] = (value >> (8*1)) & 0xff;
    command[3] = (value >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 4);
    bool success = writeByteResponse();
#ifdef DEBUGMODE
    pc.printf("\n\r DEBUG: Write Byte: %i\n\r", success);
#endif
    return success;
}

//**************************************************************************
// After writing any data to a sector, the Flush Media command should be called
// to ensure that the current sector that is being written is correctly stored
// back to the media else write operations may be unpredictable.
//**************************************************************************
bool PICASO_4DGL :: media_Flush() {

    char command[2] = "";
    
    command[0] = (MEDIA_FLUSH >> (8*1)) & 0xff;
    command[1] = (MEDIA_FLUSH >> (8*0)) & 0xff;
    
    writeCOMMAND(command, 2);
    bool success = writeByteResponse();
#ifdef DEBUGMODE
    pc.printf("\n\r DEBUG: Read Word: %i\n\r", success);
#endif
    return success;
}































