//
//  Picaso_4DGL-32PTU is a class to drive 4D Systems TFT touch screens with PICASO processor
//  Tested with NUCLEO L152RE development board
//  Copyright (C) <2016> Rihards Balass <rihards.balass@gmail.com>
//
// Picaso_4DGL-32PTU is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Picaso_4DGL-32PTU is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You can see GNU General Public License at <http://www.gnu.org/licenses/>.
//

#ifndef PICASO_H
#define PICASO_H

#include "mbed.h"
#include "Picaso_4DGL-32PTU_CONSTANTS.h"
#include <string>

// functions
#define CHECK_BIT(var,pos) ((var) & (1<<(pos))) // to check if required bit is set


// Main commands
#define BAUDRATE            0x0026
#define ORIENTATION         0xFF9E

// text commands
#define TEXT_BG_COLOR       0xFFE6
#define SET_FONT            0xFFE5
#define PUT_CHAR            0xFFFE
#define PUT_STRING          0x0018
#define MOVE_CURSOR         0xFFE9
#define TEXT_FG_COLOR       0xFFE7
#define TEXT_WIDTH          0xFFE4
#define TEXT_HEIGHT         0xFFE3
#define TEXT_X_GAP          0xFFE2
#define TEXT_Y_GAP          0xFFE1
#define TEXT_BOLD           0xFFDE
#define TEXT_INVERSE        0xFFDC
#define TEXT_ITALIC         0xFFDD
#define TEXT_OPACITY        0xFFDF
#define TEXT_UNDERLINE      0xFFDB
#define TEXT_ATTRIBUTES     0xFFDA

// graphics commands
#define CHANGE_COLOR        0xFFB4
#define DRAW_CIRCLE         0xFFC3
#define CIRCLE_FILLED       0xFFC2
#define DRAW_LINE           0xFFC8
#define DRAW_RECTANGLE      0xFFC5
#define RECTANGLE_FILLED    0xFFC4
#define DRAW_POLYLINE       0x0015
#define DRAW_POLYGON        0x0013
#define POLYGON_FILLED      0x0014
#define DRAW_TRIANGLE       0xFFBF
#define TRIANGLE_FILLED     0xFFA9
#define CALCULATE_ORBIT     0x0012
#define PUT_PIXEL           0xFFC1
#define MOVE_ORIGIN         0xFFCC
#define LINE_TO             0xFFCA
#define SET_CLIP_WINDOW     0xFFB5
#define CLIPPING            0xFFA2
#define EXTEND_CLIP         0xFFB3
#define DRAW_ELIPSE         0xFFB2
#define ELIPSE_FILLED       0xFFB1
#define DRAW_BUTTON         0x0011
#define BEVEL_SHADOW        0xFF98
#define BEVEL_WIDTH         0xFF99
#define DRAW_PANEL          0xFFAF
#define DRAW_SLIDER         0xFFAE
#define SCREEN_CP           0xFFAD
#define BG_COLOR            0xFFA4
#define OUTLINE_COLOR       0xFF9D
#define CONTRAST            0xFF9C
#define FRAME_DELAY         0xFF9F
#define LINE_PATERN         0xFF9B
#define TRANSPARENCY        0xFFA0
#define TRANSPARENT_COLOR   0xFFA1
#define SET_GRAPHICS        0xFFCE
#define GET_GRAPHICS        0xFFA6
#define CLS                 0xFFCD

// media commands
#define MEDIA_INIT          0xFF89
#define MEDIA_SET_ADD       0xFF93
#define MEDIA_SET_SECTOR    0xFF92
#define MEDIA_READ_SECTOR   0x0016
#define MEDIA_WRITE_SECTOR  0x0017
#define MEDIA_READ_BYTE     0xFF8F
#define MEDIA_READ_WORD     0xFF8E
#define MEDIA_WRITE_BYTE    0xFF8D
#define MEDIA_WRITE_WORD    0xFF8C
#define MEDIA_FLUSH         0xFF8A

// FAT16 file system commands
#define FILE_MOUNT          0xFF03
#define FILE_ERROR          0xFF1F
#define FILE_COUNT          0x0001
#define FILE_DIR            0x0002
#define FILE_FIRST          0x0006
#define FILE_FIRST_RET      0x0024
#define FILE_NEXT           0xFF1B
#define FILE_NEXT_RET       0x0025
#define FILE_EXISTS         0x0005
#define FILE_OPEN           0x000A
#define FILE_CLOSE          0xFF18
#define FILE_READ           0x000C
#define FILE_SEEK           0xFF16
#define FILE_INDEX          0xFF15
#define FILE_TELL           0x000F
#define FILE_WRITE          0x0010
#define FILE_SIZE           0x000E
#define FILE_IMAGE          0xFF11
#define FILE_S_CAPTURE      0xFF10
#define FILE_PUT_C          0x
#define FILE_GET_C          0x
#define FILE_PUT_W          0x
#define FILE_GET_W          0xFF0C
#define FILE_PUT_S          0x
#define FILE_GET_S          0x0007
#define FILE_ERASE          0x
#define FILE_REWIND         0x
#define FILE_LOAD_F         0x
#define FILE_CALL_F         0x
#define FILE_RUN            0x
#define FILE_EXECUTE        0x
#define FILE_IMAGE_CTRL     0x
#define FILE_UNMOUNT        0x
#define FILE_WAV            0x



//**************************************************************************
class PICASO_4DGL {

public :

    Serial pc; // serial variable for debug information
    PICASO_4DGL(PinName tx, PinName rx, PinName rst); // LCD serial construnctor
    int index; // received response from screen
    short Xdest, Ydest;
    char buffer[BUFFER_SIZE];
    char fileReadBuf[BUFFER_SIZE];
    char rxBuf[RXBUFLEN];
    char sendBlock[BUFFER_SIZE];
    char currentFont; // stores current used font
    char currentMode; // stores current orientation
    
// General Commands
    void rxCallback();
    void reset(); // reset screen
    void baudrate(long); // set baudrate
    void mainDemo(); // play demo with main functions
    void textDemo(); // play demo with text based functions
    void graphicsDemo(); // play demo with graphics based functions
    void mediaDemo(); // play demo with media based functions
    void fileSystemDemo(); // play FAT16 demo

// Graphics Commands
    void cls(); // clear screen
    void changeColor(short, short); // change one color to another
    void drawCircle(short, short, short, short); // draw a circle (x, y, radius, color)
    void drawFilledCircle(short, short, short, short); // draw a filled circle (x, y, radius, color)
    void drawLine(short, short, short, short, short); // draw a line (x1, y1, x2, y2, color)
    void drawRectangle(short, short, short, short, short); // draw a rectangle (x1, y1, x2, y2, color)
    void drawFilledRectangle(short, short, short, short, short); // draw a solid rectangle (x1, y1, x2, y2, color)
    void drawPolyline(short, short *, short *, short); // draw multiple lines (n, vx1…vxN, vy1…vyN, color)
    void drawPolygon(short, short *, short *, short); // draw a polygon (n, vx1…vxN, vy1…vyN, color)
    void drawFilledPolygon(short, short *, short *, short); // draw a solid polygon (n, vx1…vxN, vy1…vyN, color)
    void drawTriangle(short, short, short, short, short, short, short); // draw a triangle (x1, y1, x2, y2, x3, y3, color)
    void drawFilledTriangle(short, short, short, short, short, short, short); // draw a solid triangle (x1, y1, x2, y2, x3, y3, color)
    void drawElipse(short, short, short, short, short); // draw elipse (x1, y1, xrad, yrad, color)
    void drawFilledElipse(short, short, short, short, short); // draw a solid elipse (x1, y1, xrad, yrad, color)
    void calculateOrbit(short, short); // calculate distant point coordinates (angle, distance)
    void putPixel(short, short, short); // draw a pixel (x, y, color)
    void moveOrigin(short, short); // change origin point (x, y)
    void lineTo(short, short); // draw a line from current origin to set point (x, y) 
    void setClipWindow(short, short, short, short); // set a clipping window
    void clipping(short); // enable / disable clipping
    void extendClipRegion(); // extend clipping region to the last text or image shown
    void drawButton(short, short, short, short, short, short, short, short, char *); // draw a 3D button (state, x, y, btnColor, txtColor, font, txtWidth, txtHeight, string)
    bool bevelShadow(short); // change the drawButton bevel shadow depth
    bool bevelWidth(short); // change the drawButton bevel width
    void drawPanel(short, short, short, short, short, short); // draw a 3D panel (state, x, y, width, height, color)
    void drawSlider(short, short, short, short, short, short, short, short); // draw a slider bar (mode, x1, y1, x2, y2, color, scale, value)
    void screenCopyPaste(short, short, short, short, short, short); // copy a portion of screen (xs, ys, xd, yd, width, height)
    void bgColor(short); // change background color
    void outlineColor(short); // change outline color
    void contrast(short); // change display contrast
    bool frameDelay(short); // set inter frame delay for media video command
    void linePatern(short); // set the line draw pattern for line drawing
    void screenMode(char); // set desired orientation of a screen
    void transparency(short); // turn on or off transparency
    void transparentColor(short); // change transparent to color
    void setGraphics(short, short); // set graphics parameters
    short getGraphics(short); // get graphics parameters
    
// Media Commands
    short media_Init(); // initialise memory card
    bool media_SetAdd(int); // set byte address
    bool media_SetSector(int); // set sector address
    bool media_RdSector(); // read sector
    bool media_WrSector(char *); // write sector
    bool media_WrData(char *, int); // write sector
    bool media_ReadByte(); // read byte
    bool media_ReadWord(); // read word
    bool media_WriteByte(short); // write byte
    bool media_WriteWord(short); // write word
    bool media_Flush(); // flush written bytes or words

// Texts Commands
    void setFont(char); // set desired font from embedded fonts
    char moveCursor(short, short); // move cursot to a specific location
    char validateMoveCursor(short, short); // check if requested values are not out of range, based on font and orientation
    void textBgColor(short); // set background color of next text messages
    void putc(char); // send a single character to LCD
    void puts(char *); // send a string to LCD
    void textFgColor(short); // set text color
    int textWidth(short); // set width of a text
    int textHeight(short); // set text height
    int textXGap(short); // set gap between characters in pixels (X axis)
    int textYGap(short); // set gap between characters in pixels (Y axis)
    void textBold(short); // set text to bold
    void textInverse(short); // invert text BG and FG colors
    void textItalic(short); // set text to italic
    void textOpacity(short); // set text opacity
    void textUnderline(short); // set text to underline
    void textAttributes(short); // set 4 attributes (bold, italic, inverse, underline)

// Response commands
    bool getResponse(int); // wait for LCD response
    bool calculateOrbitResponse(); // wait for LCD response
    short getGraphicsResponse(); // wait for LCD response
    short mediaInitResponse(); // response from mediaInit command
    bool readSectorResponse(int); // response from read sector command
    bool writeSectorResponse(int); // response from write sector command
    bool readResponse(); // response from read byte or read word commands
    bool writeByteResponse(); // response from write sector command
    bool fileMountResponse(); // response from file mount
    short fileErrorResponse(); // returns error code
    short fileCountResponse(); // returns searched file count
    short getFilenameResponse(char*); // returns searched filename and length
    short readFileResponse(short); // returns count of bytes and stores read data in buffer
    int fileTellResponse(); // returns current pointer
    
// File system commands
    bool file_Mount(); // mount file system
    short file_Error(); // get latest error code
    short file_Count(char*); // get file count by given name
    short file_Dir(char*); // list the stream of filenames by given name and return count
    void file_FindFirst(char*); // list the stream of filenames by given name
    short file_FindFirstRet(char*, char*); // list the stream of filenames by given name
    void file_FindNext(); // list the stream of filenames by given name
    short file_FindNextRet(char*); // list the stream of filenames by given name
    bool file_Exists(char*); // search for file
    short file_Open(char*, char); // open a file if exists
    bool file_Close(short); // closes a file
    short file_Read(short, short); // read specified bytes from file and return count (size, handle)
    bool file_Seek(short, short, short); // set pointer in file
    bool file_Index(short, short, short, short); // set pointer in file
    int file_Tell(short); // returns current pointer position
    short file_Write(short, char*, short); // write data to file (count of bytes, string of data, handle)
    int file_Size(short); // returns file size in bytes (useful not to get EOF error 
    bool file_Image(short, short, short); // display an image
    bool file_ScreenCapture(short, short, short, short, short); // make a screenshot
    
    short file_GetW(short); // read 2 bytes from a file
    short file_GetS(short, short, char*); // read specified set of bytes from a file (size, handle, stringTo)
    
protected :

    Serial     _cmd; // serial variable, to comunicate with screen
    DigitalOut _rst; // screen reset pin
    
    
    void freeBUFFER  (void);
    void writeBYTE   (char);
    void writeCOMMAND(char *, int); // send command to screen
    void writeCOMMAND_2(char *, int); // send command to screen
};

#endif
