/*
    Project: TrafficEx
    File: main.cpp
    
    Control a hypothetical traffic light using Timeouts. Display a Violation 
    message if light is run (using polling in main).
    
    Car active high input (D6) is created by placing the photocell on the low 
    side of a voltage divider created using a series of 1 kOhm resistors (the 
    number depends on ambient conditions). Three worked well for me.
    
    The red, yellow and green LEDs are connected to pins D2, D3 and D4, 
    respectively through 330 Ohm resistors. They are active high.
    
    Created by Dr. C. S. Tritt
    Last revised: 10/10/17 (v. 1.0)
*/
#include "mbed.h"

const float grnTime = 3.0f; // Green duration.
const float ylwTime = 2.0f; // Yellow duration.
const float redTime = 4.0f; // Red duration.

// These functions change to the indicated color and schedule the next change.
void grnChange();
void ylwChange();
void redChange();
 
Timeout changeLight; // Reuse for all colors.

// Red, yellow and green LEDs.
DigitalOut redLight(D2);
DigitalOut ylwLight(D3);
DigitalOut grnLight(D4);

// Analog signal used as digital input. Threshold is about 1.6 V.
DigitalIn car(D6);

int main() {
    
    redLight = 1; // Start with red on. Seems safest.
    grnLight = 0;
    ylwLight = 0;
    
    redChange(); // Call redChange to start cycling.
    
    // redChange will return here almost immediately (after scheduling 
    // grnChange).
    
    while(true) { // Main loop.
        // Display RYG status, stay on line. Okay to use printf in main.
        printf("RYG: %d %d %d. ", (int) redLight, (int) ylwLight, 
            (int) grnLight);
        // Display car status, stay on line.    
        printf("Car: %d.", (int) car);
        if ((int) car && (int) redLight) {
            printf(" Violation!\n"); // Note violation.
        } else {
            printf("\n"); // Just advance a line.
        }     
        wait(0.5); // Pause half a second.
    }
}

void grnChange(){
    redLight = 0; // Turn off red.
    grnLight = 1; // Turn on green. 
    changeLight.attach(&ylwChange, grnTime); // Schedule change to yellow. 
}

void ylwChange(){
    grnLight = 0; // Turn off green.
    ylwLight = 1; // Turn on yellow. 
    changeLight.attach(&redChange, ylwTime); // Schedule change to red.    
}

void redChange(){
    ylwLight = 0; // Turn off yellow.
    redLight = 1; // Turn on red. 
    changeLight.attach(&grnChange, redTime); // Schedule change to green.
}