/*
    Project: FlexiBarDemo
    File: flexiBar.cpp
    Function: flexiBar (v. 1.2)
    Created by: Dr. C. S. Tritt, 10/22/17

    This function displays an integer value (range 0 to 9) on the specified bar
    graph display. 
    
    Precondition(s):
    
        Physical bar graph display correctly wired to GPIO pins.
        BusOut object that collects display GPIO pins.
    
    Postcondition(s):
    
        Bar graph display is left showing the appropriate output.
    
    Arguments/Parameters:
    
        int value - Value to be displayed. Range -1 to 9, inclusive.
        BusOut &theBar - Bus created from connected GPIO output pins.
        bool single - Single vs. stacked display format. True indicates single.
        bool active - Active high flag. True indicates active high.
    
    Returns (& Error Handling):
    
        0 - Normal operation.
        1 - Error, value out of range.
        
    Notes:
    
    To assure display is operational, at least one bar will always be lit
    (0 is indicated by a single bar and 9 by 10 bars). A special case is the 
    the value -1, which will light no bars.

    I previously thought pow would be better than left shift (<<), but now I see
    that left shift really is the best approach.

    Mbed objects generally must be passed by reference (with &). This prevents
    there possibly two or more instances of the objects possibly having 
    conflicting states.

    Pass theBar (rather than using the global symbol) for flexibility.

    Note there are 2 returns in this function! One for value being out of range
    errors. The other for no error.
*/
#include "mbed.h"

int flexiBar(int value, BusOut &theBar, bool single, bool active)
{

    if (value < -1 || value > 9) return 1; // Indicate value out-of-range error.

    int output; // Use variable to make active low case easy to compute.

    if (value == -1) {
        output = 0;  // Special case of all bars off.
    } else {
        if (single) {
            output = 1 << value; // Light single bars by shifting 1 left.
        } else {
            output = (1 << (value + 1)) - 1; // Light a stack of bars.
        }
    }

    if (active) { // Deal with active low vs. active high.
        theBar = output;
    } else {
        theBar = ~output; // Invert bits for active low.
    }
    return 0; //
}