/*
  Project: EMG_Control
  File: main.cpp
  Created by: Dr. C. S. Tritt
  Last revision on: 11/6/17 (v. 1.1)

  EMG control program for use with Olimex ECG/EMG shields and active low relays.
  For use with my Matlab OlimexChartNucleo.m program. Use for my ECG and 
  Human-Human interface activities. See external documentation for more 
  information.
*/
#include "mbed.h"
// Needed for isdigit function.
#include <ctype.h>

// Global Constants...
// EMG input pin (set by Olimex jumper).
#define EMG_IN_PIN A0
// Relay digital output pin.
#define RELAY_OUT_PIN D12

// ID String
const char idString[] = {"EMG Control Nucleo v. 1.0"};
// Buad rate, must be matched in Serial Monitor/Matlab. 115200 works.
const int PSPD = 115200;
// Default wait time between samples (mS). 50 Hz.
const int WAIT = 20;
// Default hold time for simulator output (mS).
const int HOLD = 500;
// Default decay factor (larger is faster).
const int DECAY = 2;
// Default mean of EMG signal (usually around 512).
const int MEAN = 512;
// Default scale factor (larger lowers sum).
const int SCALE = 4;
// Default threshold for relay activation.
const float THRESHOLD = 512;
// Default output code: 1 = EMG, 2 = RSum
const int CODE = 1;

// Global mbed objects...
// Relay output object & pin.
DigitalOut relay(RELAY_OUT_PIN);
// EMG analogy input object & pin.
AnalogIn oliVal(EMG_IN_PIN);
// PC serial object pins & baud rate.
Serial pc(USBTX, USBRX, PSPD);
// Ticker for millisecond clock.
Ticker milliTick;

// Global variable...
volatile unsigned int millis = 0; // Milliseconds since reset for timestamp.

// Function declarations...
// Millisecond clock callback/ISR.
void milliIncr(void);
// Converts ASCII digit code to corresponding int value.
int dig2Int(char myChar);

int main()
{
    relay.write(1); // Assure active low relay is off at start.
    // Announce program identity.
    pc.printf("%s\n", idString);

    // Define and initialize important variables...
    unsigned int actTime = 0; // Most recent relay activation time (mS).
    float waitTime = (float) WAIT / 1000.f; // Wait time between samples (S).

    int hold = HOLD; // Hold time for stimulator output (mS).
    int decay = DECAY; // Decay factor (larger is faster).
    int mean = MEAN; // Mean of EMG signal (usually around 512).
    int scale = SCALE; // Scale factor (larger lowers sum).
    int threshold = THRESHOLD; // Threshold for relay activation.
    int rSum = 0; // Initial decaying running sum of squared activity.

    int parmVal = 0; // Initial command parameter value.
    int code = CODE; // Output code: 1 = EMG, 2 = Running Sum.
    bool go = false; // Flag for sending data. Start not sending.

    milliTick.attach_us(milliIncr, 1000); // Attach ISR to mS clock ticker.
    while(true) {
        if (pc.readable()) { // Process command text...
            char myChar = pc.getc(); // Read a character.
            pc.putc(myChar); // Echo the character.
            if (isdigit(myChar)) { // Process the digit...
                parmVal = 10 * parmVal + dig2Int(myChar); // Shift & add value.
            } else { // Deal with command code letters...
                switch (myChar) { // Add cases below for more commands.
                    case 'd':  // Change decay value.
                        if (parmVal == 0) parmVal = DECAY; // Use default.
                        decay = parmVal; // Change decay.
                        break;   // All but last case needs a break.
                    case 'g':  // Switch into go (read & output data) mode.
                        go = true;
                        break;   // All but last case needs a break.
                    case 'h':  // Change hold value.
                        if (parmVal == 0) parmVal = HOLD; // Use default.
                        hold = parmVal; // Change hold.
                    case 'i':  // Send program ID.
                        pc.printf("%s\n", idString);
                        break;   // All but last case needs a break.
                    case 'm':  // Change expected mean value.
                        if (parmVal == 0) parmVal = MEAN; // Use default.
                        mean = parmVal; // Change mean.
                        break;   // All but last case needs a break.
                    case 'o':  // Change output code value.
                        if (parmVal == 0) parmVal = CODE; // Use default.
                        code = parmVal; // Change code.
                        break;   // All but last case needs a break.
                    case 's':  // Change scale factor value.
                        if (parmVal == 0) parmVal = SCALE; // Use default.
                        scale = parmVal; // Change scale.
                        break;   // All but last case needs a break.
                    case 't':  // Change activation threshold value.
                        if (parmVal == 0) parmVal = THRESHOLD; // Use default.
                        threshold = parmVal; // Change threshold.
                        break;   // All but last case needs a break.
                    case 'v':  // Verbose. Use to debug commands. Add status?
                        if (parmVal == 0) parmVal = -1; // Translate 0 to -1.
                        // Send paraVal in response to v command.
                        pc.printf("Responding to v, parmVal = %d\n", parmVal);
                        break;   // All but last case needs a break.
                    case 'w':  // Change between sample wait value (S).
                        if (parmVal == 0) parmVal = WAIT; // Use default.
                        waitTime = (float) parmVal / 1000.f; // Change waitTime.
                        break;   // All but last case needs a break.
                    case 'x':  // Send data marker (time & minus parmVal).
                        // Also use for debugging command processing.
                        pc.printf("%u, %d\n", millis, -parmVal);
                        break;   // All but last case needs a break.
                    case 'z':  // Switch out of go (read & output data) mode.
                        go = false;
                        break;   // All but last case needs a break.
                }
                parmVal = 0; // Reset parmVal to zero for next command.
            }
        } // End of command text processing.
        if (go) { // Complete go mode tasks...
            // Read and process EMG data...
            float EMG_float = oliVal.read(); // Read & save analog Olimex value.
            unsigned int time = millis; // Save read time.
            int EMG_int = (int)(1023.f * EMG_float); // Scale EMG reading.
            int diff =EMG_int - mean; // Center on specified mean.
            // Accumulate decaying running sum of squares (filtered value).
            rSum = rSum/decay + diff*diff/scale;
            
            // Activate relay if appropriate...
            if (time - actTime > hold) {
                if (rSum > threshold) {    
                    relay.write(0); // Activate active low relay.
                    actTime = time; // Save activation time.
                } else {
                    relay.write(1); // Assure relay is deactivated.               
                }
            }
            
            // Send data to PC...
            pc.printf("%u, ", time); // Send the time and a comma.
            if (code == 1) {
                pc.printf("%d\n", EMG_int);
            } else if (code == 2) {
                pc.printf("%d\n", rSum);
            } else { // Invalid code value.
                pc.printf("?\n");
            }
        } else if (!relay.read()){
            relay.write(1); // Deactivate relay in
        }
        wait(waitTime); // This wait determines sample & send rate.
    }
}

void milliIncr(void) // Call back that adds millisconds to millis.
{
    millis++; // Increment by 10 milliseconds each time.
}

int dig2Int(char myChar)
{
    /*
      Utility function for Converting a digit to the equivalent int.
    */
    int charVal = (int) myChar; // ASCII digit binary values are shifted by 48.
    return charVal - 48;
}