/*
  Project: AnalogSineFinite (based on Nucleo_sinwave_output example)
  File: main.cpp
  Modified by by: Dr. C. S. Tritt
  Last revision on: 9/26/17 (v. 1.0)

  Finite duration version. Use to generate CSV file for import into Excel or 
  Matlab.

  Generates sine and cosine wave forms to demonstrate D to A operation. Use a
  data buffer for maximum speed (not necessary for 4 mS samples & 1.0 Hz 
  signal). Output clips at a bit 0 and 3.3 V, so use AMPLITUDE <= 0.4 and 
  OFFSET = 1.1. Zoom Horizontal scale out on MSOX2012A scopes and wait for 
  refresh. Without wait, max. freq. (with both channels) is about 2 kHz.
  
  Possible improvements/modifications:
  
    Reduce number of points/cycle to 18 or 24 (more or less).
    Change design to use 2 buffers.
    Change design to use a TimeOut.
    Use pots to adjust freq. and phase to produce Lissajous patterns.
*/
#include "mbed.h"

#if !DEVICE_ANALOGOUT // Flag set by mbed.
    #error AnalogOut is not supported on this device.  // Abort compilation.
#else
    AnalogOut my_Sin(PA_4); // Sine
    AnalogOut my_Cos(PA_5); // Cosine
    
    const float PI = 3.14159265f; // Value of Pi for radians calculation.
    const float AMPLITUDE = 0.4f; // x * 3.3V. D to A is limited to < 3.3 V.
    const float OFFSET = 1.1f; // Make signal all possitive.
    
    // Configuration for sinewave output
    const int BUFFER_SIZE = 250; // Elements (i.e., points).
    float buffer[BUFFER_SIZE];
    
    void calculate_sinewave(void); // Fills buffer.
    
    int main()
    {
        printf("Finite sine wave example\n");
        calculate_sinewave();
        for (int j = 0; j < 2; j++) { // Output 2 cycles and stop.
            // sinewave output
            for (int i = 0; i < BUFFER_SIZE; i++) {
                my_Sin.write(buffer[i]); // Output a point.
                printf("%f, ", buffer[i]); // Send point to PC.
                // Generate Cosine by shifting sine by 1/4 cycle.
                my_Cos.write(buffer[(i + BUFFER_SIZE/4) % BUFFER_SIZE]);
            }
        }
    }
    
    void calculate_sinewave(void) // This function fills the buffer.
    {
        for (int i = 0; i < BUFFER_SIZE; i++) {
            // Calculate radians from 0 to 2*Pi.
            float rads = (2.0f * PI * (float) i / (float)BUFFER_SIZE);
            // Fill the buffer.
            buffer[i] = AMPLITUDE * (sin(rads) + OFFSET); // Sine of radians.
        }
    }
#endif