/*
    Project: Button Tests
    File: main.cpp
    
    This program demonstrates User Button behavior.
    
    Written by: Dr. C. S. Tritt
    Created: 9/22/21 (v. 1.0)
*/
#include "mbed.h"

DigitalIn uB(USER_BUTTON); // Active low! Normally 1 go to 0 when pressed.

int main() {
    // Enter main loop. Sequential calls to the button like this is generally
    // not good practice, but I want to use the DigitalIn object in these 
    // contexts to illustrate my points. See while loop below for a better 
    // (more common and realistic) approach.   
    //while(true) { // Standard infinite loop.
    for (;;) { // Alternative infinite loop.
        // Use the button in some printf calls.
        printf("\nuB (w/o cast) = %d.\n", uB); // Bad! Warning! Wrong!
        printf("uB (with int cast) = %d.\n", static_cast<int>(uB));
        // I found a reference that claimed there is a %b (or B) speicifer for 
        // type bool, but it doesn't work in mbed. So use %d to see it as a 
        // decimal value.
        printf("uB (with bool cast) = %d.\n", static_cast<bool>(uB));                              
        printf("uB.read = %d.\n", uB.read());  
        
        // Use the button in some if constructs. No need to compare. No need to
        // use .read().   
        if (uB) 
            printf("In if (uB).\n");
        if (uB.read()) 
            printf("In if (uB.read()).\n");
        if (!uB) 
            printf("In if (!uB).\n");
        if (!uB.read()) 
            printf("In if (!uB.read()).\n");
        ThisThread::sleep_for(500); // For 0.5 seconds after some output.
    
        // Best practice is to get and save the button state and use until you  
        // are done with your response to it.
        int count = 5; // Count for loop below.
        bool uBstate = uB; // Get the buttons state. Save it and use it.
        // uB could change states between the if below and the following while. 
        // This would result in an extra line of output without the contents of 
        // the while being output. That is what the use of uBstate prevents.
        if (!uBstate) printf("\n"); // Print extra line before while output.
        while (!uBstate && count >= 1) {
            // Count down from 5 to 1.
            // Do something here with uBstate false & assuming uB is false.
            // Decrement after use!
            printf("In while loop. Count = % d.\n", count--);
            // Now determine if state has changed.
            if (uB != uBstate) {
                printf("The button changed while in the loop. Exiting.\n");
                // Using a while and setting exit conditions is generally better 
                // than using a break. Strive for one way in and one way out in
                // all your code for more reliability and easier maintenance. 
                // But there are exceptions to this rule. That is why break (and 
                // continue exist. 
                uBstate = !uBstate; // Update uBstate to exit loop.
            } //end of if.
        ThisThread::sleep_for(500); // For 0.5 seconds in while.
        } // end of !uBstate while.
        if (uBstate)
            printf("Button released during countdown.\n\n");
        else 
            printf("Countdown completed.\n\n");
    } // end of main while.
} // end of main function.