/*******************************************************************/
/*                                                                 */
/*                 Pilote                                          */
/*                                                                 */
/*  Objet de pilotage de variateur                                 */
/*                                                                 */
/*                                                                 */
/*******************************************************************/
#ifndef _PILOTE_
#define _PILOTE_

#include "mbed.h"
#include <Serial.h>
#include "Constantes.h"
#include "Variable.h"
#include "time.h"

// Etat des ordres variateur
#define AUCUN               0
#define ATTENTE             1
#define VALIDE              2
#define DEFAUT              3

// Ordres variateur
#define ORDRE_AUCUN         0x0
#define ORDRE_ARRET         0x41
#define ORDRE_MONTE         0x4D
#define ORDRE_DESCEND       0x44
#define ORDRE_FREIN         0x46
#define ORDRE_LECTURE       0x4C
#define ORDRE_CONFIGURE     0x43

// Parametres variateur
#define NB_PARAM_VARIATEUR  15
#define VERSION             0x00
#define STARTUP             0x01
#define INCREMENT           0x02
#define SEUIL_DEMARRAGE     0x03
#define ACCELERATION        0x04
#define DECELERATION        0x05
#define KPV                 0x06
#define KIV                 0x07
#define KDV                 0x08
#define KA                  0x09
#define CONSIGNE            0x0A
#define ETAT_EL             0x0B
#define ETAT_SL             0x0C

// Etats du frein du moteur
#define SERRE               0
#define DESSERRE            1

#define LONGUEUR_CHAINE_EMISSION    100
#define LONGUEUR_CHAINE_RECEPTION   100

#define TIMEOUT_RECEPTION_ms       200
#define DELAI_COMMANDE_ARRET_ms    500

/**     Class Pilote
*       Pilotage de variateur de pile de pont via liaison série et protocole 3R
*/
class Pilote
{
public:
    /**     Constructeur
    *       @param Id : Identifiant de l'instance
    */
    Pilote ( S16 Id);
    
    /**     Initialise et ouvre le port série
    *       @param Baudrates  : Vitesse de la liaison série
    *       @param bits       : Bits de données
    *       @param Stop       : Bits de stop
    */
    void    Init ( int Baudrates, int bits,  int Stop) ;
    /**     Ordre de marche de la pile
    *       @param Mode       : Mode de fonctionnement
    *       @param Sens       : Sens de déplacement
    *       @param Hauteur    : Hauteur cible (mm)
    *       @param Vitesse    : Consigne de vitesse (tr/min)
    */
    void    Marche ( U8 Mode, U8 Sens, S16 Hauteur, S16 Vitesse ) ;
    /**     Ordre d'arret de la pile
    */
    void    Arret (void) ;
    /**     Commande manuelle du frein
    *       @param Etat       : Etat du frein {SERRE,DESSERRE}
    */
    void    Frein ( U8 Etat ) ;
    /**     Lecture d'un parametre du variateur
    *       @param Num_Parametre       : numéro du parametre à lire
    */
    void    Lecture ( U16 Num_Parametre ) ;
    /**     Configuration d'un parametre du variateur
    *       @param Num_Parametre       : numéro du parametre à configurer
    *       @param Valeur              : nouvelle valeur du parametre
    */
    void    Configure ( U16 Num_Parametre , S32 Valeur ) ;
    /**     Etalonnage de hauteur de la pile
    *       @param Points1             : Hauteur en points de la mesure basse
    *       @param Hauteur1            : Hauteur en mm de la mesure basse
    *       @param Points2             : Hauteur en points de la mesure haute
    *       @param Hauteur2            : Hauteur en mm de la mesure haute
    */
    void    Etalonnage (S16 Points1, S16 Hauteur1 , S16 Points2 , S16 Hauteur2 ) ; 
    /**     Reception de la réponse du variateur
    */ 
    void    Reception ( void ) ;
    
    /**Pointeur du port série*/
    Serial* pPort ;  
    /**Hauteur courante de la pile (mm)*/ 
    S16     Hauteur_Courante ;
    /**Vitesse courante de la pile*/
    S16     Vitesse_Courante ;
    ///Hauteur de la pile (pts)
    S16     Mesure_Courante ;
    ///Hauteur cible du mouvement en cours (mm)
    S16     Hauteur_Cible ;
    ///Identifiant de l'instance
    S16     Id ;
    ///Compteur de timeouts de la liaison série
    S16     Compteur_Timeout ;
    ///Deplacement en cours (ARRET, MONTE, DESCEND)
    U8      Etat_Deplacement ;
    ///Etat de la commande manuelle de frein (SERRE, DESSERRE)
    U8      Etat_Frein ;
    ///Etat de la liaison série
    U8      COM_OK ; 
    ///Tableau de paramètres du variateur
    S32     Parametres_Vario_S32[NB_PARAM_VARIATEUR] ;
    
private:
    void    Emission ( U8 *pChaine , U8 Longueur ) ;
    U8      Chaine_A_Emettre[LONGUEUR_CHAINE_EMISSION] ;   
    U8      Chaine_Recue[LONGUEUR_CHAINE_RECEPTION] ;
    Timer   Chrono_Pilote ;
    Timer   Chrono_Arret ;
    U8      *pEmission ;
    U8      *pReception ;
    U8      Nb_Caracteres_Recus ;
    U8      Statut_Ordre_En_Cours ;
    U8      Ordre_En_Cours ;
    U8      Dernier_Ordre_Confirme ;
    F32     MM_par_Points ;
    F32     MM_Offset ;
    int     Debut_Emission_ms ;
    int     Fin_Reception_ms ;
    int     Age_Ordre_Arret_ms ;
    
};

#endif