//--------------------------------------------------------------
// Sampling and Spectrum analysis using FFT
// Copyright (c) 2014 MIKAMI, Naoki,  2014/06/29 
//--------------------------------------------------------------

#include "mbed.h"
#include "AdcInternal.hpp"
#include "fftReal.hpp"

using namespace Mikami;

// sampling frequency
const float FS_ = 10.0e3f;
const int N_FFT_ = 256;

Adc adc_(A0);
Ticker timer_;      // for timer interrupt
DigitalIn userButton_(USER_BUTTON);
FftReal fft_(N_FFT_);

float xn_[N_FFT_+1];
volatile int count_;

// Sampling
void TimerIsr()
{
    xn_[count_] = adc_.Read();  // AD
    count_++;
}

int main()
{
    printf("Ready.\r\n");
    while(userButton_);     // wait for user button pressed
    printf("User button is pushed!\r\n");

    count_ = 0;
    timer_.attach_us(&TimerIsr, 1.0e6f/FS_);
    while (count_ < N_FFT_+1);
    timer_.detach();        // Detach TimerIsr
    printf("End of Sampling.\r\n");

    float xnW[N_FFT_];      // for windowed data
    Complex yk[N_FFT_/2+1];
    float yDb[N_FFT_/2+1];
    
    // Windowing
    float pi2N = 6.283185f/N_FFT_;
    for (int n=0; n<N_FFT_; n++)
        xnW[n] = (xn_[n+1] - xn_[n])*(0.54f - 0.46f*cosf(pi2N*n));
        
    // Execute FFT
    fft_.Execute(xnW, yk);
    
    // Square of absolute value of FFT
    for (int n=0; n<=N_FFT_/2; n++)
        yDb[n] = norm(yk[n]);
        
    // Get maximum value of yk[n]
    float max = 0;
    for (int n=0; n<=N_FFT_/2; n++)
        max = (yDb[n] > max) ? yDb[n] : max;
        
    // Normalized spectra to dB
    for (int n=0; n<=N_FFT_/2; n++)
        if (yDb[n] > 0)
            yDb[n] = 10.0f*log10f(yDb[n]/max);
        else
            yDb[n] = -100.0f;

    // Output to terminal
    float dt = 1.0e3f/FS_;
    float df = FS_/N_FFT_;
    for (int n=0; n<N_FFT_; n++)
    {
        if (n <= N_FFT_/2)
            printf("%3d, %4.1f,%8.4f, %10.4f,%6.1f\r\n",
                   n, n*dt, xn_[n], n*df, yDb[n]);
        else
            printf("%3d, %4.1f,%8.4f\r\n", n, n*dt, xn_[n]);
    }
}

