//-------------------------------------------------------
//  Class for LCD, ACM1602Ni (Header)
//
//  Default pin assignments
//      D14  SDA ---- to pin5 of LCD module
//      D15  SCL ---- to pin4 of LCD module
//
//  Assignment of I2C ports
//                SDA                SCL
//      I2C1   PB_7 or PB_9(D14)  PB_6(D10) or PB_8(D15)
//      I2C2   PB_3(D3)           PB_10(D6)
//      I2C3   PB_4(D5) or PC_9   PA_8(D7)
//
//  2014/12/14, Copyright (c) 2014 MIKAMI, Naoki
//-------------------------------------------------------
//  I2C_TypeDef: See stm32f401xe.h on following URL
//  http://mbed.org/users/mbed_official/code/mbed/file/
//       552587b429a1/TARGET_NUCLEO_F401RE
//-------------------------------------------------------

#ifndef ACM1602NI_HPP
#define ACM1602NI_HPP

#include "mbed.h"
#include <string>

namespace Mikami
{
    class Acm1602Ni
    {
    public:
        // Constructor
        Acm1602Ni(PinName sda = D14,        // SDA
                  PinName scl = D15,        // SCL
                  uint32_t clock = 100000,  // clock: 100 kHz
                  bool cursor = false,      // cursor:  off
                  bool blink = false);      // blink:   off
        // Return false if LCD is not connected
        bool GetOk() { return connected_; }
        // All clear
        bool Clear();
 
        // Send command
        bool WriteCmd(uint8_t cmd) { return LcdTx(0x00, cmd); }
 
        // Write character
        void WriteChar(char data) { LcdTx(0x80, data); }
        // Specify display position, x: 0 - 15, y: 0, 1

        void SetXY(uint8_t x = 0, uint8_t y = 0)
        { WriteCmd(x + y*0x40 | 0x80);}

        // Write string
        void WriteString(const char str[]);
        void WriteString(const string str) { WriteString(str.c_str()); }

        // Write string from specified position
        void WriteStringXY(const char str[], uint8_t x, uint8_t y);
        void WriteStringXY(const string str, uint8_t x, uint8_t y)
        { WriteStringXY(str.c_str(), x, y); }

        // Clear of specified line
        void ClearLine(uint8_t line)
        { WriteStringXY("                ", 0, line); }

    private:
        // Slave address of ACM1602NI (0x50)
        //      left-justified 7-bit address
        static const uint8_t LCD_ADDRESS_ = 0x50 << 1;

        I2C i2c_;               // Object of I2C
        I2C_TypeDef* myI2c_;    // Pointer of I2C

        bool connected_;        // false: LCD is not connected

        bool LcdTx(uint8_t cmdData, uint8_t data);
        bool Start();

        // Forbid to use copy constructor
        Acm1602Ni(const Acm1602Ni&);
        // Forbid to use substitution operator
        Acm1602Ni& operator=(const Acm1602Ni&);

        // For check contents of SR2
        bool CheckSR2(uint16_t value)
        { return value == (myI2c_->SR2 & value); }

        // For check contents of SR1 and SR2
        bool CheckSR12(uint16_t v1, uint16_t v2)
        { return (v1 == (myI2c_->SR1 & v1)) &&
                 (v2 == (myI2c_->SR2 & v2)); }

        void TxDR(uint8_t data) { myI2c_->DR = data; }
        void SetCR1(uint16_t data) { myI2c_->CR1 |= data; }
    };
}
#endif  // ACM1602NI_HPP
