/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
// Modify to change full scale gravity range 141207 sc


#include "MMA8451Q8.h"

#define REG_WHO_AM_I      0x0D
#define REG_CTRL_REG_1    0x2A
#define REG_OUT_X_MSB     0x01
#define REG_OUT_Y_MSB     0x03
#define REG_OUT_Z_MSB     0x05
#define XYZ_DATA_CFG      0x0E

#define UINT14_MAX        16383

#define MAX_2G            0x00
#define MAX_4G            0x01
#define MAX_8G            0x02

#define GSCALING          1024.0
#define NUM_DATA            2
#define ADDRESS_INDEX       0
#define DATA_INDEX          1
#define ACTIVEMASK          0x01

MMA8451Q::MMA8451Q(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr) {
    // activate the peripheral
    uint8_t data[2] = {REG_CTRL_REG_1, 0x01};
    writeRegs(data, 2);
}

MMA8451Q::~MMA8451Q() { }

void MMA8451Q::setStandbyMode(){
    //create unsigned integer to hold bits
    uint8_t registerData[1];
    //create unsigned int array with 2 spots
    //that will hold the registry control bits and 8 zeroes
    uint8_t data[NUM_DATA] = {REG_CTRL_REG_1, 0x00};
    
    //read the registry control data into the registerData array
    readRegs(REG_CTRL_REG_1, registerData, 1);
    //bitwise AND operation done between the register data being read,
    //and the flipped (~) ACTIVEMASK (00000001 becomes 11111110 before
    //being compared). The data is then set into the data array
    data[1] = registerData[0] & ~ACTIVEMASK;
    //write the new value back into the registry. The only thing that
    //has changed here is that the last bit has been switched to a 0
    //to put the device in standby mode with the bitmask.
    writeRegs(data,NUM_DATA);
}

void MMA8451Q::setActiveMode(){
    //create unsigned integer to hold bits
    uint8_t registerData[1];
    //create unsigned int array with 2 spots
    //that will hold the registry control bits and 8 zeroes
    uint8_t data[NUM_DATA] = {REG_CTRL_REG_1, 0x00};
    
    //read the registry control data into the registerData array
    readRegs(REG_CTRL_REG_1, registerData, 1);
    //bitwise OR operation done between the register data being read,
    //and ACTIVEMASK (00000001). The data is then set into the data array
    data[1] = registerData[0] | ACTIVEMASK;
    //write the new value back into the registry. The only thing that
    //has changed here is that the last bit has been switched to a 1
    //to put the device in active mode with the bitmask.
    writeRegs(data,NUM_DATA);
}

uint8_t MMA8451Q::getWhoAmI() {
    uint8_t who_am_i = 0;
    readRegs(REG_WHO_AM_I, &who_am_i, 1);
    return who_am_i;
}
void MMA8451Q::setGLimit(int gSelect) {   
    uint8_t data[NUM_DATA] = {REG_CTRL_REG_1, 0x00};
    
    //incoming gSelect integer used to select g level
    gChosen = gSelect;
    
    setStandbyMode(); //put in standby to select g mode
    
    //writeRegs(data, 2);
    
    data[ADDRESS_INDEX] = XYZ_DATA_CFG;
    data[DATA_INDEX] = gChosen;
    
    //writeRegs(data, 2);// change g limit
    //data[0] = REG_CTRL_REG_1;
    //data[1] = 0x01;
    
    writeRegs(data, 2); //write new g level to register
    setActiveMode(); //make active again
}

float MMA8451Q::getAccX() {
    return (float(getAccAxis(REG_OUT_X_MSB))/GSCALING);
}

float MMA8451Q::getAccY() {
    return (float(getAccAxis(REG_OUT_Y_MSB))/GSCALING);
}

float MMA8451Q::getAccZ() {
    return (float(getAccAxis(REG_OUT_Z_MSB))/GSCALING);
}

void MMA8451Q::getAccAllAxis(float * res) {
    res[0] = getAccX();
    res[1] = getAccY();
    res[2] = getAccZ();
}

int16_t MMA8451Q::getAccAxis(uint8_t addr) {
    int16_t acc;
    uint8_t res[2];
    readRegs(addr, res, 2);

    acc = (res[0] << 6) | (res[1] >> 2);
    if (acc > UINT14_MAX/2)
        acc -= UINT14_MAX;

    return acc;
}

void MMA8451Q::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr};
    m_i2c.write(m_addr, t, 1, true);
    m_i2c.read(m_addr, (char *)data, len);
}

void MMA8451Q::writeRegs(uint8_t * data, int len) {
    m_i2c.write(m_addr, (char *)data, len);
}
