/**    Hibernus Library
 *	   University of Southampton 2017
 *
 *		Open-source liberary that enable any of your Mbed project work with transient enegy sources.
 *		In order to use this library include the "hibernus.h" header file, and use the "Hibernus()" method at the beginning of you main funtion.
 *		For more details and example see the "main.cpp" exampe file, and the attached documnetation
 *
 *
 *     Released under the MIT License: http://mbed.org/license/mit
 */
#include "config.h"
#include "hibernus.h"

volatile unsigned int dummyread;      // Variable used for serialization
extern "C" void CMP0_IRQHandler()   //To make sure the compiler sees handler, need to declare it as extern (https://developer.mbed.org/forum/mbed/topic/419/?page=1#comment-2126)
{
	
	CMP0->SCR |= CMP_SCR_CFF_MASK | CMP_SCR_CFR_MASK;   // Clear CMP0 flags
  dummyread = CMP0->SCR;                              // Read back for serialization

    
  if(isFlagSet(getFlag_2())&& isFlagSet(getFlag_3()) && isFlagSet(getFlag_4())){  // Enter this only after complete shut down
		erase_flags_memory(); 													// Erase flash sector used to save the flags 
		setFlag(getFlag_1());
		setFlag(getFlag_4());
   }
    
	if(isFlagSet(getFlag_2())){									// Hibernate procedure           
		__disable_irq();     									// Disable interrupts
    
		volatile unsigned int sp1=(_SP+0x28), lr1= *(unsigned int*)(_SP+0x1C), pc1=*(unsigned int*)(_SP+0x20);
		// Save previous SP value from main // 0x40 for hibernate function 38 otherwsie
     *getCore_SP() = sp1;                        
    
     // Save previous LR value from main// 34 for hibernate function 2C otherwise
		*getCore_LR() = lr1;
			
     // Save previous PC value from main // 38 for hibernate function 30 otherwise
		*getCore_PC() = pc1;
    
    Save_RAM_Regs();

		setFlag(getFlag_3());
		setFlag(getFlag_4());
        
    Comparator_Setup();
    CMP0->DACCR = 0xDA;               // Enable DAC, Vdd is 6-bit reference, threshold set to P3V3 = +2.37V (V_R)
    CMP0->SCR = 0x16;                 // Enable all interrupts and clear flags
    CMP0->CR1 |= CMP_CR1_EN_MASK;     // Enable comparator module
                                        
    __enable_irq();     // Enable interrupts
		
    Enter_LLS();        // Enter LLS mode
    }
}

extern "C" void LLW_IRQHandler(){
	
    CMP0->SCR |= CMP_SCR_CFF_MASK | CMP_SCR_CFR_MASK;   // Clear CMP0 flags
    dummyread = CMP0->SCR;                              // Read back for serialization
    NVIC_ClearPendingIRQ(CMP0_IRQn);                    // Clear pending CMP0 interrupt so that the CMP_IRQ is not entered
    
		if(isFlagSet(getFlag_3())){   // Enter after hibernation with no power loss
			Comparator_Setup();
      CMP0->DACCR = 0xDC;               // Enable DAC, Vdd is 6-bit reference, threshold set to P3V3 = +2.21V (V_H)
      CMP0->SCR = 0x0E;                 // Enable falling edge interrupt and clear flags
      CMP0->CR1 |= CMP_CR1_EN_MASK;     // Enable comparator module
        
			erase_flags_memory();  // Erase flash sector used to save the flags 
			setFlag(getFlag_2());
			
      __enable_irq();     // Enable interrupts
    }
}
void Comparator_Setup(){
	
  NVIC_SetPriority(CMP0_IRQn, 1);     // Lower the CMP0 interrupt priority from 0 to 1 (smaller number = higher priority)
  NVIC_EnableIRQ(CMP0_IRQn);          // Enable CMP0 interrupts in NVIC
    
  SIM->SCGC4 |= SIM_SCGC4_CMP_MASK;       // Enable comparator module clock
  LLWU->ME |= LLWU_ME_WUME1_MASK;         // Enable CMP0 as a LLWU source
  PMC->REGSC |= PMC_REGSC_BGEN_MASK |     // Allow bangap buffer in low power operation
                  PMC_REGSC_BGBE_MASK;      // Enable bandgap buffer for +1V reference (CMP0_IN6)
                  
  // Comparator in sampled, filtered mode 4B
  CMP0->CR0 = 0x22;                 // Hysteresis level 2 (20mV) and 2 consecutive filter samples must agree
  CMP0->CR1 = 0x00;                 // Low-speed compare, non-inverted output, use filtered output and disable comparator output pin PTA2
  CMP0->FPR = 0x01;                 // Filter sample period = 1 bus clock cycle
  CMP0->SCR = 0x06;                 // Disable all interrupts and clear flags
  CMP0->DACCR = 0x40;               // Disable DAC, Vdd is 6-bit reference, threshold set to zero
  CMP0->MUXCR = 0x3E;               // CMP0_IN7 (DAC) to V+ channel and CMP0_IN6 (+1V bandgap) to V- 3E
}

void Enter_LLS(){
	
	Comparator_Setup();
  CMP0->DACCR = 0xDA;               // Enable DAC, Vdd is 6-bit reference, threshold set to P3V3 = +2.37V (V_R)
  CMP0->SCR = 0x16;                 // Enable rising edge interrupt and clear flags
  CMP0->CR1 |= CMP_CR1_EN_MASK;     // Enable comparator module
	
  NVIC_EnableIRQ(LLW_IRQn);               // Enable LLW interrupts in NVIC
  MCG->C6 &= ~(MCG_C6_CME_MASK);          // DIsable all clock monitors
  SCB->SCR = 1<<SCB_SCR_SLEEPDEEP_Pos;    // Set the SLEEPDEEP bit for stop mode
    
  SMC->PMPROT = SMC_PMPROT_ALLS_MASK;         // Allow LLS power modes
  SMC->PMCTRL &= ~(SMC_PMCTRL_STOPM_MASK);    // Serialisation
  SMC->PMCTRL = SMC_PMCTRL_STOPM(0x3);        // Select LLS as desired power mode
  dummyread = SMC->PMCTRL;                    // Read back for serialisation

  __WFI();    // Stop executing instructions and enter LLS (wait for interrupt)
}

void configure_VH_comparator_interrupt(){                            
  Comparator_Setup();
  CMP0->DACCR = 0xDC;               // Enable DAC, Vdd is 6-bit reference, threshold set to P3V3 = +2.21V (V_H)
  CMP0->SCR = 0x0E;                 // Enable falling edge interrupt and clear flags
  CMP0->CR1 |= CMP_CR1_EN_MASK;     // Enable comparator module
}

void setFlag(volatile unsigned int* add){
	const unsigned int set = Flag_set;
	program_flash(*add,(char*)&set,4);			//if the flags are stored in flash, use this line
}

bool isFlagSet(volatile unsigned int* add){
	if( *(unsigned int *)(*add) == Flag_set) return true;
	return false;
}

void erase_flash_secors_for_RAM(){
	int i;
	for(i = 2; i<= 2+ramToFlash_sectors_number; i++)				//start the erase from the 3rd last block, the first 2 are used for flags and peripheral registers
		erase_sector(flash_end-i*sector_Size);
}

void erase_flags_memory(){
	if(!SaveFlagsInFlash) {
    *getFlag_1() = Flag_erase;
    *getFlag_2() = Flag_erase;
    *getFlag_3() = Flag_erase;
    *getFlag_4() = Flag_erase;
	}else{
		erase_sector(Flash_Flags_Sector_Start);
	}
}

void copyRamToFlash(){
	// Erase flash sectors o save the RAM content
	erase_flash_secors_for_RAM();  
	program_flash(flash_ramSection_start, (char*) RAM_1_Address, RAM_Size);     // Copy all the RAM to flash
}