/**
 * @file SWCommon.h
 * @brief The header file for Smart Wheels Common functions. Note: The functions in here will be only used in debug purpose, and all of these debug functions will not be included for the release version.
 * @author Jordan Brack <jabrack@mix.wvu.edu>, Haofan Zheng <hazheng@mix.wvu.edu>
 * 
 */
#pragma once
#ifndef SW_COMMON_H
#define SW_COMMON_H

#include "ArduUTFT.h"
#include <mbed.h>

#ifdef SW_DEBUG
#define LOGI(...) char sw_common_buf_v87t790[100];\
                sprintf(sw_common_buf_v87t790, __VA_ARGS__);\
                ardu_utft_print(sw_common_buf_v87t790, 230, 100);
                
#else //SW_DEBUG
#define LOGI(...)

#endif //SW_DEBUG

#ifdef SW_DEBUGCOUNTER

/**
 * @class DebugCounter
 * @brief This class will be used as a counter during debugging. This class can be used when estimating the running speed of the system, or some specific function.
 */
class DebugCounter
{
public:
    /**
    * @brief The constructor for the DebugCounter class.
    * @param maxCount The number of counts that should be accumulated before changing the output pin status.
    * @param pin The read address for the slave device.
    */
    DebugCounter(uint16_t maxCount, PinName pin) :
        m_output(DigitalOut(pin, 0)),
        m_maxCount(maxCount),
        m_counter(0)
    {
        
    }
    
    /**
    * @brief The update function, where the counter gets count. If the number of counts approached the max count, the output pin status will be flipped.
    */
    void Update()
    {
        ++m_counter;
        if(m_counter >= m_maxCount)
        {
            m_output.write(!(m_output.read()));
            m_counter = 0;
        }
    }
    

private:
    DigitalOut m_output;        /*!< @brief The output pin. */
    const uint16_t m_maxCount;  /*!< @brief The max count. */
    uint16_t m_counter;         /*!< @brief The counter. */
    
};
#endif //SW_DEBUGCOUNTER


#endif //SW_COMMON_H