/**
 * @file ArduCAM.h
 * @brief The header file for all functions that controls the ArduCam Mini camera.
 * @author Jordan Brack <jabrack@mix.wvu.edu>, Haofan Zheng <hazheng@mix.wvu.edu>
 * 
 */
#pragma once
#ifndef ARDU_CAM_H
#define ARDU_CAM_H

#include <mbed.h>
#include "PinAssignment.h"

#define RESOLUTION_WIDTH  80   /*!< @brief The resolution width we are using. */
#define RESOLUTION_HEIGHT 60   /*!< @brief The resolution height we are using. */
//#define MANUAL_REDUCE_RESULOTION_BY2
#define CAM_ROI_UPPER_LIMIT 7  /*!< @brief The number of lines we actually using for the image processing. */


#define BOTH_NOT_FOUND   0     /*!< @brief ENUM for whether the border is found. Both sides of the border is not found. */
#define LEFT_FOUND       1     /*!< @brief ENUM for whether the border is found. Left side of the border is found. */
#define RIGHT_FOUND      2     /*!< @brief ENUM for whether the border is found. Right side of the border is found. */
#define BOTH_FOUND       3     /*!< @brief ENUM for whether the border is found. Both sides of the border is found. */

#define INTERSECTION_NULL 0
#define INTERSECTION_IN_L   1
#define INTERSECTION_IN_R   2
#define INTERSECTION_IN     3
#define INTERSECTION_OUT  5

//=====Must pick one and only one here!=====
//#define ARDUCAM_SHIELD_OV2640     /*!< @brief Use ArduCam Shield V2 with OV2640 camera module. */
//#define ARDUCAM_SHIELD_OV7725     /*!< @brief Use ArduCam Shield V2 with OV7725 camera module. */
//#define ARDUCAM_SHIELD_OV7670     /*!< @brief Use ArduCam Shield V2 with OV7670 camera module. */
#define ARDUCAM_MINI_OV2640       /*!< @brief Use ArduCam Mini with OV2640 camera module. */
//==========================================


#if defined(ARDUCAM_SHIELD_OV2640) || defined(ARDUCAM_SHIELD_OV7725) || defined(ARDUCAM_SHIELD_OV7670)
    #define ARDUCAM_SHIELD     /*!< @brief Use ArduCam Shield V2. (Automatically defined, DO NOT touch.) */
#elif defined(ARDUCAM_MINI_OV2640)
    #define ARDUCAM_MINI       /*!< @brief Use ArduCam Mini. (Automatically defined, DO NOT touch.) */
#else
    #error Must pick one and only one ArduCam type!
#endif

#if defined(ARDUCAM_SHIELD_OV2640) || defined(ARDUCAM_MINI_OV2640)
    #define ARDUCAM_OV2640     /*!< @brief Use OV2640 camera module. (Automatically defined, DO NOT touch.) */
#elif defined(ARDUCAM_SHIELD_OV7725)
    #define ARDUCAM_OV7725     /*!< @brief Use OV7725 camera module. (Automatically defined, DO NOT touch.) */
#elif defined(ARDUCAM_SHIELD_OV7670)
    #define ARDUCAM_OV7670     /*!< @brief Use OV7670 camera module. (Automatically defined, DO NOT touch.) */
#else
    #error Must pick one and only one ArduCam type!
#endif

#if defined(ARDUCAM_OV2640)
    #include "OV2640RegProg.h"
#elif defined(ARDUCAM_OV7725)
    #include "OV7725RegProg.h"
#elif defined(ARDUCAM_OV7670)
    #include "OV7670RegProg.h"
#endif

#define ARDUCHIP_VER                0x40  /*!< @brief The address for the ArduCam version register. */
#if defined(ARDUCAM_SHIELD)
    #define ARDUCHIP_VER_NUM        0x61     /*!< @brief ArduCam Version number for ArduCam Shield V2. */
#elif defined(ARDUCAM_MINI)
    #define ARDUCHIP_VER_NUM        0x55     /*!< @brief ArduCam Version number for ArduCam Mini. */
#endif

#define ARDUCHIP_TEST1          0x00  /*!< @brief The address for the ArduCam test register. */
#define ARDUCHIP_TEST_MSG       0x72     /*!< @brief The message used to write/read/verify with the test register. */

#define ARDUCHIP_CAP_CTRL       0x01  /*!< @brief The register address that used to set how many picture need to take. */

#if defined(ARDUCAM_SHIELD)
    #define ARDUCHIP_MODE           0x02  /*!< @brief The address for the ArduCam bus mode setting register. */
    #define MCU2LCD_MODE            0x00      /*!< @brief The byte used to set the ArduCam bus to MCU to LCD mode. */
    #define CAM2LCD_MODE            0x01      /*!< @brief The byte used to set the ArduCam bus to CAM to LCD mode. */
#endif

#define ARDUCHIP_FIFO           0x04  /*!< @brief The address for the ArduCam FIFO and I2C setting register. */
#define FIFO_CLEAR_MASK         0x01       /*!< @brief The mask used to clear the FIFO. */
#define FIFO_START_MASK         0x02       /*!< @brief The mask used to start the FIFO write. */
#define FIFO_RDPTR_RST_MASK     0x10       /*!< @brief The mask used to reset the FIFO read pointer. */
#define FIFO_WRPTR_RST_MASK     0x20       /*!< @brief The mask used to reset the FIFO write pointer. */

#define ARDUCHIP_TRIG           0x41  /*!< @brief The address for the ArduCam trigger status register. */
#define VSYNC_MASK              0x01    /*!< @brief The mask used to check the Vsync status. */
#define SHUTTER_MASK            0x02    /*!< @brief The mask used to check the shutter status. */
#define CAP_DONE_MASK           0x08    /*!< @brief The mask used to check if the capture is done. */

#define FIFO_SIZE1              0x42  /*!< @brief Camera write FIFO size[7:0] for burst to read. */
#define FIFO_SIZE2              0x43  /*!< @brief Camera write FIFO size[15:8]. */
#define FIFO_SIZE3              0x44  /*!< @brief Camera write FIFO size[18:16]. */

#define BURST_FIFO_READ         0x3C  /*!< @brief Burst FIFO read operation. */
#define SINGLE_FIFO_READ        0x3D  /*!< @brief Single FIFO read operation. */

#ifdef __cplusplus
extern "C" {
#endif

/**
* @brief Init the camera. There is only one camera in the system, thus, this function is required to be called only once during the system initialization.
*/
bool ardu_cam_init();

/**
* @brief Start the capture for one picture. The capture will automatically stop after a picture is captured.
*/
void ardu_cam_start_capture();

/**
* @brief Get the current FIFO length.
* @return The length of the current FIFO.
*/
uint32_t ardu_cam_get_fifo_length();

/**
* @brief Get the array that describe the center line that detected by the image processing.
* @return A  pointer points to the array.
*/
 const uint8_t* ardu_cam_get_center_array();

/**
* @brief Check if the capture is finished.
* @return True if the capture is finished, otherwise, return false.
*/
uint8_t ardu_cam_is_capture_finished();

/**
* @brief Display the image on the UTFT screen.
*/
void ardu_cam_display_img_utft();

/**
* @brief Get a new picture, and run the image processing.
*/
void image_processing();

/**
* @brief Get wether the terminate line is detected.
* @return 0 for no, 1 for yes.
*/
uint8_t ardu_cam_get_is_encounter_terminate();

/**
* @brief Get wether the borders are found.
* @return ENUM for whether the border is found. Please refer to the definition above.
*/
//uint8_t ardu_cam_get_is_border_found();

/**
* @brief Gets the version number from ArduCam.
* @return The version number.
*/
uint8_t ardu_cam_get_ver_num();

uint8_t ardu_cam_get_intersection_info();

#ifdef __cplusplus
}
#endif


#endif //ARDU_CAM_H