/**
 * @file Motor.h
 * @brief The header file for all functions that control the motors.
 * @author Jordan Brack <jabrack@mix.wvu.edu>, Haofan Zheng <hazheng@mix.wvu.edu>
 * 
 */
#pragma once
#ifndef MOTOR_H
#define MOTOR_H

#include <mbed.h>

#define MotorDir uint8_t  /*!< @brief Define the MotorDir type as a alias to the uint8_t. */
#define MDIR_Forward  0  /*!< @brief Move forward is low voltage level. */
#define MDIR_Backward 1  /*!< @brief Move backward is high voltage level. */


#define MOTOR_MAX_SPEED_LIMIT 0.55f /*!< @brief The percentage of the max speed (power) level for the motor driver. (This will be multiply to the speed that finally set to the motor driver) */

#define MOTOR_DIFF_MIN_SPEED  0.55f /*!< @brief The percentage of the digital rear differential. (It should be used outside of these functions.) */

#ifdef __cplusplus
extern "C" {
#endif

/**
* @brief Init the motors. There is only one pair of motors in the system, thus, this function is required to be called only once during the system initialization.
*/
void motor_init();

/**
* @brief Set the left motor speed (power) level.
* @param speed The percentage (between 0 ~ 1) of the speed (power) level. Positive for forward, and negative for backward.
*/
void motor_set_left_speed(const float speed);

/**
* @brief Set the right motor speed (power) level.
* @param speed The percentage (between 0 ~ 1) of the speed (power) level. Positive for forward, and negative for backward.
*/
void motor_set_right_speed(const float speed);

/**
* @brief Set the motor speed (power) level for both motors.
* @param speed_left  The percentage (between 0 ~ 1) of the speed (power) level. Positive for forward, and negative for backward.
* @param speed_right The percentage (between 0 ~ 1) of the speed (power) level. Positive for forward, and negative for backward.
*/
inline void motor_set_speeds(const float speed_left, const float speed_right)
{
    motor_set_left_speed(speed_left);
    motor_set_right_speed(speed_right);
}

/**
* @brief Set the left motor direction.
* @param dir The direction of the motor. For the datatype and available parameter values, please reference to the defines above.
*/
void motor_set_left_direction(MotorDir dir);

/**
* @brief Set the right motor direction.
* @param dir The direction of the motor. For the datatype and available parameter values, please reference to the defines above.
*/
void motor_set_right_direction(MotorDir dir);

/**
* @brief Set the motor direction for both motors.
* @param dirL The direction of the left motor. For the datatype and available parameter values, please reference to the defines above.
* @param dirR The direction of the right motor. For the datatype and available parameter values, please reference to the defines above.
*/
inline void motor_set_direction(MotorDir dirL, MotorDir dirR)
{
    motor_set_left_direction(dirL);
    motor_set_right_direction(dirR);
}

#ifdef __cplusplus
}
#endif

#endif //MOTOR_H