/**
 * @file CamRegBuf.h
 * @brief The header file for the camera register buffer class.
 * @author Jordan Brack <jabrack@mix.wvu.edu>, Haofan Zheng <hazheng@mix.wvu.edu>
 * 
 */
#pragma once
#ifndef CAM_REG_BUF_H
#define CAM_REG_BUF_H

#include <mbed.h>

struct sensor_reg;

/**
 * @class CamRegBuf
 * @brief The camera register buffer class. This class is used to load the register program to the camera (slave device) register through the SCCB control. And it can also just write/read a single byte to/from a specific register.
 */
class CamRegBuf
{
public:
    /**
    * @brief The constructor for the CamRegBuf class.
    * @param writeAddr The write address for the slave device.
    * @param readAddr The read address for the slave device.
    */
    CamRegBuf(uint8_t writeAddr,  uint8_t readAddr);
    
    /**
    * @brief The destructor for the CamRegBuf class.
    */
    ~CamRegBuf();
    
    /**
    * @brief The method that used to write a single byte to a specific register. NOTE: It is a Blocking method. Do not use during the running state!!
    * @param RegAddr The register address.
    * @param Data The byte that needs to write.
    */
    void SCCBWrite(const uint8_t RegAddr, uint8_t Data);
    
    /**
    * @brief The method that used to read a single byte from a specific register. NOTE: It is a Blocking method. Do not use during the running state!!
    * @param RegAddr The register address.
    * @return The byte that read from the register.
    */
    uint8_t SCCBRead(const uint8_t RegAddr);
    
    //int32_t I2CBufferRead(int32_t ucDevAddr, uint8_t *ucBuffer, int32_t ulSize);
    
    //int32_t I2CBufferWrite(int32_t ucDevAddr, uint8_t *ucBuffer, int32_t ulSize);
    
    //Blocking method. Do not use during the running state!!
    //void ReadRegisters();
    
    /**
    * @brief The method that used to write a entire program to the slave device. NOTE: It is a Blocking method. Do not use during the running state!!
    * @param reglist The program (A array of sensor_reg) that needs to write.
    */
    void WriteRegSet(const struct sensor_reg * reglist);
    
private:
    uint8_t m_writeAddr; /*!< @brief The write address for the slave device. */
    
    uint8_t m_readAddr;  /*!< @brief The read address for the slave device. */
    
    I2C m_sccbCtrl;      /*!< @brief The I2C connection to the slave device. */
};

#endif //CAM_REG_BUF_H