/**
 * @file ArduUTFT.h
 * @brief The header file for all functions that controls the ArduCam UTFT screen.
 * @author Jordan Brack <jabrack@mix.wvu.edu>, Haofan Zheng <hazheng@mix.wvu.edu>
 * 
 */
#pragma once
#ifndef ARDU_UTFT_H
#define ARDU_UTFT_H

#include <mbed.h>
#include "PinAssignment.h"

#define ARDUCHIP_MODE           0x02  /*!< @brief The address for the ArduCam bus mode setting register. */
#define MCU2LCD_MODE            0x00     /*!< @brief The byte used to set the ArduCam bus to MCU to LCD mode. */
#define CAM2LCD_MODE            0x01     /*!< @brief The byte used to set the ArduCam bus to CAM to LCD mode. */

#define ARDUCHIP_TEST1_UTFT          0x00  /*!< @brief The address for the ArduCam test register. */
#define ARDUCHIP_TEST_MSG_UTFT       0x72     /*!< @brief The message used to write/read/verify with the test register. */

#define ARDUCHIP_VER_UTFT            0x40  /*!< @brief The address for the ArduCam version register. */
#define ARDUCHIP_VER_NUM_UTFT        0x61     /*!< @brief The version num for the ArduCam Shield V2 */


#define UTFT_DISP_X_SIZE  239   /*!< @brief The vertical size of the UTFT screen resolution. */
#define UTFT_DISP_Y_SIZE  319   /*!< @brief The horizontal size of the UTFT screen resolution. */

#define UTFT_LEFT 0        /*!< @brief left alignment. */
#define UTFT_RIGHT 9999    /*!< @brief Right alignment. */
#define UTFT_CENTER 9998   /*!< @brief right alignment. */

#define CAM_IMG_CANVAS_ROW_OFFSET 20    /*!< @brief The horizontal postion of the canvas that used to display the camera image. */
#define CAM_IMG_CANVAS_COL_OFFSET 200   /*!< @brief The vertical postion of the canvas that used to display the camera image. */


#ifdef __cplusplus
extern "C" {
#endif

/**
* @brief Init the UTFT screen. There is only one screen in the system, thus, this function is required to be called only once during the system initialization.
*/
void ardu_utft_init(); //We only use LANDSCAPE orientation!

/**
* @brief Set the ArduCam bus mode.
* @param mode The new bus mode for the ArduCam Shield V2.
*/
void ardu_cam_set_mode(uint8_t mode);

/**
* @brief Write a pixel to the screen.
* @param VH The MSB of a RGB565 value.
* @param VL The LSB of a RGB565 value.
*/
void ardu_utft_write_DATA(uint8_t VH, uint8_t VL);

/**
* @brief Clear the screen.
*/
void ardu_utft_clr_scr();

/**
* @brief Set the drawing range.
* @param x1 The start position in the x-axis.
* @param y1 The start position in the y-axis.
* @param x2 The end position in the x-axis.
* @param y2 The end position in the y-axis.
*/
void ardu_utft_set_xy(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2);

/**
* @brief Set the drawing range from the (0,0) till the end of the screen.
*/
void ardu_utft_reset_xy();

/**
* @brief Set the drawing color.
* @param r The value for red channel.
* @param g The value for green channel.
* @param b The value for blue channel.
*/
void ardu_utft_set_color(uint8_t r, uint8_t g, uint8_t b);

/**
* @brief Draw a rectangle on the srceen.
* @param x1 The first point of the rectangle in the x-axis.
* @param y1 The first point of the rectangle in the y-axis.
* @param x2 The last point of the rectangle in the x-axis.
* @param y2 The last point of the rectangle in the y-axis.
*/
void ardu_utft_draw_rect(int x1, int y1, int x2, int y2);

/**
* @brief Draw a horizontal line on the srceen.
* @param x The start position of the line in the x-axis.
* @param y The start position of the line in the y-axis.
* @param l The length of the line.
*/
void ardu_utft_draw_hline(int x, int y, int l);

/**
* @brief Draw a vertical line on the srceen.
* @param x The start position of the line in the x-axis.
* @param y The start position of the line in the y-axis.
* @param l The length of the line.
*/
void ardu_utft_draw_vline(int x, int y, int l);

/**
* @brief Draw a single pixel on the srceen.
* @param x The position of the pixel in the x-axis.
* @param y The position of the pixel in the y-axis.
*/
void ardu_utft_draw_pixel(int x, int y);

/**
* @brief Draw a filled rectangle on the srceen.
* @param x1 The first point of the rectangle in the x-axis.
* @param y1 The first point of the rectangle in the y-axis.
* @param x2 The last point of the rectangle in the x-axis.
* @param y2 The last point of the rectangle in the y-axis.
*/
void ardu_utft_fill_rect(int x1, int y1, int x2, int y2);

/**
* @brief Set the font that used to print character on the screen.
* @param font The pointer to a bytes array that describe the font. For details, please refer to the ArduUTFTFont.h header file.
*/
void ardu_utft_set_font(uint8_t * font);

/**
* @brief print a char on the screen.
* @param c The char that needs to print.
* @param x The position of the char in x-axis.
* @param y The position of the char in y-axis.
*/
void ardu_utft_print_char(char c, int x, int y);

/**
* @brief print a char string on the screen.
* @param st The pointer to the char string.
* @param x The start position of the char string in x-axis.
* @param y The start position of the char string in y-axis.
*/
void ardu_utft_print(char * st, int x, int y);

/**
* @brief Set the current row when printing the camera image on the screen.
* @param row The row index.
*/
inline void ardu_utft_set_camimg_row(const uint16_t row)
{
    ardu_utft_set_xy(0, row + CAM_IMG_CANVAS_ROW_OFFSET, CAM_IMG_CANVAS_COL_OFFSET, row + CAM_IMG_CANVAS_ROW_OFFSET);
}

/**
* @brief Set the current row and colum when printing the camera image on the screen.
* @param row The row index.
* @param col The col index.
*/
inline void ardu_utft_set_camimg_rowcol(const uint16_t row, const uint16_t col)
{
    ardu_utft_set_xy(0, row + CAM_IMG_CANVAS_ROW_OFFSET, CAM_IMG_CANVAS_COL_OFFSET - col, row + CAM_IMG_CANVAS_ROW_OFFSET);
}

/**
* @brief Gets the version number from ArduCam Shield V2.
* @return The version number.
*/
uint8_t ardu_utft_get_ver_num();
#ifdef __cplusplus
}
#endif

#endif //ARDU_UTFT_H